/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "slabs.H"
#include "volFields.H"
#include "dictionary.H"
#include "foamTime.H"
#include "areaMesh.H"
#include "uniformDimensionedFields.H"
#include "edgeFields.H"
#include "cfdfunction.H"
#include "volSurfaceMapping.H"
#include "HormannAgathos.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(slabs, 0);
}

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //



// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::slabs::slabs
(
    const word& name,
    const objectRegistry& obr,
    const dictionary& dict,
    const bool loadFromFiles
):
    obr_(obr)
{

    read(dict);


    Info << endl << "adding slabs:" << endl;
    forAll(slabNames, areaI)
    {
        Info << "      " << slabNames[areaI] << " at t=" << releaseTime[areaI] << endl;
    }

    Info << endl << endl;


}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

Foam::slabs::~slabs()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::slabs::read(const dictionary& dict)
{
    const PtrList<entry> regions
    (
        dict.lookup("slabs")
    );

    vertices.setSize(regions.size());
    releaseTime.setSize(regions.size());
    slabNames.setSize(regions.size());

    forAll(slabNames, areaI)
    {
        const entry& regionInfo = regions[areaI];


        if (!regionInfo.isDict())
        {
            FatalIOErrorIn("slabs.C", dict)
                << "Entry " << regionInfo << " in boundary section is not a"
                << " valid dictionary." << exit(FatalIOError);
        }
        slabNames[areaI] = regionInfo.keyword();

        dictionary areaDict = regionInfo.dict();

        vector offset;

        areaDict.lookup("vertices") >> vertices[areaI];
        areaDict.lookup("releasetime") >> releaseTime[areaI];
    }

}

void Foam::slabs::execute()
{

    forAll(releaseTime, i)
    {
        if (obr_.time().value() >= releaseTime[i])
        {

            List<point2D> points;

            points.resize(vertices[i].size());

            forAll(vertices[i], vI)
            {
                points[vI] = point2D(vertices[i][vI].x(), vertices[i][vI].y());
            }
            HormannAgathos polygon(points, 0.001);

            areaScalarField &h = const_cast<areaScalarField&>(obr_.lookupObject<areaScalarField>("h"));
            areaScalarField &hentrain = const_cast<areaScalarField&>(obr_.lookupObject<areaScalarField>("hentrain"));
            areaVectorField &Us = const_cast<areaVectorField&>(obr_.lookupObject<areaVectorField>("Us"));
            const faMesh &aMesh = h.mesh();
            const areaVectorField &c = aMesh.areaCentres();

            scalar totalMass = 0;

            forAll(c.internalField(), j)
            {
                if (polygon.evaluate(point2D(c[j].x(), c[j].y())) != HormannAgathos::POINT_OUTSIDE)
                {
                    totalMass = aMesh.S()[j]*hentrain[j];
                    scalar oldH = h[j];
                    h[j] = h[j] + hentrain[j];
                    Us[j] = Us[j]*oldH/h[j];
                    hentrain[j] = 0;
                }
            }

            Info << endl << "Releasing " << slabNames[i] << " with " << totalMass << " m3" << endl << endl;
            releaseTime[i] = 1e100;
        }
    }

}


void Foam::slabs::end()
{

}


void Foam::slabs::write()
{

}

// ************************************************************************* //
