/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "PoliquenForterre.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
namespace frictionModels
{
    defineTypeNameAndDebug(PoliquenForterre, 0);
    addToRunTimeSelectionTable(frictionModel, PoliquenForterre, dictionary);
}
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::frictionModels::PoliquenForterre::PoliquenForterre
(const word& name,
    const dictionary& frictionProperties,
    const areaVectorField &Us,
    const areaScalarField &h,
    const areaScalarField &p)
:
    frictionModel(name, frictionProperties, Us, h, p),
    L_(frictionProperties_.subDict("PoliquenForterreCoeffs").lookup("L")),
    beta_(frictionProperties_.subDict("PoliquenForterreCoeffs").lookup("beta")),
    zeta1_(frictionProperties_.subDict("PoliquenForterreCoeffs").lookup("zeta1")),
    zeta2_(frictionProperties_.subDict("PoliquenForterreCoeffs").lookup("zeta2")),
    zeta3_(frictionProperties_.subDict("PoliquenForterreCoeffs").lookup("zeta3")),
    gamma_(frictionProperties_.subDict("PoliquenForterreCoeffs").lookup("gamma")),
    mu_
    (
        IOobject
        (
            "mu",
            Us_.time().timeName(),
            Us_.db(),
            IOobject::NO_READ,
            IOobject::AUTO_WRITE
        ),
        Us_.mesh(),
        dimensionedScalar("one", dimensionSet(0, 0, 0, 0, 0, 0, 0), 0)
    ),
    g_
    (
        IOobject
        (
            "g",
            Us_.time().constant(),
            Us_.db(),
            IOobject::MUST_READ,
            IOobject::NO_WRITE
        )
    )
{
    zeta1_ *= M_PI/180.;
    zeta2_ *= M_PI/180.;
    zeta3_ *= M_PI/180.;
    Info<< "    " << L_ << endl
        << "    " << beta_ << endl
        << "    " << 180./M_PI*zeta1_ << endl
        << "    " << 180./M_PI*zeta2_ << endl
        << "    " << 180./M_PI*zeta3_ << endl << endl;

}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //


const Foam::areaVectorField &Foam::frictionModels::PoliquenForterre::tauSc() const
{
    tauSc_.internalField() = vector(0, 0, 0);

    return tauSc_;
}

const Foam::areaScalarField &Foam::frictionModels::PoliquenForterre::tauSp() const
{
    tauSp_.internalField() = 0;

    const areaVectorField &n = Us_.mesh().faceAreaNormals();
    const areaScalarField gn = g_&n;
    const areaScalarField u = mag(Us_);

    const areaScalarField Fr = (u+u0_)/sqrt((h_+h0_)*gn);
    const areaScalarField hs = h_*beta_/Fr;

    const areaScalarField mu_stop = tan(zeta1_)+(tan(zeta2_)-tan(zeta1_))/(1.+hs/L_);
    const areaScalarField mu_start = tan(zeta3_)+(tan(zeta2_)-tan(zeta1_))*exp(-hs/L_);

    mu_ = mu_stop;

    forAll(mu_, i)
    {
        if (Fr[i] < beta_.value())
        {
            mu_[i] = pow(Fr[i]/beta_.value(), gamma_.value())*(mu_stop[i]-mu_start[i])+mu_start[i];
        }
    }


    tauSp_ += 1./rho_ * p_ * mu_ * 1./(u + u0_);

    return tauSp_;
}

bool Foam::frictionModels::PoliquenForterre::read
(
    const dictionary& frictionProperties
)
{
    frictionModel::read(frictionProperties);

    return true;
}


// ************************************************************************* //
