/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "totalmomentum.H"
#include "volFields.H"
#include "dictionary.H"
#include "foamTime.H"
#include "areaMesh.H"
#include "uniformDimensionedFields.H"
#include "edgeFields.H"
#include "cfdfunction.H"
#include "volSurfaceMapping.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(totalmomentum, 0);
}

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //



// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::totalmomentum::totalmomentum
(
    const word& name,
    const objectRegistry& obr,
    const dictionary& dict,
    const bool loadFromFiles
):
    obr_(obr),
    hUsRes_
    (
        IOobject
        (
            "hUsRes",
            obr.lookupObject<areaVectorField>("Us").time().timeName(),
            obr.lookupObject<areaVectorField>("Us").db(),
            IOobject::NO_READ,
            IOobject::AUTO_WRITE
        ),
        obr.lookupObject<areaVectorField>("Us").mesh(),
        dimensionedVector("zero", dimensionSet(1, -1, -2, 0, 0, 0, 0), vector::zero)
    ),
    HUsRes_
    (
        IOobject
        (
            "HUsRes",
            obr.lookupObject<volVectorField>("U").time().timeName(),
            obr.lookupObject<volVectorField>("U").db(),
            IOobject::NO_READ,
            IOobject::AUTO_WRITE
        ),
        obr.lookupObject<volVectorField>("U").mesh(),
        dimensionedVector("zero", dimensionSet(1, -1, -2, 0, 0, 0, 0), vector::zero)
    ),
    rho_(dimensionedScalar("zero", dimensionSet(1, -3, 0, 0, 0, 0, 0), 0))
{

    read(dict);


    Info << endl << "Analyzing 3D momentum:" << endl
         << "   log is " << log_ << endl
         << endl << endl;


}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

Foam::totalmomentum::~totalmomentum()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::totalmomentum::read(const dictionary& dict)
{
    log_ = Switch(dict.lookup("log"));
    rho_ = dimensionedScalar(dict.lookup("rho"));
}



void Foam::totalmomentum::execute()
{
    const areaVectorField &Us = obr_.lookupObject<areaVectorField>("Us");
    const areaVectorField &pbn = obr_.lookupObject<areaVectorField>("pbn");
    const areaScalarField &h = obr_.lookupObject<areaScalarField>("h");
    const areaVectorField &tau = obr_.lookupObject<areaVectorField>("tau");
    const edgeScalarField &phi2s = obr_.lookupObject<edgeScalarField>("phi2s");

    const faMesh &aMesh = Us.mesh();
    const areaVectorField n = aMesh.faceAreaNormals();

    const uniformDimensionedVectorField &g = obr_.lookupObject<uniformDimensionedVectorField>("g");

    cfdfunction f(Us, h, tau, pbn, g, rho_, phi2s);

    f.calcHUResidual3D(hUsRes_);


    scalar maxhUR = 0;
    label cellI = -1;

    if (log_)
    {
        forAll(hUsRes_, i)
        {
            if (mag(hUsRes_[i]) > maxhUR)
            {
                cellI = i;
                maxhUR = mag(hUsRes_[i]);
            }

        }


        Info << "Max hUResidual = " << hUsRes_[cellI]
             << " @ " << aMesh.areaCentres()[cellI]
             << ", global = " << sum(hUsRes_*aMesh.S()).value() << endl << endl;
    }
}


void Foam::totalmomentum::end()
{

}


void Foam::totalmomentum::write()
{
    const faMesh &aMesh = hUsRes_.mesh();

    volSurfaceMapping vsm(aMesh);
    vsm.mapToVolume(hUsRes_, HUsRes_.boundaryField());
    HUsRes_.write();

}

// ************************************************************************* //
