/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "fvCFD.H"
#include "faCFD.H"
#include "dynamicpressure.H"
#include "volFields.H"
#include "dictionary.H"
#include "foamTime.H"
#include "areaMesh.H"
#include "uniformDimensionedFields.H"
#include "cfdfunction.H"


// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(dynamicpressure, 0);
}

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //



// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::dynamicpressure::dynamicpressure
(
    const word& name,
    const objectRegistry& obr,
    const dictionary& dict,
    const bool loadFromFiles
):
    obr_(obr),
    rho_(dimensionedScalar("zero", dimensionSet(1, -3, 0, 0, 0, 0, 0), 0)),
    hmin_(dimensionedScalar("hmin", dimensionSet(0, 1, 0, 0, 0, 0, 0), 0))
{

    read(dict);

    const fvMesh &mesh = obr.lookupObject<volVectorField>("U").mesh();
    const faMesh &aMesh = obr.lookupObject<areaVectorField>("Us").mesh();

    areaScalarField* pd
    (
        new areaScalarField
        (
            IOobject
            (
                "pd",
                obr.lookupObject<areaVectorField>("Us").time().timeName(),
                mesh,
                IOobject::NO_READ,
                IOobject::NO_WRITE
            ),
            aMesh,
            dimensionedScalar("zero", dimensionSet(1, -1, -2, 0, 0, 0, 0), 0)
        )
    );

    volScalarField* Pd
    (
        new volScalarField
        (
            IOobject
            (
                "Pd",
                obr.lookupObject<volVectorField>("U").time().timeName(),
                mesh,
                IOobject::NO_READ,
                IOobject::NO_WRITE
            ),
            mesh,
            dimensionedScalar("zero", dimensionSet(1, -1, -2, 0, 0, 0, 0), 0)
        )
    );

    mesh.objectRegistry::store(pd);
    mesh.objectRegistry::store(Pd);

    Info << endl << "Calculating dynamic pressure:" << endl
         << "   hmin is " << hmin_ << endl
         << "   rho is " << rho_ << endl << endl;


}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

Foam::dynamicpressure::~dynamicpressure()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::dynamicpressure::read(const dictionary& dict)
{
    rho_ = dimensionedScalar(dict.lookup("rho"));
    hmin_ = dimensionedScalar(dict.lookupOrDefault<dimensionedScalar>("him", dimensionedScalar("hmin", dimensionSet(0, 1, 0, 0, 0, 0, 0), 1e-2)));
}



void Foam::dynamicpressure::execute()
{
    const areaVectorField &Us = obr_.lookupObject<areaVectorField>("Us");
    const areaScalarField &h = obr_.lookupObject<areaScalarField>("h");
    areaScalarField &pd = const_cast<areaScalarField&>
                            (
                                obr_.lookupObject<areaScalarField>("pd")
                            );

    const areaScalarField pdt = pos(h-hmin_)*magSqr(Us)*rho_;
    pd = max(pd, pdt);
}


void Foam::dynamicpressure::end()
{

}


void Foam::dynamicpressure::write()
{
    const areaVectorField &Us = obr_.lookupObject<areaVectorField>("Us");
    const faMesh &aMesh = Us.mesh();

    const areaScalarField &pd = obr_.lookupObject<areaScalarField>("pd");
    volScalarField &Pd = const_cast<volScalarField&>
                            (
                                obr_.lookupObject<volScalarField>("Pd")
                            );

    volSurfaceMapping vsm(aMesh);
    vsm.mapToVolume(pd, Pd.boundaryField());

    Pd.write();
}

// ************************************************************************* //
