/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "cfdfunction.H"
#include "volFields.H"
#include "dictionary.H"
#include "foamTime.H"
#include "areaMesh.H"
#include "faCFD.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::cfdfunction::cfdfunction
(
    const areaVectorField &Us,
    const areaScalarField &h,
    const areaVectorField &tau,
    const areaVectorField &pbn,
    const uniformDimensionedVectorField &g,
    const dimensionedScalar &rho,
    const edgeScalarField &phi2s
):
    Us_(Us),
    h_(h),
    tau_(tau),
    pbn_(pbn),
    g_(g),
    rho_(rho),
    phi2s_(phi2s)
{

}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

Foam::cfdfunction::~cfdfunction()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::cfdfunction::calcLoss(areaScalarField &loss)
{
    solve(fam::ddt(loss) == (tau_&Us_));
}

void Foam::cfdfunction::calcHUResidual3D(areaVectorField &hUsRes)
{

    const faMesh &aMesh = Us_.mesh();
    const areaVectorField &n = aMesh.faceAreaNormals();

    hUsRes = rho_*(
        fac::ddt(h_,Us_)
      + fac::div(phi2s_, Us_)
      + fac::ndiv(phi2s_, Us_)
      - g_*h_
      + fac::grad((pbn_&n)*h_/(2.*rho_))
      + fac::ngrad((pbn_&n)*h_/(2.*rho_))
      + tau_/rho_
      + pbn_/rho_
    );
}

void Foam::cfdfunction::calcHUResidual(areaVectorField &hUsRes)
{
    const faMesh &aMesh = Us_.mesh();
    const areaVectorField &n = aMesh.faceAreaNormals();

    hUsRes = rho_*(
        fac::ddt(h_,Us_)
      + fac::div(phi2s_, Us_)
      - (g_*h_-((g_*h_)&n*n))
      + fac::grad((pbn_&n)*h_/(2.*rho_))
      + tau_/rho_
    );
}

void Foam::cfdfunction::calcHUResidualN(areaVectorField &hUsRes)
{

    const faMesh &aMesh = Us_.mesh();
    const areaVectorField &n = aMesh.faceAreaNormals();

    hUsRes = rho_*(
        fac::ndiv(phi2s_, Us_)
      - ((g_*h_)&n*n)
      + fac::ngrad((pbn_&n)*h_/(2.*rho_))
      + pbn_/rho_
    );
}


// ************************************************************************* //
