/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "muI.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
namespace frictionModels
{
    defineTypeNameAndDebug(MuI, 0);
    addToRunTimeSelectionTable(frictionModel, MuI, dictionary);
}
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::frictionModels::MuI::MuI
(const word& name,
    const dictionary& frictionProperties,
    const areaVectorField &Us,
    const areaScalarField &h,
    const areaScalarField &p)
:
    frictionModel(name, frictionProperties, Us, h, p),
    d_(frictionProperties_.subDict("MuICoeffs").lookup("d")),
    rhop_(frictionProperties_.subDict("MuICoeffs").lookup("rho_p")),
    mus_(frictionProperties_.subDict("MuICoeffs").lookup("mu_s")),
    mu2_(frictionProperties_.subDict("MuICoeffs").lookup("mu_2")),
    I0_(frictionProperties_.subDict("MuICoeffs").lookup("I_0")),
    mu_
    (
        IOobject
        (
            "mu",
            Us_.time().timeName(),
            Us_.db(),
            IOobject::NO_READ,
            IOobject::NO_WRITE
        ),
        Us_.mesh(),
        dimensionedScalar("one", dimensionSet(0, 0, 0, 0, 0, 0, 0), 0)
    )
{
    Info<< "    " << d_ << endl
        << "    " << rhop_ << endl
        << "    " << mus_ << endl
        << "    " << mu2_ << endl
        << "    " << I0_ << endl << endl;

}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //


const Foam::areaVectorField &Foam::frictionModels::MuI::tauSc() const
{
    tauSc_.internalField() = vector(0, 0, 0);

    return tauSc_;
}

const Foam::areaScalarField &Foam::frictionModels::MuI::tauSp() const
{
    tauSp_.internalField() = 0;
    areaScalarField u = mag(Us_);

    areaScalarField gamma = 5./2.*u/(h_+h0_);
    areaScalarField I = gamma*d_/(sqrt(p_/rhop_)+u0_);

    mu_ = mus_ + (mu2_-mus_)/(I0_/(I+SMALL)+1.);

    tauSp_ += 1./rho_ * p_ * mu_ *
              1./(u + u0_);

    return tauSp_;
}

bool Foam::frictionModels::MuI::read
(
    const dictionary& frictionProperties
)
{
    frictionModel::read(frictionProperties);

    return true;
}


// ************************************************************************* //
