/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    foam-extend is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation, either version 3 of the License, or (at your
    option) any later version.

    foam-extend is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with foam-extend.  If not, see <http://www.gnu.org/licenses/>.

Namespace
    Foam::depositionModels

Description
    A namespace for various deposition model implementations.

Class
    Foam::depositionModel

Description
    An abstract base class for deposition models

SourceFiles
    depositionModel.C
    depositionModelNew.C

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#ifndef depositionModel_H
#define depositionModel_H

#include "IOdictionary.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"
#include "dimensionedScalar.H"
#include "tmp.H"
#include "autoPtr.H"
#include "faMatrices.H"
#include "areaFieldsFwd.H"
#include "FieldFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                         Class depositionModel Declaration
\*---------------------------------------------------------------------------*/

class depositionModel
{

protected:

    // Protected data

        word name_;
        dictionary depositionProperties_;


        //The density
        dimensionedScalar rho_;

        //Pointer to the surface velocity field
        const areaVectorField &Us_;

        //Pointer to the flow height field
        const areaScalarField &h_;

        //Pointer to the cover height field
        const areaScalarField &hentrain_;

        //Pointer to the basal pressure field
        const areaScalarField &pb_;

        //Pointer to the bottom stress field
        const areaVectorField &tau_;

        //Sink term
        mutable areaScalarField Sd_;


    // Private Member Functions

        //- Disallow copy construct
        depositionModel(const depositionModel&);

        //- Disallow default bitwise assignment
        void operator=(const depositionModel&);


public:

    //- Runtime type information
    TypeName("depositionModel");


    // Declare run-time constructor selection table

#ifndef SWIG
        declareRunTimeSelectionTable
        (
            autoPtr,
            depositionModel,
            dictionary,
            (
                const word& name,
                const dictionary& depositionProperties,
                const areaVectorField &Us,
                const areaScalarField &h,
                const areaScalarField &hentrain,
                const areaScalarField &pb,
                const areaVectorField &tau
            ),
            (name, depositionProperties, Us, h, hentrain, pb, tau)
        );
#endif


    // Selectors

        //- Return a reference to the selected deposition model
        static autoPtr<depositionModel> New
        (
            const word& name,
            const dictionary& depositionProperties,
            const areaVectorField &Us,
            const areaScalarField &h,
            const areaScalarField &hentrain,
            const areaScalarField &pb,
            const areaVectorField &tau
        );


    // Constructors

        //- Construct from components
        depositionModel
        (const word& name,
            const dictionary& depositionProperties,
            const areaVectorField &Us,
            const areaScalarField &h,
            const areaScalarField &hentrain,
            const areaScalarField &pb,
            const areaVectorField &tau
        );


    // Destructor

        virtual ~depositionModel()
        {}


    // Member Functions

        //- Return Sink by deposition
        virtual const areaScalarField &Sd() const = 0;

        //- Return the deposition properties dictionary
        const dictionary& depositionProperties() const
        {
            return depositionProperties_;
        }

        //- Read depositionProperties dictionary
        virtual bool read(const dictionary& depositionProperties) = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
