/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | faSavageHutterFOAM
    \\  /    A nd           | Copyright (C) 2017 Matthias Rauter
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Author
    Matthias Rauter matthias.rauter@uibk.ac.at

\*---------------------------------------------------------------------------*/

#include "faCFD.H"
#include "Stopingprofile.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
namespace depositionModels
{
    defineTypeNameAndDebug(Stopingprofile, 0);
    addToRunTimeSelectionTable(depositionModel, Stopingprofile, dictionary);
}
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::depositionModels::Stopingprofile::Stopingprofile
(
    const word& name,
    const dictionary& depositionProperties,
    const areaVectorField &Us,
    const areaScalarField &h,
    const areaScalarField &hentrain,
    const areaScalarField &pb,
    const areaVectorField &tau
)
:
    depositionModel(name, depositionProperties, Us, h, hentrain, pb, tau),
    ud_(depositionProperties.subDict("StopingprofileCoeffs").lookup("ud")),
    ad_(depositionProperties.subDict("StopingprofileCoeffs").lookup("ad")),
    gs_(Us.db().lookupObject<areaVectorField>("gs")),
    gn_(Us.db().lookupObject<areaScalarField>("gn"))
{
    Info << "    " << ud_ << endl
         << "    " << ad_ << endl << endl;

}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //


const Foam::areaScalarField &Foam::depositionModels::Stopingprofile::Sd() const
{
    areaScalarField a = pow(max(min(1. - mag(Us_)/ud_, 1.), 0.), ad_);

    areaScalarField hUddt = (-tau_/rho_+gs_*h_-fac::grad(pb_*h_/(2.*rho_)))//-gn_*h_*gradhentrain_)
                    &Us_.oldTime()/(mag(Us_.oldTime())+ dimensionedScalar("small", dimVelocity, SMALL));

    hUddt.internalField() = min(hUddt.internalField(), 0.);
    Sd_ = -a/(mag(Us_.oldTime())+dimensionedScalar("small", dimVelocity, SMALL))*hUddt;
    Sd_ = max(Sd_, dimensionedScalar("0", dimVelocity, 0));
    Sd_ = min(Sd_, h_/Us_.db().time().deltaT());

    forAll(Sd_, i)
    {
        if(mag(Us_.oldTime()[i]) < VSMALL)
            Sd_[i] = 0;
    }
    return Sd_;
}

bool Foam::depositionModels::Stopingprofile::read
(
    const dictionary& depositionProperties
)
{
    depositionModel::read(depositionProperties);

    depositionProperties_.lookup("ud") >> ud_;
    depositionProperties_.lookup("ad") >> ad_;

    return true;
}


// ************************************************************************* //
