import argparse
import funcsPostPro as fPP
from class_definitions import *

"""
.. module:: netCDFPlotVarsfromPfl
    :platform: Unix, Mac
    :synopsis: Read pfl nc files and convert to diagnoistic ParFlow vars and delete unwanted output

.. moduleauthor:: Wendy Sharples <w.sharples@fz-juelich.de>, Fabian Gasper <f.gasper@fz-juelich.de>

"""

__author__ = 'wendy, fabian'

# Assumes all files in ncDir are in netCDF4 format, user must specify:
# 1. initDate
# 2. outDir
# 3. ncFileDir
# 4. runName
# 5. timesteps
# 6. dzmult
# 7. dz
# 8. dx
# 9. dy
# 10. M
# assumes varList={"WTD":["WTD"],"WTDC":["WTDC"],"WC":["WC"],"WCC":["WCC"],"PAW":["PAW"],"SRO":["SRO"],"RUS":["RUS"],"ARC":["ARC"]}
# water table depth, water table depth change, total water storage in column, water storage in column change, plant available water, surface runoff, absolute recharge 
# Eg: python netCDFPlotVarsfromPfl.py 'ncDir' 'runName' 49 '7.5,7.5,5.0,5.0,2.0,0.5,0.35,0.25,0.15,0.10,0.065,0.035,0.025,0.015,0.01' 2 12500 12500 0.00001 436 424 'outDir'


class netCDFPlotVarsfromPfl:
    """

    """
    def __init__(self, ncDir, runName, timesteps, dzmult, dz, dx, dy, M, gx, gy, outDir):
        """

        :return:
        """
       
        self.outDir = outDir + '/'
        self.timesteps = timesteps
        #self.varList = {"WTD":["WTD"],"WTDC":["WTDC"],"WC":["WC"],"WCC":["WCC"]"PAW":["PAW"],"SRO":["SRO"],"RUS":["RUS"],"ARC":["ARC"]}
        self.varList = {"WTDA":["WTDA"],"WTDC":["WTDC"],"WC":["WC"],"WCC":["WCC"],"PAW":["PAW"],"SRO":["SRO"],"RUS":["RUS"]}
        self.ncDir = ncDir + '/'
        self.runName = runName
        results = [x.strip() for x in dzmult.split(',')]
        results = [float(i) for i in results]
        self.dzmult = results
        self.dz = dz
        self.dx = dx
        self.dy = dy
        self.M = M
        self.gx = gx
        self.gy = gy
	
    def loadAndPrintOutVars(self, VL, dz, dx, dy, dzmult, M, gx, gy, timesteps, ncDir, runName, outDir):
        variables={}
	
        if "PAW" in VL or "WC" in VL or "WCC" in VL or "ARC" in VL or "WTD" in VL or "WTDC" in VL or "SRO" in VL or "RUS" in VL:
            variables["PSI"]=Var(4,'PSI', vpath=ncDir+runName+'.out.pressure_profile.nc',off=0,numSteps=timesteps+1 )
        if "PAW" in VL or "WC" in VL or "WCC" in VL or "WTD" in VL or "RUS" in VL:
            variables["PORO"]=Var(4,'PORO', vpath=ncDir+runName+'.out.porosity_profile.nc',off=0,numSteps=timesteps+1 )
            variables["SAT"]=Var(4,'SATUR', vpath=ncDir+runName+'.out.saturation_profile.nc',off=0,numSteps=timesteps+1 )
        if "SRO" in VL :
            variables["SLOPES"]=Var(4,'SLOPES', vpath=ncDir+runName+'.out.slopes_profile.nc',off=0,numSteps=timesteps+1 )
        if "ARC" in VL :
            #VANGEN file is an array with dimensions [2][NZ][NX][NY]
	    #where vanGen[0][..] is alpha and vanGen[1][..] is N
            variables["VANGEN"]=Var(4,'VANGEN', vpath=ncDir+runName+'.vanGen.nc',off=0,numSteps=2 )     

        if "WTDC" in VL:
            variables["WTDC"]=Var(4,'WTDC',data=fPP.wtdc(variables["PSI"]), units="m")
        if "WTDA" in VL:
            dzsum = 0
            for element in dzmult:
                dzsum+=element*dz
            variables["WTDA"]=Var(4,'WTDA',data=fPP.wtdapprox(variables["PSI"],dzmult,dz),units="m")
        if "WC" in VL:
            variables["WCC"]=Var(4,'WC',data=fPP.colStorChange(variables["PSI"],variables["SAT"],variables["PORO"],dzmult,dz))
        if "RUS" in VL:
            variables["RUS"]=Var(4,'RUS',data=fPP.runsatstor(variables["PSI"],variables["SAT"],variables["PORO"],dzmult,dz),gpp=[fPP.gCut0])
        if "PAW" in VL:
            variables["PAW"]=Var(4,'PAW',data=fPP.plantAvailStor(variables["PSI"],variables["SAT"],variables["PORO"],dzmult,dz),gpp=[])
        if "SRO" in VL: 
            variables["SRO"]=Var(4,'SRO',data=fPP.surfaceRunOff(variables["PSI"],variables["SLOPES"],dx,dy,M),gpp=[])
 	
        if "WTD" in VL:
            dzsum = 0
            for element in dzmult:
                dzsum+=element*dz
            variables["WTD"]=Var(4,'WTD',data=fPP.wtdc(variables["PSI"],dzmult,dz,dzsum),units="m")

        diagnosticVariables = []
        for var in variables:
            if(var == "PAW" or var == "WC" or var == "WCC" or var == "RUS" or var == "WTDC" or var == "WTDA" or var == "SRO" or var == "ARC"):
                diagnosticVariables.append(variables[var])
	#Then create the netCDF file and write to it via the funcsPostPro functions
        ncfile = openNC(outDir + 'DiagnosticPflVars.nc','w')
        fPP.writeVarOutToNCFile(ncfile, diagnosticVariables, timesteps, 1, gy, gx)
	  
if __name__ == '__main__':

    data = netCDFPlotVarsfromPfl(sys.argv[1], sys.argv[2], sys.argv[3], sys.argv[4], sys.argv[5], sys.argv[6], sys.argv[7], sys.argv[8], sys.argv[9], sys.argv[10], sys.argv[11])
        
    #Load up and write out the new netCDF4 diagnostic vars
    data.loadAndPrintOutVars(data.varList, float(data.dz), float(data.dx), float(data.dy), data.dzmult, float(data.M), int(data.gx), int(data.gy), int(data.timesteps), data.ncDir, data.runName, data.outDir)
