#!/usr/bin/env python
"""
This script is running the darshan-parser with the output file of the darshan
profiling tool.

- requires the output file of the darshan profiling tool as an argument
"""
from __future__ import print_function, division
import argparse
from subprocess import Popen
from subprocess import PIPE
import shlex
import numpy as np
import re
from itertools import groupby
try: 
    from StringIO import StringIO
except ImportError:
    from io import StringIO


def io_percentage(nprocs, runtime, counters, data, out_file):
    """
    calculates average I/O cost per process (top left histogram of darshan pdf)

    :param nprocs: number of processes
    :param runtime: application runtime
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """
    posix_read_time = data[counters["CP_F_POSIX_READ_TIME"]]["value"]
    posix_write_time = data[counters["CP_F_POSIX_WRITE_TIME"]]["value"]
    posix_meta_time = data[counters["CP_F_POSIX_META_TIME"]]["value"]

    posix_read_sum = np.sum(posix_read_time)
    posix_write_sum = np.sum(posix_write_time)
    posix_meta_sum = np.sum(posix_meta_time)

    posix_read_perc = (posix_read_sum/(runtime*nprocs))*100
    posix_write_perc = (posix_write_sum/(runtime*nprocs))*100
    posix_meta_perc = (posix_meta_sum/(runtime*nprocs))*100

    posix_io_time = (posix_read_perc + posix_write_perc + posix_meta_perc)/100 * runtime

    posix_other = ((runtime*nprocs - posix_read_sum -
                   posix_write_sum - posix_meta_sum)/(runtime*nprocs))*100

    posix = ["POSIX:", str(posix_other), str(posix_read_perc),
             str(posix_write_perc), str(posix_meta_perc)]

    mpi_read_time = data[counters["CP_F_MPI_READ_TIME"]]["value"]
    mpi_write_time = data[counters["CP_F_MPI_WRITE_TIME"]]["value"]
    mpi_meta_time = data[counters["CP_F_MPI_META_TIME"]]["value"]

    mpi_read_sum = np.sum(mpi_read_time)
    mpi_write_sum = np.sum(mpi_write_time)
    mpi_meta_sum = np.sum(mpi_meta_time)

    mpi_read_perc = (mpi_read_sum/(runtime*nprocs))*100
    mpi_write_perc = (mpi_write_sum/(runtime*nprocs))*100
    mpi_meta_perc = (mpi_meta_sum/(runtime*nprocs))*100

    mpi_io_time = (mpi_read_perc + mpi_write_perc + mpi_meta_perc)/100 * runtime

    mpi_other = ((runtime*nprocs - mpi_read_sum -
                  mpi_write_sum - mpi_meta_sum)/(runtime*nprocs))*100

    mpi = ["MPI-IO:", str(mpi_other), str(mpi_read_perc),
           str(mpi_write_perc), str(mpi_meta_perc)]

    table_head = ["", "other", "read", "write", "metadata"]

    io_table = [table_head, posix, mpi]
    table_head = "Average I/O cost per process"
    write_jube_table(table_head, io_table)

    for elem in posix:
        out_file.write('{0} '.format(elem))
    out_file.write('\n')
    for elem in mpi:
        out_file.write('{0} '.format(elem))
    out_file.write('\n')

    total_io_time = mpi_io_time if mpi_io_time > posix_io_time else posix_io_time
    out_file.write('io_time: {0}\n\n'.format(total_io_time))


##############################################################
#darshan3: nur counter namen geaendert!
def io_percentage3(nprocs, runtime, counters, data, out_file):
    """
    calculates average I/O cost per process (top left histogram of darshan pdf)

    :param nprocs: number of processes
    :param runtime: application runtime
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """
    posix_read_time = data[counters["POSIX_F_READ_TIME"]]["value"]
    posix_write_time = data[counters["POSIX_F_WRITE_TIME"]]["value"]
    posix_meta_time = data[counters["POSIX_F_META_TIME"]]["value"]

    posix_read_sum = np.sum(posix_read_time)
    posix_write_sum = np.sum(posix_write_time)
    posix_meta_sum = np.sum(posix_meta_time)

    posix_read_perc = (posix_read_sum/(runtime*nprocs))*100
    posix_write_perc = (posix_write_sum/(runtime*nprocs))*100
    posix_meta_perc = (posix_meta_sum/(runtime*nprocs))*100

    posix_io_time = (posix_read_perc + posix_write_perc + posix_meta_perc)/100 * runtime

    posix_other = ((runtime*nprocs - posix_read_sum -
                   posix_write_sum - posix_meta_sum)/(runtime*nprocs))*100

    posix = ["POSIX:", str(posix_other), str(posix_read_perc),
             str(posix_write_perc), str(posix_meta_perc)]

    if "MPIIO_INDEP_OPENS" in counters:
        mpi_read_time = data[counters["MPIIO_F_READ_TIME"]]["value"]
        mpi_write_time = data[counters["MPIIO_F_WRITE_TIME"]]["value"]
        mpi_meta_time = data[counters["MPIIO_F_META_TIME"]]["value"]

        mpi_read_sum = np.sum(mpi_read_time)
        mpi_write_sum = np.sum(mpi_write_time)
        mpi_meta_sum = np.sum(mpi_meta_time)

        mpi_read_perc = (mpi_read_sum/(runtime*nprocs))*100
        mpi_write_perc = (mpi_write_sum/(runtime*nprocs))*100
        mpi_meta_perc = (mpi_meta_sum/(runtime*nprocs))*100

        mpi_io_time = (mpi_read_perc + mpi_write_perc + mpi_meta_perc)/100 * runtime

        mpi_other = ((runtime*nprocs - mpi_read_sum -
                      mpi_write_sum - mpi_meta_sum)/(runtime*nprocs))*100
    else:
        mpi_other = 0
        mpi_read_perc = 0
        mpi_write_perc = 0
        mpi_meta_perc = 0
        mpi_io_time = 0

    mpi = ["MPI-IO:", str(mpi_other), str(mpi_read_perc),
           str(mpi_write_perc), str(mpi_meta_perc)]

    table_head = ["", "other", "read", "write", "metadata"]

    io_table = [table_head, posix, mpi]
    table_head = "Average I/O cost per process"
    write_jube_table(table_head, io_table)

    for elem in posix:
        out_file.write('{0} '.format(elem))
    out_file.write('\n')
    for elem in mpi:
        out_file.write('{0} '.format(elem))
    out_file.write('\n')

    total_io_time = mpi_io_time if mpi_io_time > posix_io_time else posix_io_time
    out_file.write('io_time: {0}\n\n'.format(total_io_time))

##############################################################
#darshan3.1:add stdio counters
def io_percentage31(nprocs, runtime, counters, data, out_file):
    """
    calculates average I/O cost per process (top left histogram of darshan pdf)

    :param nprocs: number of processes
    :param runtime: application runtime
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """
    if "POSIX_OPENS" in counters:
        posix_read_time = data[counters["POSIX_F_READ_TIME"]]["value"]
        posix_write_time = data[counters["POSIX_F_WRITE_TIME"]]["value"]
        posix_meta_time = data[counters["POSIX_F_META_TIME"]]["value"]
        
        posix_read_sum = np.sum(posix_read_time)
        posix_write_sum = np.sum(posix_write_time)
        posix_meta_sum = np.sum(posix_meta_time)
        
        posix_read_perc = (posix_read_sum/(runtime*nprocs))*100
        posix_write_perc = (posix_write_sum/(runtime*nprocs))*100
        posix_meta_perc = (posix_meta_sum/(runtime*nprocs))*100
        
        posix_io_time = (posix_read_perc + posix_write_perc + posix_meta_perc)/100 * runtime

        posix_other = ((runtime*nprocs - posix_read_sum -
                        posix_write_sum - posix_meta_sum)/(runtime*nprocs))*100        
    else:
        posix_read_perc = 0
        posix_write_perc = 0
        posix_meta_perc = 0
        posix_io_time = 0
        posix_other = 0

    posix = ["POSIX:", str(posix_other), str(posix_read_perc),
                 str(posix_write_perc), str(posix_meta_perc)]

    if "MPIIO_INDEP_OPENS" in counters:
        mpi_read_time = data[counters["MPIIO_F_READ_TIME"]]["value"]
        mpi_write_time = data[counters["MPIIO_F_WRITE_TIME"]]["value"]
        mpi_meta_time = data[counters["MPIIO_F_META_TIME"]]["value"]

        mpi_read_sum = np.sum(mpi_read_time)
        mpi_write_sum = np.sum(mpi_write_time)
        mpi_meta_sum = np.sum(mpi_meta_time)

        mpi_read_perc = (mpi_read_sum/(runtime*nprocs))*100
        mpi_write_perc = (mpi_write_sum/(runtime*nprocs))*100
        mpi_meta_perc = (mpi_meta_sum/(runtime*nprocs))*100

        mpi_io_time = (mpi_read_perc + mpi_write_perc + mpi_meta_perc)/100 * runtime

        mpi_other = ((runtime*nprocs - mpi_read_sum -
                      mpi_write_sum - mpi_meta_sum)/(runtime*nprocs))*100
    else:
        mpi_other = 0
        mpi_read_perc = 0
        mpi_write_perc = 0
        mpi_meta_perc = 0
        mpi_io_time = 0

    mpi = ["MPI-IO:", str(mpi_other), str(mpi_read_perc),
           str(mpi_write_perc), str(mpi_meta_perc)]

    if "STDIO_OPENS" in counters:
        stdio_read_time = data[counters["STDIO_F_READ_TIME"]]["value"]
        stdio_write_time = data[counters["STDIO_F_WRITE_TIME"]]["value"]
        stdio_meta_time = data[counters["STDIO_F_META_TIME"]]["value"]
        
        stdio_read_sum = np.sum(stdio_read_time)
        stdio_write_sum = np.sum(stdio_write_time)
        stdio_meta_sum = np.sum(stdio_meta_time)
        
        stdio_read_perc = (stdio_read_sum/(runtime*nprocs))*100
        stdio_write_perc = (stdio_write_sum/(runtime*nprocs))*100
        stdio_meta_perc = (stdio_meta_sum/(runtime*nprocs))*100
        
        stdio_io_time = (stdio_read_perc + stdio_write_perc + stdio_meta_perc)/100 * runtime

        stdio_other = ((runtime*nprocs - stdio_read_sum -
                        stdio_write_sum - stdio_meta_sum)/(runtime*nprocs))*100        
    else:
        stdio_read_perc = 0
        stdio_write_perc = 0
        stdio_meta_perc = 0
        stdio_io_time = 0
        stdio_other = 0

    stdio = ["STDIO:", str(stdio_other), str(stdio_read_perc),
                 str(stdio_write_perc), str(stdio_meta_perc)]


    table_head = ["", "other", "read", "write", "metadata"]

    io_table = [table_head, posix, mpi, stdio]
    table_head = "Average I/O cost per process"
    write_jube_table(table_head, io_table)

    # do not write stdio in output file
    for elem in posix:
        out_file.write('{0} '.format(elem))
    out_file.write('\n')
    for elem in mpi:
        out_file.write('{0} '.format(elem))
    out_file.write('\n')

    total_io_time = np.max([posix_io_time, mpi_io_time, stdio_io_time])
    out_file.write('io_time: {0}\n\n'.format(total_io_time))
##############################################################


def operation_counts(counters, data, out_file):
    """
    creates a table for the io operation counts
    (in darshan_job_summary first page, first row, second column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """

    posix_opens = data[counters["CP_POSIX_OPENS"]]["value"]
    posix_fopens = data[counters["CP_POSIX_FOPENS"]]["value"]
    posix_fsyncs = data[counters["CP_POSIX_FSYNCS"]]["value"]
    posix_fdsyncs = data[counters["CP_POSIX_FDSYNCS"]]["value"]

    posix_reads = data[counters["CP_POSIX_READS"]]["value"]
    posix_freads = data[counters["CP_POSIX_FREADS"]]["value"]
    indep_reads = data[counters["CP_INDEP_READS"]]["value"]
    coll_reads = data[counters["CP_COLL_READS"]]["value"]
    posix_writes = data[counters["CP_POSIX_WRITES"]]["value"]
    posix_fwrites = data[counters["CP_POSIX_FWRITES"]]["value"]
    indep_writes = data[counters["CP_INDEP_WRITES"]]["value"]
    coll_writes = data[counters["CP_COLL_WRITES"]]["value"]
    indep_opens = data[counters["CP_INDEP_OPENS"]]["value"]
    coll_opens = data[counters["CP_COLL_OPENS"]]["value"]
    posix_stats = data[counters["CP_POSIX_STATS"]]["value"]
    posix_seeks = data[counters["CP_POSIX_SEEKS"]]["value"]
    posix_mmaps = data[counters["CP_POSIX_MMAPS"]]["value"]

    sum_posix_reads = np.sum(posix_reads) + np.sum(posix_freads)
    sum_indep_reads = np.sum(indep_reads)
    sum_coll_reads = np.sum(coll_reads)
    sum_posix_writes = np.sum(posix_writes) + np.sum(posix_fwrites)
    sum_indep_writes = np.sum(indep_writes)
    sum_coll_writes = np.sum(coll_writes)
    sum_posix_opens = np.sum(posix_opens) + np.sum(posix_fopens)
    sum_indep_opens = np.sum(indep_opens)
    sum_coll_opens = np.sum(coll_opens)
    sum_posix_stats = np.sum(posix_stats)
    sum_posix_seeks = np.sum(posix_seeks)
    sum_posix_mmaps = np.sum(posix_mmaps)
    sum_posix_fsyncs = np.sum(posix_fsyncs) + np.sum(posix_fdsyncs)

    reads = ['Read', int(sum_posix_reads), int(sum_indep_reads), int(sum_coll_reads)]
    writes = ['Write', int(sum_posix_writes), int(sum_indep_writes), int(sum_coll_writes)]
    opens = ['Open', int(sum_posix_opens), int(sum_indep_opens), int(sum_coll_opens)]
    stats = ['Stat', int(sum_posix_stats), 0, 0]
    seeks = ['Seek', int(sum_posix_seeks), 0, 0]
    mmaps = ['Mmap', int(sum_posix_mmaps), 0, 0]
    syncs = ['FSync', int(sum_posix_fsyncs), 0, 0]

    out_file.write('# read write open stats seeks mmaps fsyncs\n')
    out_file.write('posix calls: {0} {1} {2} {3} {4} {5} {6}\n\n'.format(
        int(sum_posix_reads), int(sum_posix_writes), int(sum_posix_opens),
        int(sum_posix_stats), int(sum_posix_seeks), int(sum_posix_mmaps),
        int(sum_posix_fsyncs)))

    col_head = ['', 'POSIX', 'MPI-IO Indep.', 'MPI-IO Coll.']
    op_cnt_table = [col_head, reads, writes, opens, stats, seeks, mmaps, syncs]
    op_cnt_table = [[str(col) for col in row] for row in op_cnt_table]
    table_head = "I/O Operation Counts"
    write_jube_table(table_head, op_cnt_table)


###################################################################################################
#darshan3: counter namen geaendert und counter hinzugefuegt
def operation_counts3(counters, data, out_file):
    """
    creates a table for the io operation counts
    (in darshan_job_summary first page, first row, second column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """

    posix_reads = data[counters["POSIX_READS"]]["value"]
    posix_freads = data[counters["POSIX_FREADS"]]["value"]
    posix_writes = data[counters["POSIX_WRITES"]]["value"]
    posix_fwrites = data[counters["POSIX_FWRITES"]]["value"]
    posix_opens = data[counters["POSIX_OPENS"]]["value"]
    posix_fopens = data[counters["POSIX_FOPENS"]]["value"]
    posix_stats = data[counters["POSIX_STATS"]]["value"]
    posix_seeks = data[counters["POSIX_SEEKS"]]["value"]
    posix_mmaps = data[counters["POSIX_MMAPS"]]["value"]
    posix_fsyncs = data[counters["POSIX_FSYNCS"]]["value"]
    posix_fdsyncs = data[counters["POSIX_FDSYNCS"]]["value"]

    sum_posix_reads = np.sum(posix_reads) + np.sum(posix_freads)
    sum_posix_writes = np.sum(posix_writes) + np.sum(posix_fwrites)
    sum_posix_opens = np.sum(posix_opens) + np.sum(posix_fopens)
    sum_posix_stats = np.sum(posix_stats)
    sum_posix_seeks = np.sum(posix_seeks)
    sum_posix_mmaps = np.sum(posix_mmaps)
    sum_posix_fsyncs = np.sum(posix_fsyncs) + np.sum(posix_fdsyncs)

    if "MPIIO_INDEP_OPENS" in counters:
        indep_reads = data[counters["MPIIO_INDEP_READS"]]["value"]
        coll_reads = data[counters["MPIIO_COLL_READS"]]["value"]
        indep_writes = data[counters["MPIIO_INDEP_WRITES"]]["value"]
        coll_writes = data[counters["MPIIO_COLL_WRITES"]]["value"]
        indep_opens = data[counters["MPIIO_INDEP_OPENS"]]["value"]
        coll_opens = data[counters["MPIIO_COLL_OPENS"]]["value"]
        coll_syncs = data[counters["MPIIO_SYNCS"]]["value"]

        sum_indep_reads = np.sum(indep_reads)
        sum_coll_reads = np.sum(coll_reads)
        sum_indep_writes = np.sum(indep_writes)
        sum_coll_writes = np.sum(coll_writes)
        sum_indep_opens = np.sum(indep_opens)
        sum_coll_opens = np.sum(coll_opens)
        sum_coll_syncs = np.sum(coll_syncs)
    else:
        sum_indep_reads = 0
        sum_coll_reads = 0
        sum_indep_writes = 0
        sum_coll_writes = 0
        sum_indep_opens = 0
        sum_coll_opens = 0
        sum_coll_syncs = 0

    reads = ['Read', int(sum_posix_reads), int(sum_indep_reads), int(sum_coll_reads)]
    writes = ['Write', int(sum_posix_writes), int(sum_indep_writes), int(sum_coll_writes)]
    opens = ['Open', int(sum_posix_opens), int(sum_indep_opens), int(sum_coll_opens)]
    stats = ['Stat', int(sum_posix_stats), 0, 0]
    seeks = ['Seek', int(sum_posix_seeks), 0, 0]
    mmaps = ['Mmap', int(sum_posix_mmaps), 0, 0]
    syncs = ['FSync', int(sum_posix_fsyncs), 0, int(sum_coll_syncs)]

    out_file.write('# read write open stats seeks mmaps fsyncs\n')
    out_file.write('posix calls: {0} {1} {2} {3} {4} {5} {6}\n'.format(
        int(sum_posix_reads), int(sum_posix_writes), int(sum_posix_opens),
        int(sum_posix_stats), int(sum_posix_seeks), int(sum_posix_mmaps),
        int(sum_posix_fsyncs)))
    out_file.write('mpiio independent calls: {0} {1} {2} {3} {4} {5} {6}\n'.format(
        int(sum_indep_reads), int(sum_indep_writes), int(sum_indep_opens),
        0, 0, 0, 0))
    out_file.write('mpiio collective calls: {0} {1} {2} {3} {4} {5} {6}\n\n'.format(
        int(sum_coll_reads), int(sum_coll_writes), int(sum_coll_opens),
        0, 0, 0, int(sum_coll_syncs)))

    col_head = ['', 'POSIX', 'MPI-IO Indep.', 'MPI-IO Coll.']
    op_cnt_table = [col_head, reads, writes, opens, stats, seeks, mmaps, syncs]
    op_cnt_table = [[str(col) for col in row] for row in op_cnt_table]
    table_head = "I/O Operation Counts"
    write_jube_table(table_head, op_cnt_table)

###################################################################################################
#darshan3.1: counter namen freads, fwrites und fopen existieren nicht mehr; testen, ob posix_opens existiert
def operation_counts31(counters, data, out_file):
    """
    creates a table for the io operation counts
    (in darshan_job_summary first page, first row, second column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """

    if "POSIX_OPENS" in counters:
        posix_reads = data[counters["POSIX_READS"]]["value"]
        posix_writes = data[counters["POSIX_WRITES"]]["value"]
        posix_opens = data[counters["POSIX_OPENS"]]["value"]
        posix_stats = data[counters["POSIX_STATS"]]["value"]
        posix_seeks = data[counters["POSIX_SEEKS"]]["value"]
        posix_mmaps = data[counters["POSIX_MMAPS"]]["value"]
        posix_fsyncs = data[counters["POSIX_FSYNCS"]]["value"]
        posix_fdsyncs = data[counters["POSIX_FDSYNCS"]]["value"]
    
        sum_posix_reads = np.sum(posix_reads)
        sum_posix_writes = np.sum(posix_writes)
        sum_posix_opens = np.sum(posix_opens)
        sum_posix_stats = np.sum(posix_stats)
        sum_posix_seeks = np.sum(posix_seeks)
        sum_posix_mmaps = np.sum(posix_mmaps)
        sum_posix_fsyncs = np.sum(posix_fsyncs) + np.sum(posix_fdsyncs)
    else:
        sum_posix_reads = 0
        sum_posix_writes = 0
        sum_posix_opens = 0
        sum_posix_stats = 0
        sum_posix_seeks = 0
        sum_posix_mmaps = 0
        sum_posix_fsyncs = 0

    if "MPIIO_INDEP_OPENS" in counters:
        indep_reads = data[counters["MPIIO_INDEP_READS"]]["value"]
        coll_reads = data[counters["MPIIO_COLL_READS"]]["value"]
        indep_writes = data[counters["MPIIO_INDEP_WRITES"]]["value"]
        coll_writes = data[counters["MPIIO_COLL_WRITES"]]["value"]
        indep_opens = data[counters["MPIIO_INDEP_OPENS"]]["value"]
        coll_opens = data[counters["MPIIO_COLL_OPENS"]]["value"]
        coll_syncs = data[counters["MPIIO_SYNCS"]]["value"]

        sum_indep_reads = np.sum(indep_reads)
        sum_coll_reads = np.sum(coll_reads)
        sum_indep_writes = np.sum(indep_writes)
        sum_coll_writes = np.sum(coll_writes)
        sum_indep_opens = np.sum(indep_opens)
        sum_coll_opens = np.sum(coll_opens)
        sum_coll_syncs = np.sum(coll_syncs)
    else:
        sum_indep_reads = 0
        sum_coll_reads = 0
        sum_indep_writes = 0
        sum_coll_writes = 0
        sum_indep_opens = 0
        sum_coll_opens = 0
        sum_coll_syncs = 0

    if "STDIO_OPENS" in counters:
        stdio_reads = data[counters["STDIO_READS"]]["value"]
        stdio_writes = data[counters["STDIO_WRITES"]]["value"]
        stdio_opens = data[counters["STDIO_OPENS"]]["value"]
        stdio_seeks = data[counters["STDIO_SEEKS"]]["value"]
        stdio_fsyncs = data[counters["STDIO_FLUSHES"]]["value"]

        sum_stdio_reads = np.sum(stdio_reads)
        sum_stdio_writes = np.sum(stdio_writes)
        sum_stdio_opens = np.sum(stdio_opens)
        sum_stdio_seeks = np.sum(stdio_seeks)
        sum_stdio_fsyncs = np.sum(stdio_fsyncs)
    else:
        sum_stdio_reads = 0
        sum_stdio_writes = 0
        sum_stdio_opens = 0
        sum_stdio_seeks = 0
        sum_stdio_fsyncs = 0

    reads = ['Read', int(sum_posix_reads), int(sum_indep_reads), int(sum_coll_reads), int(sum_stdio_reads)]
    writes = ['Write', int(sum_posix_writes), int(sum_indep_writes), int(sum_coll_writes), int(sum_stdio_writes)]
    opens = ['Open', int(sum_posix_opens), int(sum_indep_opens), int(sum_coll_opens), int(sum_stdio_opens)]
    stats = ['Stat', int(sum_posix_stats), 0, 0, 0]
    seeks = ['Seek', int(sum_posix_seeks), 0, 0, int(sum_stdio_seeks)]
    mmaps = ['Mmap', int(sum_posix_mmaps), 0, 0, 0]
    syncs = ['FSync', int(sum_posix_fsyncs), 0, int(sum_coll_syncs), int(sum_stdio_fsyncs)]

    # do not write stdio in output file
    out_file.write('# read write open stats seeks mmaps fsyncs\n')
    out_file.write('posix calls: {0} {1} {2} {3} {4} {5} {6}\n'.format(
        int(sum_posix_reads), int(sum_posix_writes), int(sum_posix_opens),
        int(sum_posix_stats), int(sum_posix_seeks), int(sum_posix_mmaps),
        int(sum_posix_fsyncs)))
    out_file.write('mpiio independent calls: {0} {1} {2} {3} {4} {5} {6}\n'.format(
        int(sum_indep_reads), int(sum_indep_writes), int(sum_indep_opens),
        0, 0, 0, 0))
    out_file.write('mpiio collective calls: {0} {1} {2} {3} {4} {5} {6}\n\n'.format(
        int(sum_coll_reads), int(sum_coll_writes), int(sum_coll_opens),
        0, 0, 0, int(sum_coll_syncs)))

    col_head = ['', 'POSIX', 'MPI-IO Indep.', 'MPI-IO Coll.', 'STDIO']
    op_cnt_table = [col_head, reads, writes, opens, stats, seeks, mmaps, syncs]
    op_cnt_table = [[str(col) for col in row] for row in op_cnt_table]
    table_head = "I/O Operation Counts"
    write_jube_table(table_head, op_cnt_table)

##########################################################################


def io_sizes(counters, data):
    """
    creates a table that includes the io size counts
    (in darshan_job_summary first page, second row, first column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """

    column_heads = ['0-100', '100-1K', '1K-10K', '10K-100K', '100K-1M',
                    '1M-4M', '4M-10M', '10M-100M', '100M-1G', '1G+']
    acc_sizes = [col.replace('-', '_').replace('+', '_PLUS')
                     for col in column_heads]
    posix_read_cnt = 'CP_SIZE_READ_{0}'
    posix_write_cnt = 'CP_SIZE_WRITE_{0}'

    posix_reads = get_detailed_access_sizes(posix_read_cnt, acc_sizes, counters, data)
    posix_writes = get_detailed_access_sizes(posix_write_cnt, acc_sizes, counters, data)

    posix_reads.insert(0, 'POSIX Read')
    posix_writes.insert(0, 'POSIX Write')
    column_heads.insert(0, '')
    table_head = 'I/O Sizes'
    io_size_table = [column_heads, posix_reads, posix_writes]
    write_jube_table(table_head, io_size_table)


def get_detailed_access_sizes(cnt_name, acc_sizes, counters, data):
    """
    generate counter names for every access size range and
    sum up the data for each counter separately

    :param cnt_name: first part of the counter name
    :param acc_sizes: second part of the counter name
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    cnt_str = [cnt_name.format(size) for size in acc_sizes]
    cnt_data = [str(int(np.sum(data[counters[cnt]]["value"])))
                for cnt in cnt_str]
    return cnt_data


###############################################################################
#darshan3: counter umbenannt und hinzugefuegt
def io_sizes3(counters, data):
    """
    creates a table that includes the io size counts
    (in darshan_job_summary first page, second row, first column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """

    column_heads = ['0-100', '100-1K', '1K-10K', '10K-100K', '100K-1M',
                    '1M-4M', '4M-10M', '10M-100M', '100M-1G', '1G+']
    acc_sizes = [col.replace('-', '_').replace('+', '_PLUS')
                 for col in column_heads]
    posix_read_cnt = 'POSIX_SIZE_READ_{0}'
    posix_write_cnt = 'POSIX_SIZE_WRITE_{0}'

    posix_reads = get_detailed_access_sizes(posix_read_cnt, acc_sizes, counters, data)
    posix_writes = get_detailed_access_sizes(posix_write_cnt, acc_sizes, counters, data)
    posix_reads.insert(0, 'POSIX Read')
    posix_writes.insert(0, 'POSIX Write')

    if "MPIIO_INDEP_OPENS" in counters:
        mpiio_read_cnt = 'MPIIO_SIZE_READ_AGG_{0}'
        mpiio_write_cnt = 'MPIIO_SIZE_WRITE_AGG_{0}'
        mpiio_reads = get_detailed_access_sizes(mpiio_read_cnt, acc_sizes, counters, data)
        mpiio_writes = get_detailed_access_sizes(mpiio_write_cnt, acc_sizes, counters, data)
    else:
        mpiio_reads = [str(0)] * len(column_heads)
        mpiio_writes = [str(0)] * len(column_heads)
    mpiio_reads.insert(0, 'MPIIO Read')
    mpiio_writes.insert(0, 'MPIIO Write')

    column_heads.insert(0, '')
    table_head = 'I/O Sizes'
    io_size_table = [column_heads, posix_reads, posix_writes,
                     mpiio_reads, mpiio_writes]
    write_jube_table(table_head, io_size_table)

###############################################################################
#darshan3.1: check if POSIX_OPENS exists
def io_sizes31(counters, data):
    """
    creates a table that includes the io size counts
    (in darshan_job_summary first page, second row, first column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """

    column_heads = ['0-100', '100-1K', '1K-10K', '10K-100K', '100K-1M',
                    '1M-4M', '4M-10M', '10M-100M', '100M-1G', '1G+']
    acc_sizes = [col.replace('-', '_').replace('+', '_PLUS')
                 for col in column_heads]

    if "POSIX_OPENS" in counters:
        posix_read_cnt = 'POSIX_SIZE_READ_{0}'
        posix_write_cnt = 'POSIX_SIZE_WRITE_{0}'
        posix_reads = get_detailed_access_sizes(posix_read_cnt, acc_sizes, counters, data)
        posix_writes = get_detailed_access_sizes(posix_write_cnt, acc_sizes, counters, data)
    else:
        posix_reads = [str(0)] * len(column_heads)
        posix_writes = [str(0)] * len(column_heads)
    posix_reads.insert(0, 'POSIX Read')
    posix_writes.insert(0, 'POSIX Write')

    if "MPIIO_INDEP_OPENS" in counters:
        mpiio_read_cnt = 'MPIIO_SIZE_READ_AGG_{0}'
        mpiio_write_cnt = 'MPIIO_SIZE_WRITE_AGG_{0}'
        mpiio_reads = get_detailed_access_sizes(mpiio_read_cnt, acc_sizes, counters, data)
        mpiio_writes = get_detailed_access_sizes(mpiio_write_cnt, acc_sizes, counters, data)
    else:
        mpiio_reads = [str(0)] * len(column_heads)
        mpiio_writes = [str(0)] * len(column_heads)
    mpiio_reads.insert(0, 'MPIIO Read')
    mpiio_writes.insert(0, 'MPIIO Write')

    column_heads.insert(0, '')
    table_head = 'I/O Sizes'
    io_size_table = [column_heads, posix_reads, posix_writes,
                     mpiio_reads, mpiio_writes]
    write_jube_table(table_head, io_size_table)

##############################################################################


def io_pattern(counters, data):
    """
    creates a table that includes the io pattern
    (in darshan_job_summary first page, second row, second column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    posix_reads = int(np.sum(data[counters["CP_POSIX_READS"]]["value"]))
    posix_freads = int(np.sum(data[counters["CP_POSIX_FREADS"]]["value"]))
    seq_reads = int(np.sum(data[counters["CP_SEQ_READS"]]["value"]))
    consec_reads = int(np.sum(data[counters["CP_CONSEC_READS"]]["value"]))

    posix_writes = int(np.sum(data[counters["CP_POSIX_WRITES"]]["value"]))
    posix_fwrites = int(np.sum(data[counters["CP_POSIX_FWRITES"]]["value"]))
    seq_writes = int(np.sum(data[counters["CP_SEQ_WRITES"]]["value"]))
    consec_writes = int(np.sum(data[counters["CP_CONSEC_WRITES"]]["value"]))

    column_heads = ['', 'Total', 'Sequential', 'Consecutive']
    reads = ['Read', str(posix_reads + posix_freads),
             str(seq_reads), str(consec_reads)]
    writes = ['Write', str(posix_writes + posix_fwrites),
              str(seq_writes), str(consec_writes)]
    pattern_table = [column_heads, reads, writes]
    table_head = 'I/O Pattern'
    write_jube_table(table_head, pattern_table)
##############################################################################

#darshan3: counter namen geaendert
def io_pattern3(counters, data):
    """
    creates a table that includes the io pattern
    (in darshan_job_summary first page, second row, second column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    posix_reads = int(np.sum(data[counters["POSIX_READS"]]["value"]))
    posix_freads = int(np.sum(data[counters["POSIX_FREADS"]]["value"]))
    seq_reads = int(np.sum(data[counters["POSIX_SEQ_READS"]]["value"]))
    consec_reads = int(np.sum(data[counters["POSIX_CONSEC_READS"]]["value"]))

    posix_writes = int(np.sum(data[counters["POSIX_WRITES"]]["value"]))
    posix_fwrites = int(np.sum(data[counters["POSIX_FWRITES"]]["value"]))
    seq_writes = int(np.sum(data[counters["POSIX_SEQ_WRITES"]]["value"]))
    consec_writes = int(np.sum(data[counters["POSIX_CONSEC_WRITES"]]["value"]))

    column_heads = ['', 'Total', 'Sequential', 'Consecutive']
    reads = ['Read', str(posix_reads + posix_freads),
             str(seq_reads), str(consec_reads)]
    writes = ['Write', str(posix_writes + posix_fwrites),
              str(seq_writes), str(consec_writes)]
    pattern_table = [column_heads, reads, writes]
    table_head = 'I/O Pattern'
    write_jube_table(table_head, pattern_table)

##############################################################################
#darshan3.1: remove freads/fwrites; check if posix_opens exists
def io_pattern31(counters, data):
    """
    creates a table that includes the io pattern
    (in darshan_job_summary first page, second row, second column)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    if "POSIX_OPENS" in counters:
        posix_reads = int(np.sum(data[counters["POSIX_READS"]]["value"]))
        seq_reads = int(np.sum(data[counters["POSIX_SEQ_READS"]]["value"]))
        consec_reads = int(np.sum(data[counters["POSIX_CONSEC_READS"]]["value"]))

        posix_writes = int(np.sum(data[counters["POSIX_WRITES"]]["value"]))
        seq_writes = int(np.sum(data[counters["POSIX_SEQ_WRITES"]]["value"]))
        consec_writes = int(np.sum(data[counters["POSIX_CONSEC_WRITES"]]["value"]))
    else:
        posix_reads = 0
        seq_reads = 0
        consec_reads = 0
        posix_writes = 0
        seq_writes = 0
        consec_writes = 0

    column_heads = ['', 'Total', 'Sequential', 'Consecutive']
    reads = ['Read', str(posix_reads), str(seq_reads), str(consec_reads)]
    writes = ['Write', str(posix_writes), str(seq_writes), str(consec_writes)]
    pattern_table = [column_heads, reads, writes]
    table_head = 'I/O Pattern'
    write_jube_table(table_head, pattern_table)

##############################################################################


def access_size(counters, data, out_file, length=4):
    """
    calculates the 4 most common access sizes (bottom left table of darshan pdf)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :param length: number of entries in the output table
    :return:
    """

    # get the four most common access sizes of each file
    ac_size = np.array([data[counters["CP_ACCESS{0}_ACCESS".format(i)]]["value"]
                        for i in range(1, 5)], np.int)
    # get the count of the four most common access sizes of each file
    ac_cnt = np.array([data[counters["CP_ACCESS{0}_COUNT".format(i)]]["value"]
                       for i in range(1, 5)], np.int)
    merged = np.array(list(zip(ac_size.flatten(), ac_cnt.flatten())))
    ac_sorted = merged[np.argsort(merged[:, 0])]
    # get different access sizes
    ac_val = [key for key, group in groupby(ac_sorted[:, 0])]
    # sum up the count of each access size (sorted by increasing access size
    cnt_sum = np.array([[val, np.sum(ac_sorted[ac_sorted[:, 0] == val][:, 1])]
                       for val in ac_val])
    # cnt_sum sorted by decreasing access counter
    cnt_sum_sort = cnt_sum[np.argsort(cnt_sum[:, 1])[::-1]]

    length = length if length > 0 else len(cnt_sum_sort)

    most_common_ac = [[str(cell) for cell in row]
                      for row in cnt_sum_sort[:length]]
    most_common_titel = ['access size', 'count']
    most_common_ac.insert(0, most_common_titel)

    table_head = "Most Common Access Sizes"
    write_jube_table(table_head, most_common_ac)

    out_file.write('# {0}:\n'.format(table_head))
    for i, title in enumerate(most_common_titel):
        out_file.write('{0}: '.format(title))
        for elem in np.array(most_common_ac[1:]).T[i]:
            out_file.write("{0} ".format(elem))
        out_file.write('\n')

    avg_io_ac_size = np.inner(cnt_sum_sort[:, 0], cnt_sum_sort[:, 1]) / \
                     np.sum(cnt_sum_sort[:, 1])
    out_file.write('{0} avg_io_ac_size: {1}\n'.format('POSIX', avg_io_ac_size))
    out_file.write('\n')
##############################################################################
#darshan3:counter namen geaendert, in 3 Methoden geteilt, paatern in darshan.jube geaendert
#might cause in error for version 3.1, does not check if posix_opens exist
def access_size3(counters, data, out_file, length=4):
    """
    calculates the most common access sizes (bottom left table of darshan pdf)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :param length: number of entries in the output table
    :return:
    """
    posix = 'POSIX'
    mpiio = 'MPIIO'
    posix_acc_sizes = get_access_size_data(posix, counters, data)
    print_access_size(posix, posix_acc_sizes, out_file, length)

    if "MPIIO_INDEP_OPENS" in counters:
        mpiio_acc_sizes = get_access_size_data(mpiio, counters, data)
        print_access_size(mpiio, mpiio_acc_sizes, out_file, length)
##############################################################################
#darshan3.1: 
def access_size31(counters, data, out_file, length=4):
    """
    calculates the most common access sizes (bottom left table of darshan pdf)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :param length: number of entries in the output table
    :return:
    """
    posix = 'POSIX'
    mpiio = 'MPIIO'
    if "POSIX_OPENS" in counters:
        posix_acc_sizes = get_access_size_data(posix, counters, data)
        print_access_size(posix, posix_acc_sizes, out_file, length)

    if "MPIIO_INDEP_OPENS" in counters:
        mpiio_acc_sizes = get_access_size_data(mpiio, counters, data)
        print_access_size(mpiio, mpiio_acc_sizes, out_file, length)

    if "POSIX_OPENS" not in counters and "MPIIO_INDEP_OPENS" not in counters:
        out_file.write('avg_io_ac_size: 0.0\n')
        out_file.write('\n')


def get_access_size_data(io_type, counters, data):
    """
    calculates the most common access sizes (bottom left table of darshan pdf)

    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :param length: number of entries in the output table
    :return:
    """

    # get the four most common access sizes of each file
    ac_size = np.array([data[counters["{0}_ACCESS{1}_ACCESS".format(io_type, i)]]["value"]
                        for i in range(1, 5)], np.int)
    # get the count of the four most common access sizes of each file
    ac_cnt = np.array([data[counters["{0}_ACCESS{1}_COUNT".format(io_type, i)]]["value"]
                       for i in range(1, 5)], np.int)
    merged = np.array(list(zip(ac_size.flatten(), ac_cnt.flatten())))
    ac_sorted = merged[np.argsort(merged[:, 0])]
    # get different access sizes
    ac_val = [key for key, group in groupby(ac_sorted[:, 0])]
    # sum up the count of each access size (sorted by increasing access size
    cnt_sum = np.array([[val, np.sum(ac_sorted[ac_sorted[:, 0] == val][:, 1])]
                       for val in ac_val])
    # cnt_sum sorted by decreasing access counter
    cnt_sum_sort = cnt_sum[np.argsort(cnt_sum[:, 1])[::-1]]

    return cnt_sum_sort


def print_access_size(io_type, acc_sizes, out_file, length):

    length = length if length > 0 else len(acc_sizes)

    most_common_ac = [[str(cell) for cell in row]
                      for row in acc_sizes[:length]]
    most_common_titel = ['{0} access size'.format(io_type),
                         '{0} count'.format(io_type)]
    most_common_ac.insert(0, most_common_titel)

    table_head = '{0} Most Common Access Sizes'.format(io_type)
    write_jube_table(table_head, most_common_ac)

    out_file.write('# {0}:\n'.format(table_head))
    for i, title in enumerate(most_common_titel):
        out_file.write('{0}: '.format(title))
        for elem in np.array(most_common_ac[1:]).T[i]:
            out_file.write("{0} ".format(elem))
        out_file.write('\n')

    avg_io_ac_size = np.inner(acc_sizes[:, 0], acc_sizes[:, 1]) / \
                     np.sum(acc_sizes[:, 1])
    out_file.write('{0} avg_io_ac_size: {1}\n'.format(io_type, avg_io_ac_size))
    out_file.write('\n')

##############################################################################


def file_cnt_summary(file_hash, counters, data):
    """
    file count summary (bottom right table in darshan pdf)

    :param file_hash:
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    max_bytes_read = data[counters["CP_MAX_BYTE_READ"]]["value"]
    max_bytes_written = data[counters["CP_MAX_BYTE_WRITTEN"]]["value"]
    size_at_open = data[counters["CP_SIZE_AT_OPEN"]]["value"]
    size_at_open[size_at_open == -1] = 0

    # POSIX
    posix_read = data[counters["CP_POSIX_READS"]]["value"]
    posix_fread = data[counters["CP_POSIX_FREADS"]]["value"]
    posix_write = data[counters["CP_POSIX_WRITES"]]["value"]
    posix_fwrite = data[counters["CP_POSIX_FWRITES"]]["value"]

    # MPI-IO
    indep_opens = data[counters["CP_INDEP_OPENS"]]["value"]
    coll_opens = data[counters["CP_COLL_OPENS"]]["value"]
    indep_reads = data[counters["CP_INDEP_READS"]]["value"]
    coll_reads = data[counters["CP_COLL_READS"]]["value"]
    split_reads = data[counters["CP_SPLIT_READS"]]["value"]
    nb_reads = data[counters["CP_NB_READS"]]["value"]
    indep_writes = data[counters["CP_INDEP_WRITES"]]["value"]
    coll_writes = data[counters["CP_COLL_WRITES"]]["value"]
    split_writes = data[counters["CP_SPLIT_WRITES"]]["value"]
    nb_writes = data[counters["CP_NB_WRITES"]]["value"]

    # get the minimum CP_SIZE_AT_OPEN (min_open_size) and the maximum of
    # CP_MAX_BYTE_READ/WRITTEN (max_size) for each hash(file) and
    # decide if it was_read and/or was_written
    hash_files = dict()
    for i, f_hash in enumerate(file_hash):
        max_tmp = np.max([max_bytes_read[i]+1, max_bytes_written[i]+1])
        if f_hash not in hash_files.keys():
            hash_files[f_hash] = {"min_open_size": size_at_open[i],
                                  "max_size": max_tmp,
                                  "was_read": False,
                                  "was_written": False}
        else:
            if hash_files[f_hash]["min_open_size"] > size_at_open[i]:
                hash_files[f_hash]["min_open_size"] = size_at_open[i]
            if hash_files[f_hash]["max_size"] < max_tmp:
                hash_files[f_hash]["max_size"] = max_tmp

        if (indep_opens[i] > 0) or (coll_opens[i] > 0):
            # mpi file
            if (indep_reads[i] > 0) or (coll_reads[i] > 0) or \
                    (split_reads[i] > 0) or (nb_reads[i] > 0):
                hash_files[f_hash]['was_read'] = True
            if (indep_writes[i] > 0) or (coll_writes[i] > 0) or \
                    (split_writes[i] > 0) or (nb_writes[i] > 0):
                hash_files[f_hash]['was_written'] = True
        else:
            # posix file
            if (posix_read[i] > 0) or (posix_fread[i] > 0):
                hash_files[f_hash]['was_read'] = True
            if (posix_write[i] > 0) or (posix_fwrite[i] > 0):
                hash_files[f_hash]['was_written'] = True

    max_size = np.array([hash_files[key]['max_size']
                         for key in hash_files.keys()])
    min_open_size = np.array([hash_files[key]['min_open_size']
                              for key in hash_files.keys()])
    was_read = np.array([hash_files[key]['was_read']
                         for key in hash_files.keys()])
    was_written = np.array([hash_files[key]['was_written']
                            for key in hash_files.keys()])

    # total opened
    total = np.array([True]*len(hash_files))
    # read only files
    read_only = (was_read == True) & (was_written == False)
    # write only files
    write_only = (was_read == False) & (was_written == True)
    # read and write files
    read_write = (was_read == True) & (was_written == True)
    # created files
    create = (was_written == True) & (min_open_size == 0) & (max_size > 0)

    table_rows = [['total opened', total], ['read-only files', read_only],
                  ['write-only files', write_only],
                  ['read/write files', read_write], ['created files', create]]

    f_cnt_sum_title = [['', 'number of files', 'avg. size', 'max size']]
    table_str = f_cnt_sum_title
    for row in table_rows:
        #mb = True if row[0] != 'write-only files' else False
        cnt, avg, max_val = get_cnt_avg_max(row[1], max_size,
                                            min_open_size, False)
        table_str.append([row[0], str(int(np.ceil(cnt))),
                          str(int(np.ceil(avg))), str(int(np.ceil(max_val)))])
    table_head = 'File Count I/O Summary'
    write_jube_table(table_head, table_str)


def get_cnt_avg_max(mask, max_size, min_open_size, mb=True):
    """
    calculates count, average and max value depending on the mask

    :param mask: boolean array
    :param max_size:
    :param min_open_size:
    :param mb: convert avg and max value into MB
    :return:
    """

    mb = 2**20 if mb else 1
    count = len(max_size[mask])
    if count == 0:
        avg = 0
        max_value = 0
    else:
        max_elem = np.maximum(max_size[mask], min_open_size[mask])
        avg = np.sum(max_elem)/count/mb
        max_value = np.max([max_size[mask], min_open_size[mask]])/mb

    return count, avg, max_value
###############################################################################
#darshan3: counter namen geaendert, test auf mpiio, file_hash selbst bestimmt
def file_cnt_summary3(counters, data):
    """
    file count summary (bottom right table in darshan pdf)

    :param file_hash:
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    file_hash = data['file'][data['counter'] == 'POSIX_BYTES_READ']

    max_bytes_read = data[counters["POSIX_MAX_BYTE_READ"]]["value"]
    max_bytes_written = data[counters["POSIX_MAX_BYTE_WRITTEN"]]["value"]
    size_at_open = [0] * len(max_bytes_read)

    # POSIX
    posix_read = data[counters["POSIX_READS"]]["value"]
    posix_fread = data[counters["POSIX_FREADS"]]["value"]
    posix_write = data[counters["POSIX_WRITES"]]["value"]
    posix_fwrite = data[counters["POSIX_FWRITES"]]["value"]

    # MPI-IO
    if "MPIIO_INDEP_OPENS" in counters:
        indep_opens = data[counters["MPIIO_INDEP_OPENS"]]["value"]
        coll_opens = data[counters["MPIIO_COLL_OPENS"]]["value"]
        indep_reads = data[counters["MPIIO_INDEP_READS"]]["value"]
        coll_reads = data[counters["MPIIO_COLL_READS"]]["value"]
        split_reads = data[counters["MPIIO_SPLIT_READS"]]["value"]
        nb_reads = data[counters["MPIIO_NB_READS"]]["value"]
        indep_writes = data[counters["MPIIO_INDEP_WRITES"]]["value"]
        coll_writes = data[counters["MPIIO_COLL_WRITES"]]["value"]
        split_writes = data[counters["MPIIO_SPLIT_WRITES"]]["value"]
        nb_writes = data[counters["MPIIO_NB_WRITES"]]["value"]

    # get the minimum CP_SIZE_AT_OPEN (min_open_size) and the maximum of
    # CP_MAX_BYTE_READ/WRITTEN (max_size) for each hash(file) and
    # decide if it was_read and/or was_written
    hash_files = dict()
    for i, f_hash in enumerate(file_hash):
        max_tmp = np.max([max_bytes_read[i]+1, max_bytes_written[i]+1])
        if f_hash not in hash_files.keys():
            hash_files[f_hash] = {"min_open_size": size_at_open[i],
                                  "max_size": max_tmp,
                                  "was_read": False,
                                  "was_written": False}
        else:
            if hash_files[f_hash]["min_open_size"] > size_at_open[i]:
                hash_files[f_hash]["min_open_size"] = size_at_open[i]
            if hash_files[f_hash]["max_size"] < max_tmp:
                hash_files[f_hash]["max_size"] = max_tmp

        if "MPIIO_INDEP_OPENS" in counters and \
                ((indep_opens[i] > 0) or (coll_opens[i] > 0)):
            # mpi file
            if (indep_reads[i] > 0) or (coll_reads[i] > 0) or \
                    (split_reads[i] > 0) or (nb_reads[i] > 0):
                hash_files[f_hash]['was_read'] = True
            if (indep_writes[i] > 0) or (coll_writes[i] > 0) or \
                    (split_writes[i] > 0) or (nb_writes[i] > 0):
                hash_files[f_hash]['was_written'] = True
        else:
            # posix file
            if (posix_read[i] > 0) or (posix_fread[i] > 0):
                hash_files[f_hash]['was_read'] = True
            if (posix_write[i] > 0) or (posix_fwrite[i] > 0):
                hash_files[f_hash]['was_written'] = True

    max_size = np.array([hash_files[key]['max_size']
                         for key in hash_files.keys()])
    min_open_size = np.array([hash_files[key]['min_open_size']
                              for key in hash_files.keys()])
    was_read = np.array([hash_files[key]['was_read']
                         for key in hash_files.keys()])
    was_written = np.array([hash_files[key]['was_written']
                            for key in hash_files.keys()])

    # total opened
    total = np.array([True]*len(hash_files))
    # read only files
    read_only = (was_read == True) & (was_written == False)
    # write only files
    write_only = (was_read == False) & (was_written == True)
    # read and write files
    read_write = (was_read == True) & (was_written == True)
    # created files
    create = (was_written == True) & (min_open_size == 0) & (max_size > 0)

    table_rows = [['total opened', total], ['read-only files', read_only],
                  ['write-only files', write_only],
                  ['read/write files', read_write], ['created files', create]]

    f_cnt_sum_title = [['', 'number of files', 'avg. size', 'max size']]
    table_str = f_cnt_sum_title
    for row in table_rows:
        #mb = True if row[0] != 'write-only files' else False
        cnt, avg, max_val = get_cnt_avg_max(row[1], max_size,
                                            min_open_size, False)
        table_str.append([row[0], str(int(np.ceil(cnt))),
                          str(int(np.ceil(avg))), str(int(np.ceil(max_val)))])
    table_head = 'File Count I/O Summary'
    write_jube_table(table_head, table_str)

    ###############################################################################
#darshan3.1: remove freads/fwrites
def file_cnt_summary31(counters, data):
    """
    file count summary (bottom right table in darshan pdf)

    :param file_hash:
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :return:
    """
    file_hash = data['file'][(data['counter'] == "POSIX_BYTES_READ") | (data['counter'] == "STDIO_BYTES_READ")]
    max_bytes_read = data['value'][(data['counter'] == "POSIX_MAX_BYTE_READ") | (data['counter'] == "STDIO_MAX_BYTE_READ")]
    max_bytes_written = data['value'][(data['counter'] == "POSIX_MAX_BYTE_WRITTEN") | (data['counter'] == "STDIO_MAX_BYTE_WRITTEN")]
    size_at_open = [0] * len(max_bytes_read)

    # POSIX
    posix_stdio_read = data['value'][(data['counter'] == "POSIX_READS") | (data['counter'] == "STDIO_READS")]
    posix_stdio_write = data['value'][(data['counter'] == "POSIX_WRITES") | (data['counter'] == "STDIO_WRITES")]

    # get the minimum CP_SIZE_AT_OPEN (min_open_size) and the maximum of
    # CP_MAX_BYTE_READ/WRITTEN (max_size) for each hash(file) and
    # decide if it was_read and/or was_written
    hash_files = dict()
    for i, f_hash in enumerate(file_hash):
        max_tmp = np.max([max_bytes_read[i]+1, max_bytes_written[i]+1])
        if f_hash not in hash_files.keys():
            hash_files[f_hash] = {"min_open_size": size_at_open[i],
                                  "max_size": max_tmp,
                                  "was_read": False,
                                  "was_written": False}
        else:
            if hash_files[f_hash]["min_open_size"] > size_at_open[i]:
                hash_files[f_hash]["min_open_size"] = size_at_open[i]
            if hash_files[f_hash]["max_size"] < max_tmp:
                hash_files[f_hash]["max_size"] = max_tmp
                
        # MPI-IO
        if (data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_INDEP_OPENS")] > 0) or \
                (data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_COL_OPENS")] > 0):
            # mpi file
            indep_read = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_INDEP_READS")] 
            coll_read = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_COLL_READS")] 
            split_read = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_SPLIT_READS")] 
            nb_read = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_NB_READS")] 
            indep_write = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_INDEP_WRITES")] 
            coll_write = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_COLL_WRITES")] 
            split_write = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_SPLIT_WRITES")] 
            nb_write = data['value'][(data['file'] == f_hash) & (data['counter'] == "MPIIO_NB_WRITES")] 

            if (indep_read > 0) or (coll_read > 0) or \
                    (split_read > 0) or (nb_read > 0):
                hash_files[f_hash]['was_read'] = True
            if (indep_write > 0) or (coll_write > 0) or \
                    (split_write > 0) or (nb_write > 0):
                hash_files[f_hash]['was_written'] = True
        else:
            # posix/stdio file
            if (posix_stdio_read[i] > 0):
                hash_files[f_hash]['was_read'] = True
            if (posix_stdio_write[i] > 0):
                hash_files[f_hash]['was_written'] = True

    max_size = np.array([hash_files[key]['max_size']
                         for key in hash_files.keys()])
    min_open_size = np.array([hash_files[key]['min_open_size']
                              for key in hash_files.keys()])
    was_read = np.array([hash_files[key]['was_read']
                         for key in hash_files.keys()])
    was_written = np.array([hash_files[key]['was_written']
                            for key in hash_files.keys()])

    # total opened
    total = np.array([True]*len(hash_files))
    # read only files
    read_only = (was_read == True) & (was_written == False)
    # write only files
    write_only = (was_read == False) & (was_written == True)
    # read and write files
    read_write = (was_read == True) & (was_written == True)
    # created files
    create = (was_written == True) & (min_open_size == 0) & (max_size > 0)

    table_rows = [['total opened', total], ['read-only files', read_only],
                  ['write-only files', write_only],
                  ['read/write files', read_write], ['created files', create]]

    f_cnt_sum_title = [['', 'number of files', 'avg. size', 'max size']]
    table_str = f_cnt_sum_title
    for row in table_rows:
        #mb = True if row[0] != 'write-only files' else False
        cnt, avg, max_val = get_cnt_avg_max(row[1], max_size,
                                            min_open_size, False)
        table_str.append([row[0], str(int(np.ceil(cnt))),
                          str(int(np.ceil(avg))), str(int(np.ceil(max_val)))])
    table_head = 'File Count I/O Summary'
    write_jube_table(table_head, table_str)

###############################################################################


def avg_io(nprocs, ranks, counters, data, out_file):
    """
    average cumulative time spent in i/o functions and average amount of i/o
    for independent/shared reads/writes/metadata.
    (table on darshan pdf second page)

    :param nprocs: number of processes
    :param ranks: rank that opened the file/counter
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """
    mb = 2**20

    posix_read_time = data[counters["CP_F_POSIX_READ_TIME"]]["value"]
    posix_write_time = data[counters["CP_F_POSIX_WRITE_TIME"]]["value"]
    posix_meta_time = data[counters["CP_F_POSIX_META_TIME"]]["value"]
    bytes_read = data[counters["CP_BYTES_READ"]]["value"]
    bytes_written = data[counters["CP_BYTES_WRITTEN"]]["value"]

    indep_reads_avg = np.sum(posix_read_time[ranks != -1]/nprocs)
    indep_read_bytes_avg = np.sum(bytes_read[ranks != -1]/nprocs)/mb
    indep_writes_avg = np.sum(posix_write_time[ranks != -1]/nprocs)
    indep_written_bytes_avg = np.sum(bytes_written[ranks != -1]/nprocs)/mb
    indep_meta_avg = np.sum(posix_meta_time[ranks != -1]/nprocs)

    shared_reads_avg = np.sum(posix_read_time[ranks == -1]/nprocs)
    shared_read_bytes_avg = np.sum(bytes_read[ranks == -1]/nprocs)/mb
    shared_writes_avg = np.sum(posix_write_time[ranks == -1]/nprocs)
    shared_written_bytes_avg = np.sum(bytes_written[ranks == -1]/nprocs)/mb
    shared_meta_avg = np.sum(posix_meta_time[ranks == -1]/nprocs)

    avg_io_table = [[indep_reads_avg, indep_read_bytes_avg],
                    [indep_writes_avg, indep_written_bytes_avg],
                    [indep_meta_avg, np.nan],
                    [shared_reads_avg, shared_read_bytes_avg],
                    [shared_writes_avg, shared_written_bytes_avg],
                    [shared_meta_avg, np.nan]]
    avg_io_table_str = [[str(np.around(cell, decimals=6)) for cell in row]
                        for row in avg_io_table]

    title = ['cumul_avg_io_time', 'avg_amout_io']
    for i, row in enumerate(np.array(avg_io_table_str).T):
        out_file.write('{0}: '.format(title[i]))
        for elem in row:
            out_file.write('{0} '.format(elem))
        out_file.write('\n')

    avg_table_titel = ['Cumulative time spent in I/O functions (seconds)',
                       'Amount of I/O (MB)']
    avg_io_table_str.insert(0, avg_table_titel)

    #a = ['', 'Independent reads', 'Independent writes', 'Independent metadada',
    #     'Shared reads', 'Shared writes', 'Shared metadata']
    #avg_io_table_str = [np.insert(row, 0, a[i])
    #                    for i, row in enumerate(avg_io_table_str)]

    table_head = "Average I/O per process"
    write_jube_table(table_head, avg_io_table_str)

    # calculate independent/shared read/write bandwidth
    if indep_reads_avg != 0:
        in_r_bw = np.around(indep_read_bytes_avg/indep_reads_avg*nprocs, 2)
    else:
        in_r_bw = 0.
    if indep_writes_avg != 0:
        in_w_bw = np.around(indep_written_bytes_avg/indep_writes_avg*nprocs, 2)
    else:
        in_w_bw = 0.

    if shared_reads_avg != 0:
        sh_r_bw = np.around(shared_read_bytes_avg/shared_reads_avg*nprocs, 2)
    else:
        sh_r_bw = 0.
    if shared_writes_avg != 0:
        sh_w_bw = np.around(shared_written_bytes_avg/shared_writes_avg*nprocs, 2)
    else:
        sh_w_bw = 0.

    # write information into the output file
    out_file.write('# Bandwidth: read write\n')
    out_file.write('Independent Bandwidth: {0} {1}\n'.format(in_r_bw, in_w_bw))
    out_file.write('Shared Bandwidth: {0} {1}\n'.format(sh_r_bw, sh_w_bw))

    total_mb_read = np.sum(bytes_read)/mb
    total_mb_written = np.sum(bytes_written)/mb
    out_file.write('# I/O Volume:\n')
    out_file.write('Total MB read: {0}\n'.format(total_mb_read))
    out_file.write('Total MB written: {0}\n'.format(total_mb_written))
    out_file.write('\n')
#############################################################################
#darshan3: counter geaendert und ranks selbst ausgelesen
def avg_io3(nprocs, counters, data, out_file):
    """
    average cumulative time spent in i/o functions and average amount of i/o
    for independent/shared reads/writes/metadata.
    (table on darshan pdf second page)

    :param nprocs: number of processes
    :param ranks: rank that opened the file/counter
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """
    mb = 2**20

    ranks = data['rank'][data['counter'] == 'POSIX_F_READ_TIME']

    posix_read_time = data[counters["POSIX_F_READ_TIME"]]["value"]
    posix_write_time = data[counters["POSIX_F_WRITE_TIME"]]["value"]
    posix_meta_time = data[counters["POSIX_F_META_TIME"]]["value"]
    bytes_read = data[counters["POSIX_BYTES_READ"]]["value"]
    bytes_written = data[counters["POSIX_BYTES_WRITTEN"]]["value"]

    indep_reads_avg = np.sum(posix_read_time[ranks != -1]/nprocs)
    indep_read_bytes_avg = np.sum(bytes_read[ranks != -1]/nprocs)/mb
    indep_writes_avg = np.sum(posix_write_time[ranks != -1]/nprocs)
    indep_written_bytes_avg = np.sum(bytes_written[ranks != -1]/nprocs)/mb
    indep_meta_avg = np.sum(posix_meta_time[ranks != -1]/nprocs)

    shared_reads_avg = np.sum(posix_read_time[ranks == -1]/nprocs)
    shared_read_bytes_avg = np.sum(bytes_read[ranks == -1]/nprocs)/mb
    shared_writes_avg = np.sum(posix_write_time[ranks == -1]/nprocs)
    shared_written_bytes_avg = np.sum(bytes_written[ranks == -1]/nprocs)/mb
    shared_meta_avg = np.sum(posix_meta_time[ranks == -1]/nprocs)

    avg_io_table = [[indep_reads_avg, indep_read_bytes_avg],
                    [indep_writes_avg, indep_written_bytes_avg],
                    [indep_meta_avg, np.nan],
                    [shared_reads_avg, shared_read_bytes_avg],
                    [shared_writes_avg, shared_written_bytes_avg],
                    [shared_meta_avg, np.nan]]
    avg_io_table_str = [[str(np.around(cell, decimals=6)) for cell in row]
                        for row in avg_io_table]

    title = ['cumul_avg_io_time', 'avg_amout_io']
    for i, row in enumerate(np.array(avg_io_table_str).T):
        out_file.write('{0}: '.format(title[i]))
        for elem in row:
            out_file.write('{0} '.format(elem))
        out_file.write('\n')

    avg_table_titel = ['Cumulative time spent in I/O functions (seconds)',
                       'Amount of I/O (MB)']
    avg_io_table_str.insert(0, avg_table_titel)

    #a = ['', 'Independent reads', 'Independent writes', 'Independent metadada',
    #     'Shared reads', 'Shared writes', 'Shared metadata']
    #avg_io_table_str = [np.insert(row, 0, a[i])
    #                    for i, row in enumerate(avg_io_table_str)]

    table_head = "Average I/O per process (POSIX)"
    write_jube_table(table_head, avg_io_table_str)

    # calculate independent/shared read/write bandwidth
    if indep_reads_avg != 0:
        in_r_bw = np.around(indep_read_bytes_avg/indep_reads_avg*nprocs, 2)
    else:
        in_r_bw = 0.
    if indep_writes_avg != 0:
        in_w_bw = np.around(indep_written_bytes_avg/indep_writes_avg*nprocs, 2)
    else:
        in_w_bw = 0.

    if shared_reads_avg != 0:
        sh_r_bw = np.around(shared_read_bytes_avg/shared_reads_avg*nprocs, 2)
    else:
        sh_r_bw = 0.
    if shared_writes_avg != 0:
        sh_w_bw = np.around(shared_written_bytes_avg/shared_writes_avg*nprocs, 2)
    else:
        sh_w_bw = 0.

    # write information into the output file
    out_file.write('# Bandwidth: read write\n')
    out_file.write('Independent Bandwidth: {0} {1}\n'.format(in_r_bw, in_w_bw))
    out_file.write('Shared Bandwidth: {0} {1}\n'.format(sh_r_bw, sh_w_bw))

    total_mb_read = np.sum(bytes_read)/mb
    total_mb_written = np.sum(bytes_written)/mb
    out_file.write('# I/O Volume:\n')
    out_file.write('Total MB read: {0}\n'.format(total_mb_read))
    out_file.write('Total MB written: {0}\n'.format(total_mb_written))
    out_file.write('\n')

#############################################################################
#darshan3.1: 
def avg_io31(nprocs, counters, data, out_file):
    """
    average cumulative time spent in i/o functions and average amount of i/o
    for independent/shared reads/writes/metadata.
    (table on darshan pdf second page)

    :param nprocs: number of processes
    :param ranks: rank that opened the file/counter
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param out_file: file handle for output file
    :return:
    """
    mb = 2**20

    ranks = data['rank'][(data['counter'] == 'POSIX_F_READ_TIME') | (data['counter'] == 'STDIO_F_READ_TIME')]

    read_time = data['value'][(data['counter'] == 'POSIX_F_READ_TIME') | (data['counter'] == 'STDIO_F_READ_TIME')]
    write_time = data['value'][(data['counter'] == 'POSIX_F_WRITE_TIME') | (data['counter'] == 'STDIO_F_WRITE_TIME')]
    meta_time = data['value'][(data['counter'] == 'POSIX_F_META_TIME') | (data['counter'] == 'STDIO_F_META_TIME')]
    bytes_read = data["value"][(data['counter'] == "POSIX_BYTES_READ") | (data['counter'] == "STDIO_BYTES_READ")]
    bytes_written = data["value"][(data['counter'] == "POSIX_BYTES_WRITTEN") | (data['counter'] == "STDIO_BYTES_WRITTEN")]

    indep_reads_avg = np.sum(read_time[ranks != -1]/nprocs)
    indep_read_bytes_avg = np.sum(bytes_read[ranks != -1]/nprocs)/mb
    indep_writes_avg = np.sum(write_time[ranks != -1]/nprocs)
    indep_written_bytes_avg = np.sum(bytes_written[ranks != -1]/nprocs)/mb
    indep_meta_avg = np.sum(meta_time[ranks != -1]/nprocs)

    shared_reads_avg = np.sum(read_time[ranks == -1]/nprocs)
    shared_read_bytes_avg = np.sum(bytes_read[ranks == -1]/nprocs)/mb
    shared_writes_avg = np.sum(write_time[ranks == -1]/nprocs)
    shared_written_bytes_avg = np.sum(bytes_written[ranks == -1]/nprocs)/mb
    shared_meta_avg = np.sum(meta_time[ranks == -1]/nprocs)

    avg_io_table = [[indep_reads_avg, indep_read_bytes_avg],
                    [indep_writes_avg, indep_written_bytes_avg],
                    [indep_meta_avg, np.nan],
                    [shared_reads_avg, shared_read_bytes_avg],
                    [shared_writes_avg, shared_written_bytes_avg],
                    [shared_meta_avg, np.nan]]
    avg_io_table_str = [[str(np.around(cell, decimals=6)) for cell in row]
                        for row in avg_io_table]

    title = ['cumul_avg_io_time', 'avg_amout_io']
    for i, row in enumerate(np.array(avg_io_table_str).T):
        out_file.write('{0}: '.format(title[i]))
        for elem in row:
            out_file.write('{0} '.format(elem))
        out_file.write('\n')

    avg_table_titel = ['Cumulative time spent in I/O functions (seconds)',
                       'Amount of I/O (MB)']
    avg_io_table_str.insert(0, avg_table_titel)

    #a = ['', 'Independent reads', 'Independent writes', 'Independent metadada',
    #     'Shared reads', 'Shared writes', 'Shared metadata']
    #avg_io_table_str = [np.insert(row, 0, a[i])
    #                    for i, row in enumerate(avg_io_table_str)]

    table_head = "Average I/O per process (POSIX)"
    write_jube_table(table_head, avg_io_table_str)

    # calculate independent/shared read/write bandwidth
    if indep_reads_avg != 0:
        in_r_bw = np.around(indep_read_bytes_avg/indep_reads_avg*nprocs, 2)
    else:
        in_r_bw = 0.
    if indep_writes_avg != 0:
        in_w_bw = np.around(indep_written_bytes_avg/indep_writes_avg*nprocs, 2)
    else:
        in_w_bw = 0.

    if shared_reads_avg != 0:
        sh_r_bw = np.around(shared_read_bytes_avg/shared_reads_avg*nprocs, 2)
    else:
        sh_r_bw = 0.
    if shared_writes_avg != 0:
        sh_w_bw = np.around(shared_written_bytes_avg/shared_writes_avg*nprocs, 2)
    else:
        sh_w_bw = 0.

    # write information into the output file
    out_file.write('# Bandwidth: read write\n')
    out_file.write('Independent Bandwidth: {0} {1}\n'.format(in_r_bw, in_w_bw))
    out_file.write('Shared Bandwidth: {0} {1}\n'.format(sh_r_bw, sh_w_bw))

    total_mb_read = np.sum(bytes_read)/mb
    total_mb_written = np.sum(bytes_written)/mb
    out_file.write('# I/O Volume:\n')
    out_file.write('Total MB read: {0}\n'.format(total_mb_read))
    out_file.write('Total MB written: {0}\n'.format(total_mb_written))
    out_file.write('\n')


###########################################################################

def data_transfer(counters, data, mount_pt):
    """

    :param counters:
    :param data:
    :param mount_pt:
    :return:
    """

    bytes_read = data[counters["CP_BYTES_READ"]]["value"]
    bytes_written = data[counters["CP_BYTES_WRITTEN"]]["value"]
    filesystems = set(mount_pt)

    byte_table = [[str(key), str(int(np.sum(bytes_written[mount_pt == key]))),
                   str(int(np.sum(bytes_read[mount_pt == key])))]
                  for key in filesystems]

    table_head = "Data Transfer Per Filesystem"
    column_heads = ['File System', 'write', 'read']

    # test output converted to MB
    #mb_out = [[i[0], str(np.around(float(i[1])/2**20, 5)),
    #           str(np.around(float(i[2])/2**20, 5))] for i in byte_table]
    #mb_out.insert(0, column_heads)
    #print(mb_out)
    #write_jube_table(table_head, mb_out)

    byte_table.insert(0, column_heads)
    #print(byte_table)
    write_jube_table(table_head, byte_table)

    # read/write/metadata time spent in each file system
    mpi_read_time = data[counters["CP_F_MPI_READ_TIME"]]["value"]
    mpi_write_time = data[counters["CP_F_MPI_WRITE_TIME"]]["value"]
    mpi_meta_time = data[counters["CP_F_MPI_META_TIME"]]["value"]
    posix_read_time = data[counters["CP_F_POSIX_READ_TIME"]]["value"]
    posix_write_time = data[counters["CP_F_POSIX_WRITE_TIME"]]["value"]
    posix_meta_time = data[counters["CP_F_POSIX_META_TIME"]]["value"]

    table_head = "Time For Data Transfer Per Filesystem"
    column_heads = ['File System', 'mpi_read', 'mpi_write', 'mpi_meta',
                    'posix_read', 'posix_write', 'posix_meta']
    time_table = [[str(key), str(np.sum(mpi_read_time[mount_pt == key])),
                   str(np.sum(mpi_write_time[mount_pt == key])),
                   str(np.sum(mpi_meta_time[mount_pt == key])),
                   str(np.sum(posix_read_time[mount_pt == key])),
                   str(np.sum(posix_write_time[mount_pt == key])),
                   str(np.sum(posix_meta_time[mount_pt == key]))]
                  for key in filesystems]
    time_table.insert(0, column_heads)
    write_jube_table(table_head, time_table)


##########################################################################
#darshan3: counter namen geaendert und mount selbst bestimmt, optionales mpiio
def data_transfer3(counters, data):
    """

    :param counters:
    :param data:
    :param mount_pt:
    :return:
    """
    mount_pt = data['mount_pt'][data['counter'] == 'POSIX_BYTES_READ']

    bytes_read = data[counters["POSIX_BYTES_READ"]]["value"]
    bytes_written = data[counters["POSIX_BYTES_WRITTEN"]]["value"]
    filesystems = set(mount_pt)

    byte_table = [[str(key), str(int(np.sum(bytes_written[mount_pt == key]))),
                   str(int(np.sum(bytes_read[mount_pt == key])))]
                  for key in filesystems]

    table_head = "Data Transfer Per Filesystem"
    column_heads = ['File System', 'write', 'read']

    byte_table.insert(0, column_heads)
    #print(byte_table)
    write_jube_table(table_head, byte_table)

    # read/write/metadata time spent in each file system
    posix_read_time = data[counters["POSIX_F_READ_TIME"]]["value"]
    posix_write_time = data[counters["POSIX_F_WRITE_TIME"]]["value"]
    posix_meta_time = data[counters["POSIX_F_META_TIME"]]["value"]
    if "MPIIO_INDEP_OPENS" in counters:
        mpi_read_time = data[counters["MPIIO_F_READ_TIME"]]["value"]
        mpi_write_time = data[counters["MPIIO_F_WRITE_TIME"]]["value"]
        mpi_meta_time = data[counters["MPIIO_F_META_TIME"]]["value"]
        time_table = [[str(key), str(np.sum(mpi_read_time[mount_pt == key])),
                       str(np.sum(mpi_write_time[mount_pt == key])),
                       str(np.sum(mpi_meta_time[mount_pt == key])),
                       str(np.sum(posix_read_time[mount_pt == key])),
                       str(np.sum(posix_write_time[mount_pt == key])),
                       str(np.sum(posix_meta_time[mount_pt == key]))]
                      for key in filesystems]
    else:
        time_table = [[str(key), str(0), str(0), str(0),
                       str(np.sum(posix_read_time[mount_pt == key])),
                       str(np.sum(posix_write_time[mount_pt == key])),
                       str(np.sum(posix_meta_time[mount_pt == key]))]
                      for key in filesystems]

    table_head = "Time For Data Transfer Per Filesystem"
    column_heads = ['File System', 'mpi_read', 'mpi_write', 'mpi_meta',
                    'posix_read', 'posix_write', 'posix_meta']

    time_table.insert(0, column_heads)
    write_jube_table(table_head, time_table)

#############################################################################
#darshan3.1: test for mpi/io output logs
def data_transfer31(counters, data):
    """

    :param counters:
    :param data:
    :param mount_pt:
    :return:
    """
    bytes_read = data["value"][(data['counter'] == "POSIX_BYTES_READ") | (data['counter'] == "STDIO_BYTES_READ")]
    bytes_written = data["value"][(data['counter'] == "POSIX_BYTES_WRITTEN") | (data['counter'] == "STDIO_BYTES_WRITTEN")]
    mount_pt = data['mount_pt'][(data['counter'] == 'POSIX_BYTES_READ') | (data['counter'] == 'STDIO_BYTES_READ')]
    filesystems = set(data['mount_pt'])
    mpiio_mount_pt = data['mount_pt'][(data['counter'] == 'MPIIO_INDEP_OPENS')]


    byte_table = [[str(key), str(int(np.sum(bytes_written[mount_pt == key]))),
                   str(int(np.sum(bytes_read[mount_pt == key])))]
                  for key in filesystems]


    table_head = "Data Transfer Per Filesystem"
    column_heads = ['File System', 'write', 'read']

    byte_table.insert(0, column_heads)
    #print(byte_table)
    write_jube_table(table_head, byte_table)

    # read/write/metadata time spent in each file system
    read_time = data['value'][(data['counter'] == 'POSIX_F_READ_TIME') | (data['counter'] == 'STDIO_F_READ_TIME')]
    write_time = data['value'][(data['counter'] == 'POSIX_F_WRITE_TIME') | (data['counter'] == 'STDIO_F_WRITE_TIME')]
    meta_time = data['value'][(data['counter'] == 'POSIX_F_META_TIME') | (data['counter'] == 'STDIO_F_META_TIME')]
    if "MPIIO_INDEP_OPENS" in counters:
        mpi_read_time = data[counters["MPIIO_F_READ_TIME"]]["value"]
        mpi_write_time = data[counters["MPIIO_F_WRITE_TIME"]]["value"]
        mpi_meta_time = data[counters["MPIIO_F_META_TIME"]]["value"]
        time_table = [[str(key), str(np.sum(mpi_read_time[mpiio_mount_pt == key])),
                       str(np.sum(mpi_write_time[mpiio_mount_pt == key])),
                       str(np.sum(mpi_meta_time[mpiio_mount_pt == key])),
                       str(np.sum(read_time[mount_pt == key])),
                       str(np.sum(write_time[mount_pt == key])),
                       str(np.sum(meta_time[mount_pt == key]))]
                      for key in filesystems]
    else:
        time_table = [[str(key), str(0), str(0), str(0),
                       str(np.sum(read_time[mount_pt == key])),
                       str(np.sum(write_time[mount_pt == key])),
                       str(np.sum(meta_time[mount_pt == key]))]
                      for key in filesystems]

    table_head = "Time For Data Transfer Per Filesystem"
    column_heads = ['File System', 'mpi_read', 'mpi_write', 'mpi_meta',
                    'posix_read', 'posix_write', 'posix_meta']

    time_table.insert(0, column_heads)
    write_jube_table(table_head, time_table)

#############################################################################

def create_ind_sha_table(nprocs, ranks, file_hash, counters, data, mount_pt):
    """
    creates a table containing every independent file and a table containing
    every shared file, because the darshan-job-summary.pl shows statistics of
    only 20 files.

    :param nprocs: number of processes
    :param ranks: rank corresponding to files in data
    :param file_hash: has for every file in data
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param mount_pt:
    :return:
    """

    name = data[counters["CP_F_OPEN_TIMESTAMP"]]["name_suffix"]
    read_start_time = data[counters["CP_F_READ_START_TIMESTAMP"]]["value"]
    read_end_time = data[counters["CP_F_READ_END_TIMESTAMP"]]["value"]
    posix_read_time = data[counters["CP_F_POSIX_READ_TIME"]]["value"]
    write_start_time = data[counters["CP_F_WRITE_START_TIMESTAMP"]]["value"]
    write_end_time = data[counters["CP_F_WRITE_END_TIMESTAMP"]]["value"]
    posix_write_time = data[counters["CP_F_POSIX_WRITE_TIME"]]["value"]

    file_count = [list(file_hash).count(i) for i in file_hash]

    file_count = [nprocs if r == -1 else file_count[i]
                  for i, r in enumerate(ranks)]

    collected_dtype = [("rank", np.int), ("name_suffix", object),
                       ("file_cnt", np.int), ("rstart", np.float64),
                       ("rend", np.float64), ("maxr", np.float64),
                       ("wstart", np.float64), ("wend", np.float64),
                       ("maxw", np.float64), ("mount_pt", object)]
    collected_data = list(zip(ranks, name, file_count, read_start_time,
                              read_end_time, posix_read_time, write_start_time,
                              write_end_time, posix_write_time, mount_pt))
    collected_data = np.array(collected_data, dtype=collected_dtype)

    table_headline = [['rank', 'file_suffix', 'np', 'read_start', 'read_end',
                      'cumul_read', 'write_start', 'write_end', 'cumul_write',
                       'mount_pt']]

    indep_data = collected_data[collected_data["rank"] != -1]
    indep_str = [[str(cell) for cell in row] for row in indep_data]
    indep_str = table_headline + indep_str

    table_head = "independent files"
    write_jube_table(table_head, indep_str)

    shared_data = collected_data[collected_data["rank"] == -1]
    shared_str = [[str(cell) for cell in row] for row in shared_data]
    shared_str = table_headline + shared_str

    table_head = "shared files"
    write_jube_table(table_head, shared_str)

######################################################################
#darshan3:counter namen geaendert, mount_pt, ranks, file_hash selbst bestimmt
def create_ind_sha_table3(nprocs, counters, data):
    """
    creates a table containing every independent file and a table containing
    every shared file, because the darshan-job-summary.pl shows statistics of
    only 20 files.

    :param nprocs: number of processes
    :param ranks: rank corresponding to files in data
    :param file_hash: has for every file in data
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param mount_pt:
    :return:
    """
    mount_pt = data['mount_pt'][data['counter'] == 'POSIX_BYTES_READ']
    ranks = data['rank'][data['counter'] == 'POSIX_BYTES_READ']
    file_hash = data['file'][data['counter'] == 'POSIX_BYTES_READ']

    name = data[counters["POSIX_F_OPEN_TIMESTAMP"]]["name_suffix"]
    read_start_time = data[counters["POSIX_F_READ_START_TIMESTAMP"]]["value"]
    read_end_time = data[counters["POSIX_F_READ_END_TIMESTAMP"]]["value"]
    posix_read_time = data[counters["POSIX_F_READ_TIME"]]["value"]
    write_start_time = data[counters["POSIX_F_WRITE_START_TIMESTAMP"]]["value"]
    write_end_time = data[counters["POSIX_F_WRITE_END_TIMESTAMP"]]["value"]
    posix_write_time = data[counters["POSIX_F_WRITE_TIME"]]["value"]

    file_count = [list(file_hash).count(i) for i in file_hash]

    file_count = [nprocs if r == -1 else file_count[i]
                  for i, r in enumerate(ranks)]

    collected_dtype = [("rank", np.int), ("name_suffix", object),
                       ("file_cnt", np.int), ("rstart", np.float64),
                       ("rend", np.float64), ("maxr", np.float64),
                       ("wstart", np.float64), ("wend", np.float64),
                       ("maxw", np.float64), ("mount_pt", object)]
    collected_data = list(zip(ranks, name, file_count, read_start_time,
                              read_end_time, posix_read_time, write_start_time,
                              write_end_time, posix_write_time, mount_pt))
    collected_data = np.array(collected_data, dtype=collected_dtype)

    table_headline = [['rank', 'file_suffix', 'np', 'read_start', 'read_end',
                      'cumul_read', 'write_start', 'write_end', 'cumul_write',
                       'mount_pt']]

    indep_data = collected_data[collected_data["rank"] != -1]
    indep_str = [[str(cell) for cell in row] for row in indep_data]
    indep_str = table_headline + indep_str

    table_head = "independent files"
    write_jube_table(table_head, indep_str)

    shared_data = collected_data[collected_data["rank"] == -1]
    shared_str = [[str(cell) for cell in row] for row in shared_data]
    shared_str = table_headline + shared_str

    table_head = "shared files"
    write_jube_table(table_head, shared_str)

######################################################################################
#darshan3.1: 
def create_ind_sha_table31(nprocs, counters, data):
    """
    creates a table containing every independent file and a table containing
    every shared file, because the darshan-job-summary.pl shows statistics of
    only 20 files.

    :param nprocs: number of processes
    :param ranks: rank corresponding to files in data
    :param file_hash: has for every file in data
    :param counters: mask for counters in data
    :param data: data generated by darshan-parser
    :param mount_pt:
    :return:
    """
    name = data['name_suffix'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]
    ranks = data['rank'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]
    file_hash = data['file'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]
    mount_pt = data['mount_pt'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]

    posix_stdio_read = data['value'][(data['counter'] == "POSIX_F_READ_TIME") | (data['counter'] == "STDIO_F_READ_TIME")]
    read_start_time = data['value'][(data['counter'] == "POSIX_F_READ_START_TIMESTAMP") | (data['counter'] == "STDIO_F_READ_START_TIMESTAMP")]
    read_end_time = data['value'][(data['counter'] == "POSIX_F_READ_END_TIMESTAMP") | (data['counter'] == "STDIO_F_READ_END_TIMESTAMP")]
    posix_stdio_write = data['value'][(data['counter'] == "POSIX_F_WRITE_TIME") | (data['counter'] == "STDIO_F_WRITE_TIME")]
    write_start_time = data['value'][(data['counter'] == "POSIX_F_WRITE_START_TIMESTAMP") | (data['counter'] == "STDIO_F_WRITE_START_TIMESTAMP")]
    write_end_time = data['value'][(data['counter'] == "POSIX_F_WRITE_END_TIMESTAMP") | (data['counter'] == "STDIO_F_WRITE_END_TIMESTAMP")]

    file_count = [list(file_hash).count(i) for i in file_hash]

    file_count = [nprocs if r == -1 else file_count[i]
                  for i, r in enumerate(ranks)]

    collected_dtype = [("rank", np.int), ("name_suffix", object),
                       ("file_cnt", np.int), ("rstart", np.float64),
                       ("rend", np.float64), ("maxr", np.float64),
                       ("wstart", np.float64), ("wend", np.float64),
                       ("maxw", np.float64), ("mount_pt", object)]
    collected_data = list(zip(ranks, name, file_count, read_start_time,
                              read_end_time, posix_stdio_read, write_start_time,
                              write_end_time, posix_stdio_write, mount_pt))
    collected_data = np.array(collected_data, dtype=collected_dtype)

    table_headline = [['rank', 'file_suffix', 'np', 'read_start', 'read_end',
                      'cumul_read', 'write_start', 'write_end', 'cumul_write',
                       'mount_pt']]

    indep_data = collected_data[collected_data["rank"] != -1]
    indep_str = [[str(cell) for cell in row] for row in indep_data]
    indep_str = table_headline + indep_str

    table_head = "independent files"
    write_jube_table(table_head, indep_str)

    shared_data = collected_data[collected_data["rank"] == -1]
    shared_str = [[str(cell) for cell in row] for row in shared_data]
    shared_str = table_headline + shared_str

    table_head = "shared files"
    write_jube_table(table_head, shared_str)

############################################################################

def write_jube_table(table_head, data):
    """

    :param table_head:
    :param data:
    :return:
    """
    if not NOSTDOUT and JUBE_TABLE:
        try:
            from jube2.util.output import text_table
        except ImportError:
            from jube2.util import text_table
        print(table_head)
        print(text_table(data, use_header_line=True))
        print()
    elif not NOSTDOUT:
        import sys
        print(table_head)
        np.savetxt(sys.stdout, np.array(data), fmt='%s', delimiter=' | ')
        print()


def variance_table(nprocs, ranks, file_hash, counters, data):
    """

    :param nprocs:
    :param ranks:
    :param file_hash:
    :param counters:
    :param data:
    :return:
    """

    names = data[counters["CP_F_OPEN_TIMESTAMP"]]["name_suffix"]
    slow_rank = data[counters['CP_SLOWEST_RANK']]["value"]
    slow_rank_time = data[counters['CP_F_SLOWEST_RANK_TIME']]["value"]
    slow_rank_byte = data[counters['CP_SLOWEST_RANK_BYTES']]["value"]
    fast_rank = data[counters['CP_FASTEST_RANK']]["value"]
    fast_rank_time = data[counters['CP_F_FASTEST_RANK_TIME']]["value"]
    fast_rank_byte = data[counters['CP_FASTEST_RANK_BYTES']]["value"]
    var_rank_time = data[counters['CP_F_VARIANCE_RANK_TIME']]["value"]
    var_rank_byte = data[counters['CP_F_VARIANCE_RANK_BYTES']]["value"]
    posix_meta = data[counters['CP_F_POSIX_META_TIME']]["value"]
    posix_read = data[counters['CP_F_POSIX_READ_TIME']]["value"]
    posix_write = data[counters['CP_F_POSIX_WRITE_TIME']]["value"]
    bytes_read = data[counters['CP_BYTES_READ']]["value"]
    bytes_written = data[counters['CP_BYTES_WRITTEN']]["value"]

    table_dtype = [('hash', np.uint64), ('name_suffix', object),
                   ('procs', np.int), ('fastest_rank', np.int),
                   ('fastest_time', np.float64), ('fastest_bytes', np.uint64),
                   ('slowest_rank', np.int), ('slowest_time', np.float64),
                   ('slowest_bytes', np.uint64), ('variance_time', np.float64),
                   ('variance_bytes', np.float64)]
    table = np.array(list(), dtype=table_dtype)

    var_dtype = [('hash', np.uint64), ('var_time_S', np.float64),
                 ('var_time_T', np.float64), ('var_time_n', np.int),
                 ('var_bytes_S', np.float64), ('var_bytes_T', np.float64),
                 ('var_bytes_n', np.int)]
    var_help = np.array(list(), dtype=var_dtype)

    for i, f_hash in enumerate(file_hash):
        if ranks[i] == -1:
            row = np.array([(f_hash, names[i], nprocs, fast_rank[i],
                             fast_rank_time[i], fast_rank_byte[i], slow_rank[i],
                             slow_rank_time[i], slow_rank_byte[i],
                             var_rank_time[i], var_rank_byte[i])],
                           dtype=table_dtype)
            table = np.append(table, row)
        else:
            total_time = posix_meta[i] + posix_read[i] + posix_write[i]
            total_bytes = bytes_read[i] + bytes_written[i]
            if f_hash not in table['hash']:
                row = np.array([(f_hash, names[i], 1, ranks[i], total_time,
                                 total_bytes, ranks[i], total_time, total_bytes,
                                 0, 0)], dtype=table_dtype)
                table = np.append(table, row)
                # set temporary table to calculate the variance
                row = np.array([(f_hash, 0, total_time, 1, 0, total_bytes, 1)],
                               dtype=var_dtype)
                var_help = np.append(var_help, row)
            else:
                hash_ind = var_help['hash'] == f_hash
                # calculate variance of time
                n_tmp = var_help['var_time_n'][hash_ind]
                m_tmp = 1.
                t_tmp = var_help['var_time_T'][hash_ind]
                var_help['var_time_S'][hash_ind] += \
                    (m_tmp/(n_tmp*(n_tmp+m_tmp))) * \
                    ((n_tmp/m_tmp)*total_time - t_tmp)**2
                var_help['var_time_T'][hash_ind] += total_time
                var_help['var_time_n'][hash_ind] += 1
                table['variance_time'][table['hash'] == f_hash] =  \
                    var_help['var_time_S'][hash_ind] / \
                    var_help['var_time_n'][hash_ind]

                # calculate variance of bytes
                n_tmp = var_help['var_bytes_n'][hash_ind]
                m_tmp = 1.
                t_tmp = var_help['var_bytes_T'][hash_ind]
                var_help['var_bytes_S'][hash_ind] += \
                    (m_tmp/(n_tmp*(n_tmp+m_tmp))) * \
                    ((n_tmp/m_tmp)*total_bytes - t_tmp)**2
                var_help['var_bytes_T'][hash_ind] += total_bytes
                var_help['var_bytes_n'][hash_ind] += 1
                table['variance_bytes'][table['hash'] == f_hash] =  \
                    var_help['var_bytes_S'][hash_ind] / \
                    var_help['var_bytes_n'][hash_ind]

                # increase number of processes accessing the file
                table['procs'][table['hash'] == f_hash] = \
                    var_help['var_time_n'][hash_ind]

                # update slowest/fastest time/rank/bytes
                if table['slowest_time'][table['hash'] == f_hash] < total_time:
                    table['slowest_time'][table['hash'] == f_hash] = total_time
                    table['slowest_rank'][table['hash'] == f_hash] = ranks[i]
                    table['slowest_bytes'][table['hash'] == f_hash] = total_bytes
                if table['fastest_time'][table['hash'] == f_hash] > total_time:
                    table['fastest_time'][table['hash'] == f_hash] = total_time
                    table['fastest_rank'][table['hash'] == f_hash] = ranks[i]
                    table['fastest_bytes'][table['hash'] == f_hash] = total_bytes

    table['variance_bytes'] = np.around(np.sqrt(table['variance_bytes']),
                                        decimals=5)
    table['variance_time'] = np.around(np.sqrt(table['variance_time']),
                                       decimals=5)

    #tab_sort = np.sort(table[table['procs'] > 1], order='slowest_time')[::-1]
    tab_sort = np.sort(table, order='slowest_time')[::-1]

    # remove first column including the hash
    cols = list(table.dtype.names[1:])
    table_str = [[str(row[cell]) for cell in cols] for row in tab_sort[:20]]
    # alternative: remove first column including the hash
    # tab_sort = tab_sort[list(table.dtype.names[1:])]
    # table_str = [[str(cell) for cell in row] for row in tab_sort]
    table_head = "Variance in Files"
    column_heads = ['File', '#Procs', 'Fast Rank', 'Fast Time', 'Fast Bytes',
                    'Slow Rank', 'Slow Time', 'Slow Bytes',
                    'Std Time', 'Std Bytes']
    table_str.insert(0, column_heads)
    write_jube_table(table_head, table_str)

##################################################################################
#darshan3: counter namen geaendert, file_hash und ranks selbst bestimmt, kurz Tabelle Option
def variance_table3(nprocs, counters, data, short=True):
    """

    :param nprocs:
    :param file_hash:
    :param counters:
    :param data:
    :param short:
    :return:
    """
    ranks = data['rank'][data['counter'] == 'POSIX_BYTES_READ']
    file_hash = data['file'][data['counter'] == 'POSIX_BYTES_READ']

    names = data[counters["POSIX_F_OPEN_TIMESTAMP"]]["name_suffix"]
    slow_rank = data[counters['POSIX_SLOWEST_RANK']]["value"]
    slow_rank_time = data[counters['POSIX_F_SLOWEST_RANK_TIME']]["value"]
    slow_rank_byte = data[counters['POSIX_SLOWEST_RANK_BYTES']]["value"]
    fast_rank = data[counters['POSIX_FASTEST_RANK']]["value"]
    fast_rank_time = data[counters['POSIX_F_FASTEST_RANK_TIME']]["value"]
    fast_rank_byte = data[counters['POSIX_FASTEST_RANK_BYTES']]["value"]
    var_rank_time = data[counters['POSIX_F_VARIANCE_RANK_TIME']]["value"]
    var_rank_byte = data[counters['POSIX_F_VARIANCE_RANK_BYTES']]["value"]
    posix_meta = data[counters['POSIX_F_META_TIME']]["value"]
    posix_read = data[counters['POSIX_F_READ_TIME']]["value"]
    posix_write = data[counters['POSIX_F_WRITE_TIME']]["value"]
    bytes_read = data[counters['POSIX_BYTES_READ']]["value"]
    bytes_written = data[counters['POSIX_BYTES_WRITTEN']]["value"]

    table_dtype = [('hash', np.uint64), ('name_suffix', object),
                   ('procs', np.int), ('fastest_rank', np.int),
                   ('fastest_time', np.float64), ('fastest_bytes', np.uint64),
                   ('slowest_rank', np.int), ('slowest_time', np.float64),
                   ('slowest_bytes', np.uint64), ('variance_time', np.float64),
                   ('variance_bytes', np.float64)]
    table = np.array(list(), dtype=table_dtype)

    var_dtype = [('hash', np.uint64), ('var_time_S', np.float64),
                 ('var_time_T', np.float64), ('var_time_n', np.int),
                 ('var_bytes_S', np.float64), ('var_bytes_T', np.float64),
                 ('var_bytes_n', np.int)]
    var_help = np.array(list(), dtype=var_dtype)

    for i, f_hash in enumerate(file_hash):
        if ranks[i] == -1:
            row = np.array([(f_hash, names[i], nprocs, fast_rank[i],
                             fast_rank_time[i], fast_rank_byte[i], slow_rank[i],
                             slow_rank_time[i], slow_rank_byte[i],
                             var_rank_time[i], var_rank_byte[i])],
                           dtype=table_dtype)
            table = np.append(table, row)
        else:
            total_time = posix_meta[i] + posix_read[i] + posix_write[i]
            total_bytes = bytes_read[i] + bytes_written[i]
            if f_hash not in table['hash']:
                row = np.array([(f_hash, names[i], 1, ranks[i], total_time,
                                 total_bytes, ranks[i], total_time, total_bytes,
                                 0, 0)], dtype=table_dtype)
                table = np.append(table, row)
                # set temporary table to calculate the variance
                row = np.array([(f_hash, 0, total_time, 1, 0, total_bytes, 1)],
                               dtype=var_dtype)
                var_help = np.append(var_help, row)
            else:
                hash_ind = var_help['hash'] == f_hash
                # calculate variance of time
                n_tmp = var_help['var_time_n'][hash_ind]
                m_tmp = 1.
                t_tmp = var_help['var_time_T'][hash_ind]
                var_help['var_time_S'][hash_ind] += \
                    (m_tmp/(n_tmp*(n_tmp+m_tmp))) * \
                    ((n_tmp/m_tmp)*total_time - t_tmp)**2
                var_help['var_time_T'][hash_ind] += total_time
                var_help['var_time_n'][hash_ind] += 1
                table['variance_time'][table['hash'] == f_hash] =  \
                    var_help['var_time_S'][hash_ind] / \
                    var_help['var_time_n'][hash_ind]

                # calculate variance of bytes
                n_tmp = var_help['var_bytes_n'][hash_ind]
                m_tmp = 1.
                t_tmp = var_help['var_bytes_T'][hash_ind]
                var_help['var_bytes_S'][hash_ind] += \
                    (m_tmp/(n_tmp*(n_tmp+m_tmp))) * \
                    ((n_tmp/m_tmp)*total_bytes - t_tmp)**2
                var_help['var_bytes_T'][hash_ind] += total_bytes
                var_help['var_bytes_n'][hash_ind] += 1
                table['variance_bytes'][table['hash'] == f_hash] =  \
                    var_help['var_bytes_S'][hash_ind] / \
                    var_help['var_bytes_n'][hash_ind]

                # increase number of processes accessing the file
                table['procs'][table['hash'] == f_hash] = \
                    var_help['var_time_n'][hash_ind]

                # update slowest/fastest time/rank/bytes
                if table['slowest_time'][table['hash'] == f_hash] < total_time:
                    table['slowest_time'][table['hash'] == f_hash] = total_time
                    table['slowest_rank'][table['hash'] == f_hash] = ranks[i]
                    table['slowest_bytes'][table['hash'] == f_hash] = total_bytes
                if table['fastest_time'][table['hash'] == f_hash] > total_time:
                    table['fastest_time'][table['hash'] == f_hash] = total_time
                    table['fastest_rank'][table['hash'] == f_hash] = ranks[i]
                    table['fastest_bytes'][table['hash'] == f_hash] = total_bytes

    table['variance_bytes'] = np.around(np.sqrt(table['variance_bytes']),
                                        decimals=5)
    table['variance_time'] = np.around(np.sqrt(table['variance_time']),
                                       decimals=5)

    if short:
        tab_sort = np.sort(table[table['procs'] == nprocs],
                           order='slowest_time')[::-1][:20]
    else:
        tab_sort = np.sort(table, order='slowest_time')[::-1]

    # remove first column including the hash
    cols = list(table.dtype.names[1:])
    table_str = [[str(row[cell]) for cell in cols] for row in tab_sort]
    # alternative: remove first column including the hash
    # tab_sort = tab_sort[list(table.dtype.names[1:])]
    # table_str = [[str(cell) for cell in row] for row in tab_sort]
    table_head = "Variance in Files (POSIX)"
    column_heads = ['File', '#Procs', 'Fast Rank', 'Fast Time', 'Fast Bytes',
                    'Slow Rank', 'Slow Time', 'Slow Bytes',
                    'Std Time', 'Std Bytes']
    table_str.insert(0, column_heads)
    write_jube_table(table_head, table_str)

##################################################################################
#darshan3.1: 
def variance_table31(nprocs, counters, data, short=True):
    """

    :param nprocs:
    :param file_hash:
    :param counters:
    :param data:
    :param short:
    :return:
    """
    names = data['name_suffix'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]
    ranks = data['rank'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]
    file_hash = data['file'][(data['counter'] == "POSIX_OPENS") | (data['counter'] == "STDIO_OPENS")]

    slow_rank = data['value'][(data['counter'] == "POSIX_SLOWEST_RANK") | (data['counter'] == "STDIO_SLOWEST_RANK")]
    slow_rank_time = data['value'][(data['counter'] == "POSIX_F_SLOWEST_RANK_TIME") | (data['counter'] == "STDIO_F_SLOWEST_RANK_TIME")]
    slow_rank_byte = data['value'][(data['counter'] == "POSIX_SLOWEST_RANK_BYTES") | (data['counter'] == "STDIO_SLOWEST_RANK_BYTES")]
    fast_rank = data['value'][(data['counter'] == "POSIX_FASTEST_RANK") | (data['counter'] == "STDIO_FASTEST_RANK")]
    fast_rank_time = data['value'][(data['counter'] == "POSIX_F_FASTEST_RANK_TIME") | (data['counter'] == "STDIO_F_FASTEST_RANK_TIME")]
    fast_rank_byte = data['value'][(data['counter'] == "POSIX_FASTEST_RANK_BYTES") | (data['counter'] == "STDIO_FASTEST_RANK_BYTES")]
    var_rank_time = data['value'][(data['counter'] == "POSIX_F_VARIANCE_RANK_TIME") | (data['counter'] == "STDIO_F_VARIANCE_RANK_TIME")]
    var_rank_byte = data['value'][(data['counter'] == "POSIX_F_VARIANCE_RANK_BYTES") | (data['counter'] == "STDIO_F_VARIANCE_RANK_BYTES")]    
    posix_stdio_meta = data['value'][(data['counter'] == "POSIX_F_META_TIME") | (data['counter'] == "STDIO_F_META_TIME")]
    posix_stdio_read = data['value'][(data['counter'] == "POSIX_F_READ_TIME") | (data['counter'] == "STDIO_F_READ_TIME")]
    posix_stdio_write = data['value'][(data['counter'] == "POSIX_F_WRITE_TIME") | (data['counter'] == "STDIO_F_WRITE_TIME")]
    bytes_read = data['value'][(data['counter'] == "POSIX_BYTES_READ") | (data['counter'] == "STDIO_BYTES_READ")]
    bytes_written = data['value'][(data['counter'] == "POSIX_BYTES_WRITTEN") | (data['counter'] == "STDIO_BYTES_WRITTEN")]

    table_dtype = [('hash', np.uint64), ('name_suffix', object),
                   ('procs', np.int), ('fastest_rank', np.int),
                   ('fastest_time', np.float64), ('fastest_bytes', np.uint64),
                   ('slowest_rank', np.int), ('slowest_time', np.float64),
                   ('slowest_bytes', np.uint64), ('variance_time', np.float64), 
                  ('variance_bytes', np.float64)]
    table = np.array(list(), dtype=table_dtype)

    var_dtype = [('hash', np.uint64), ('var_time_S', np.float64),
                 ('var_time_T', np.float64), ('var_time_n', np.int),
                 ('var_bytes_S', np.float64), ('var_bytes_T', np.float64),
                 ('var_bytes_n', np.int)]
    var_help = np.array(list(), dtype=var_dtype)

    for i, f_hash in enumerate(file_hash):
        if ranks[i] == -1:
            row = np.array([(f_hash, names[i], nprocs, fast_rank[i],
                             fast_rank_time[i], fast_rank_byte[i], slow_rank[i],
                             slow_rank_time[i], slow_rank_byte[i],
                             var_rank_time[i], var_rank_byte[i])],
                           dtype=table_dtype)
            table = np.append(table, row)
        else:
            total_time = posix_stdio_meta[i] + posix_stdio_read[i] + posix_stdio_write[i]
            total_bytes = bytes_read[i] + bytes_written[i]
            if f_hash not in table['hash']:
                row = np.array([(f_hash, names[i], 1, ranks[i], total_time,
                                 total_bytes, ranks[i], total_time, total_bytes,
                                 0, 0)], dtype=table_dtype)
                table = np.append(table, row)
                # set temporary table to calculate the variance
                row = np.array([(f_hash, 0, total_time, 1, 0, total_bytes, 1)],
                               dtype=var_dtype)
                var_help = np.append(var_help, row)
            else:
                hash_ind = var_help['hash'] == f_hash
                # calculate variance of time
                n_tmp = var_help['var_time_n'][hash_ind]
                m_tmp = 1.
                t_tmp = var_help['var_time_T'][hash_ind]
                var_help['var_time_S'][hash_ind] += \
                    (m_tmp/(n_tmp*(n_tmp+m_tmp))) * \
                    ((n_tmp/m_tmp)*total_time - t_tmp)**2
                var_help['var_time_T'][hash_ind] += total_time
                var_help['var_time_n'][hash_ind] += 1
                table['variance_time'][table['hash'] == f_hash] =  \
                    var_help['var_time_S'][hash_ind] / \
                    var_help['var_time_n'][hash_ind]

                # calculate variance of bytes
                n_tmp = var_help['var_bytes_n'][hash_ind]
                m_tmp = 1.
                t_tmp = var_help['var_bytes_T'][hash_ind]
                var_help['var_bytes_S'][hash_ind] += \
                    (m_tmp/(n_tmp*(n_tmp+m_tmp))) * \
                    ((n_tmp/m_tmp)*total_bytes - t_tmp)**2
                var_help['var_bytes_T'][hash_ind] += total_bytes
                var_help['var_bytes_n'][hash_ind] += 1
                table['variance_bytes'][table['hash'] == f_hash] =  \
                    var_help['var_bytes_S'][hash_ind] / \
                    var_help['var_bytes_n'][hash_ind]

                # increase number of processes accessing the file
                table['procs'][table['hash'] == f_hash] = \
                    var_help['var_time_n'][hash_ind]

                # update slowest/fastest time/rank/bytes
                if table['slowest_time'][table['hash'] == f_hash] < total_time:
                    table['slowest_time'][table['hash'] == f_hash] = total_time
                    table['slowest_rank'][table['hash'] == f_hash] = ranks[i]
                    table['slowest_bytes'][table['hash'] == f_hash] = total_bytes
                if table['fastest_time'][table['hash'] == f_hash] > total_time:
                    table['fastest_time'][table['hash'] == f_hash] = total_time
                    table['fastest_rank'][table['hash'] == f_hash] = ranks[i]
                    table['fastest_bytes'][table['hash'] == f_hash] = total_bytes

    table['variance_bytes'] = np.around(np.sqrt(table['variance_bytes']),
                                        decimals=5)
    table['variance_time'] = np.around(np.sqrt(table['variance_time']),
                                       decimals=5)

    if short:
        print(table)
        tab_sort = np.sort(table[table['procs'] == nprocs], order='slowest_time')[::-1][:20]
    else:
        tab_sort = np.sort(table, order='slowest_time')[::-1]

    # remove first column including the hash
    cols = list(table.dtype.names[1:])
    table_str = [[str(row[cell]) for cell in cols] for row in tab_sort]
    # alternative: remove first column including the hash
    # tab_sort = tab_sort[list(table.dtype.names[1:])]
    # table_str = [[str(cell) for cell in row] for row in tab_sort]
    table_head = "Variance in Files (POSIX)"
    column_heads = ['File', '#Procs', 'Fast Rank', 'Fast Time', 'Fast Bytes',
                    'Slow Rank', 'Slow Time', 'Slow Bytes',
                    'Std Time', 'Std Bytes']
    table_str.insert(0, column_heads)
    write_jube_table(table_head, table_str)

##################################################################################


def darshan_2(output, runtime, nprocs, out_file, short):

    raw = [tuple(line.split()) for line in
                output[output.find("<fs type>\n") + 10:].split("\n")]

    # save input file column-wise
    log_dtype = [("rank", np.int64), ("file", np.uint64), ("counter", object),
                 ("value", np.float64), ("name_suffix", object),
                 ("mount_pt", object), ("fs type", object)]
    data = np.array(raw, dtype=log_dtype)

    # get all counter names; remove doubles
    all_counters = np.array(sorted(list(
            set(data[data["file"] == data["file"][0]]["counter"]))))
    # get boolean mask for every counter to collect
    # all values belonging to a counter in data
    counters = dict([(counter, data["counter"] == counter)
                     for counter in all_counters])
    # alternative
    # counters = {counter: data["counter"] == counter for counter in all_counters}

    ranks = data["rank"][0::len(all_counters)]
    file_hash = data["file"][0::len(all_counters)]
    mount_pt = data["mount_pt"][0::len(all_counters)]

    io_percentage(nprocs, runtime, counters, data, out_file)
    operation_counts(counters, data, out_file)
    io_sizes(counters, data)
    io_pattern(counters, data)
    access_size(counters, data, out_file, length=-1) #4)
    file_cnt_summary(file_hash, counters, data)
    avg_io(nprocs, ranks, counters, data, out_file)
    data_transfer(counters, data, mount_pt)
    variance_table(nprocs, ranks, file_hash, counters, data)
    if not short and not NOSTDOUT:
        create_ind_sha_table(nprocs, ranks, file_hash, counters, data, mount_pt)


def darshan_3(output, runtime, nprocs, out_file, short):

    output = re.sub(r'(?m)^\#.*\n?', '', output)
    raw = [tuple(line.strip().split()) for line in output.split("\n")
           if len(line) > 0]

    # save input file column-wise
    log_dtype = [("io_type", np.object), ("rank", np.int64),
                 ("file", np.uint64), ("counter", object), #file_hash
                 ("value", np.float64), ("name_suffix", object), #f_name
                 ("mount_pt", object), ("fs type", object)]
    data = np.array(raw, dtype=log_dtype)
    # get all counter names; remove doubles
    all_counters = np.array(sorted(list(
            set(data[data["file"] == data["file"][0]]["counter"]))))
    # get boolean mask for every counter to collect
    # all values belonging to a counter in data
    counters = dict([(counter, data["counter"] == counter)
                     for counter in all_counters])

    io_percentage3(nprocs, runtime, counters, data, out_file)
    operation_counts3(counters, data, out_file)
    io_sizes3(counters, data)
    io_pattern3(counters, data)
    access_size3(counters, data, out_file, length=-1) #4)
    file_cnt_summary3(counters, data)
    avg_io3(nprocs, counters, data, out_file)
    data_transfer3(counters, data)

    variance_table3(nprocs, counters, data, short=True)
    if not short and not NOSTDOUT:
        create_ind_sha_table3(nprocs, counters, data)

def darshan_31(output, runtime, nprocs, out_file, short):

    output = re.sub(r'(?m)^\#.*\n?', '', output)
    raw = [tuple(line.strip().split()) for line in output.split("\n")
           if len(line) > 0]

    # save input file column-wise
    log_dtype = [("io_type", np.object), ("rank", np.int64),
                 ("file", np.uint64), ("counter", object), #file_hash
                 ("value", np.float64), ("name_suffix", object), #f_name
                 ("mount_pt", object), ("fs type", object)]
    data = np.array(raw, dtype=log_dtype)

    # get all counter names; remove doubles
    #all_counters = np.array(sorted(list(
    #        set(data[data["file"] == data["file"][0]]["counter"]))))
    all_counters = set(data["counter"])

    # get boolean mask for every counter to collect
    # all values belonging to a counter in data
    counters = dict([(counter, data["counter"] == counter)
                     for counter in all_counters])

    io_percentage31(nprocs, runtime, counters, data, out_file)
    operation_counts31(counters, data, out_file)
    io_sizes31(counters, data)
    io_pattern31(counters, data)
    access_size31(counters, data, out_file, length=-1) #4)
    file_cnt_summary31(counters, data)
    avg_io31(nprocs, counters, data, out_file)
    data_transfer31(counters, data)
    variance_table31(nprocs, counters, data, short=short)
    if not short and not NOSTDOUT:
        create_ind_sha_table31(nprocs, counters, data)

def extract_darshan_metrics(filename):
    global NOSTDOUT
    global JUBE_TABLE
    NOSTDOUT = True
    JUBE_TABLE = False
    out_file = StringIO()
    open_and_parse(filename,True,out_file)
    result = out_file.getvalue()
    out_file.close()
    return result

def open_and_parse(darshan_logfile,short_output,out_file):
    try:
        output = Popen(shlex.split("darshan-parser {0}".format(darshan_logfile)),
                       stdout=PIPE).communicate()[0]
    except OSError:
        print("The darshan-parser commandline tool wasn't found or couldn't be executed. \n" +
              "Please check if you prepared the darshan environment.")
        exit(1)
    output = output.decode().strip()

    re_log_version = re.search("# darshan log version: (?P<log_v>\\d+\.\d+)", output)
    log_version = float(re_log_version.group("log_v"))
    re_nprocs = re.search("# nprocs: (?P<nprocs>\d*)", output)
    nprocs = int(re_nprocs.group("nprocs"))
    re_runtime = re.search("# run time: (?P<runtime>\d*)", output)
    runtime = int(re_runtime.group("runtime"))
    if not NOSTDOUT: print("Runtime: ", runtime)

    out_file.write('runtime: {0}\n\n'.format(runtime))

    if log_version == 3.1: 
        darshan_31(output, runtime, nprocs, out_file, short_output)
    elif log_version == 3.0:
        darshan_3(output, runtime, nprocs, out_file, short_output)
    elif int(log_version) == 2:
        darshan_2(output, runtime, nprocs, out_file, short_output)
    else:
        print("Darshan version unknown: {}".format(log_version))

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--file', required=True,
                        help='output file of the darshan profiling tool')
    parser.add_argument('-j', '--jube', action='store_true',
                        help='create jube table output')
    parser.add_argument('-nout', '--nostdout', action='store_true',
                        help='disable stdout output')
    parser.add_argument('-s', '--short', action='store_true',
                        help='without list of files')
    parser.add_argument('-v', '--version', default='3.1', choices=['2', '3.0', '3.1'],
                        help='darshan version')

    args = parser.parse_args()
    global NOSTDOUT
    NOSTDOUT = args.nostdout
    global JUBE_TABLE
    JUBE_TABLE = args.jube
    global DARSHAN_VERSION
    DARSHAN_VERSION = args.version

    out_file = open('./darshan.jube', 'w')
    open_and_parse(args.file,args.short,out_file)
    out_file.close()

if __name__ == "__main__":
    main()
