
! MODULE       : stomate_soilcarbon
!
! CONTACT      : orchidee-help _at_ ipsl.jussieu.fr
!
! LICENCE      : IPSL (2006)
! This software is governed by the CeCILL licence see ORCHIDEE/ORCHIDEE_CeCILL.LIC
!
!>\BRIEF       Calculate soil dynamics largely following the Century model
!
!!	
!!\n DESCRIPTION: 11 layers discretization: Layers 1-11 are soil layers
!!
!! RECENT CHANGE(S): None
!!
!! SVN		:
!! $HeadURL: svn://forge.ipsl.jussieu.fr/orchidee/perso/ronny.lauerwald/ORCHIDEE/src_stomate/stomate_soilcarbon.f90 $
!! $Date: 2016-11-03 14:49:26 +0100 (Thu, 03 Nov 2016) $
!! $Revision: 3876 $
!! \n
!_ ================================================================================================================================

MODULE stomate_soilcarbon

  ! modules used:
  USE xios_orchidee
  USE ioipsl_para
  USE stomate_data
  USE constantes
  USE constantes_soil
  USE pft_parameters
  USE pft_parameters_var
  USE vertical_soil
  IMPLICIT NONE

  ! private & public routines

  PRIVATE
  PUBLIC soilcarbon,soilcarbon_clear

  ! Variables shared by all subroutines in this module
  
  LOGICAL, SAVE    :: firstcall = .TRUE.   !! Is this the first call? (true/false)
!$OMP THREADPRIVATE(firstcall)

CONTAINS


!! ================================================================================================================================
!!  SUBROUTINE   : soilcarbon_clear
!!
!>\BRIEF        Set the flag ::firstcall to .TRUE. and as such activate sections 1.1.2 and 1.2 of the subroutine soilcarbon 
!! (see below).
!! 
!_ ================================================================================================================================
  
  SUBROUTINE soilcarbon_clear
    firstcall=.TRUE.
  ENDSUBROUTINE soilcarbon_clear


!! ================================================================================================================================
!!  SUBROUTINE   : soilcarbon
!!
!>\BRIEF        Computes the soil respiration and carbon stocks, essentially 
!! following Parton et al. (1987).
!!
!! DESCRIPTION	: The soil is divided into 3 carbon pools, with different 
!! characteristic turnover times : active (1-5 years), slow (20-40 years) 
!! and passive (200-1500 years).\n
!! There are three types of carbon transferred in the soil:\n
!! - carbon input in active and slow pools from litter decomposition,\n
!! - carbon fluxes between the three pools,\n
!! - carbon losses from the pools to the atmosphere, i.e., soil respiration.\n
!!
!! The subroutine performs the following tasks:\n
!!
!! Section 1.\n
!! The flux fractions (f) between carbon pools are defined based on Parton et 
!! al. (1987). The fractions are constants, except for the flux fraction from
!! the active pool to the slow pool, which depends on the clay content,\n
!! \latexonly
!! \input{soilcarbon_eq1.tex}
!! \endlatexonly\n
!! In addition, to each pool is assigned a constant turnover time.\n
!!
!! Section 2.\n
!! The carbon input, calculated in the stomate_litter module, is added to the 
!! carbon stock of the different pools.\n
!!
!! Section 3.\n
!! First, the outgoing carbon flux of each pool is calculated. It is 
!! proportional to the product of the carbon stock and the ratio between the 
!! iteration time step and the residence time:\n
!! \latexonly
!! \input{soilcarbon_eq2.tex}
!! \endlatexonly
!! ,\n
!! Note that in the case of crops, the additional multiplicative factor 
!! integrates the faster decomposition due to tillage (following Gervois et 
!! al. (2008)).
!! In addition, the flux from the active pool depends on the clay content:\n
!! \latexonly
!! \input{soilcarbon_eq3.tex}
!! \endlatexonly
!! ,\n
!! Each pool is then cut from the carbon amount corresponding to each outgoing
!! flux:\n
!! \latexonly
!! \input{soilcarbon_eq4.tex}
!! \endlatexonly\n
!! Second, the flux fractions lost to the atmosphere is calculated in each pool
!! by subtracting from 1 the pool-to-pool flux fractions. The soil respiration 
!! is then the summed contribution of all the pools,\n
!! \latexonly
!! \input{soilcarbon_eq5.tex}
!! \endlatexonly\n
!! Finally, each carbon pool accumulates the contribution of the other pools:
!! \latexonly
!! \input{soilcarbon_eq6.tex}
!! \endlatexonly
!!
!! RECENT CHANGE(S): None
!! 
!! MAIN OUTPUTS VARIABLE(S): carbon, resp_hetero_soil
!!
!! REFERENCE(S)   :
!! - Parton, W.J., D.S. Schimel, C.V. Cole, and D.S. Ojima. 1987. Analysis of 
!! factors controlling soil organic matter levels in Great Plains grasslands. 
!! Soil Sci. Soc. Am. J., 51, 1173-1179.
!! - Gervois, S., P. Ciais, N. de Noblet-Ducoudre, N. Brisson, N. Vuichard, 
!! and N. Viovy (2008), Carbon and water balance of European croplands 
!! throughout the 20th century, Global Biogeochem. Cycles, 22, GB2022, 
!! doi:10.1029/2007GB003018.
!! - Lardy, R, et al., A new method to determine soil organic carbon equilibrium,
!! Environmental Modelling & Software (2011), doi:10.1016|j.envsoft.2011.05.016
!!
!! FLOWCHART    :
!! \latexonly
!! \includegraphics[scale=0.5]{soilcarbon_flowchart.jpg}
!! \endlatexonly
!! \n
!_ ================================================================================================================================

  SUBROUTINE soilcarbon (npts, dt, clay, &
       soilcarbon_input, floodcarbon_input, control_temp_soil, control_moist_soil, &
       carbon, &
       resp_hetero_soil, resp_flood_soil, &
       litter_above,litter_below,&
       soilhum, DOC, moist_soil, DOC_EXP, &
       lignin_struc_above, lignin_struc_below, &
       floodout, runoff_per_soil, drainage_per_soil, wat_flux0, wat_flux,&
       bulk_dens, soil_ph, poor_soils, veget_max, soil_mc, soiltile, &
       DOC_to_topsoil, DOC_to_subsoil, flood_frac, &
       precip2ground, precip2canopy, canopy2ground, &
       dry_dep_canopy, DOC_precip2ground, DOC_precip2canopy, DOC_canopy2ground, &
       DOC_infil, DOC_noinfil, interception_storage, biomass, fastr)

!! 0. Variable and parameter declaration

    !! 0.1 Input variables
    
    INTEGER(i_std), INTENT(in)                                        :: npts                  !! Domain size (unitless)
    REAL(r_std), INTENT(in)                                           :: dt                    !! Time step \f$(dtradia one_day^{-1})$\f
    REAL(r_std), DIMENSION(npts), INTENT(inout)                       :: clay                  !! Clay fraction (unitless, 0-1) 
    REAL(r_std), DIMENSION(npts), INTENT(inout)                       :: bulk_dens             !! Variable local of bulk density for the moment must change in the futur (kg m-3)

    REAL(r_std), DIMENSION(npts), INTENT(in)                          :: soil_ph               !! soil pH (pH unit, 0-14)
    REAL(r_std), DIMENSION(npts), INTENT(in)                          :: poor_soils            !! Fraction of poor soils (0-1) 
    REAL(r_std), DIMENSION(npts,nbdl,npool*2), INTENT(in)             :: control_temp_soil     !! Temperature control of heterotrophic respiration (unitless: 0->1)
    REAL(r_std), DIMENSION(npts,nbdl,nvm), INTENT(in)                 :: control_moist_soil    !! Moisture control of heterotrophic respiration (unitless: 0.25->1)

    REAL(r_std), DIMENSION(npts,nlitt,nvm,nelements), INTENT(in)      :: litter_above          !! Metabolic and structural litter,above ground
                                                                                               !! The unit is given by m^2 of ground
                                                                                               !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements), INTENT(in) ::litter_below           !! Metabolic and structural litter, below ground
                                                                                               !! The unit is given by m^2 of
                                                                                               !! ground @tex $(gC m^{-2}) $ @endtex
    REAL(r_std), DIMENSION(npts,nbdl), INTENT(in)                     :: soilhum               !! Daily soil humidity of each soil layer 
                                                                                               !! (unitless)
    REAL(r_std), DIMENSION(npts,nvm), INTENT(in)                      :: lignin_struc_above    !! Ratio Lignin content in structural litter,
                                                                                               !! above ground, (0-1, unitless)
    REAL(r_std), DIMENSION(npts,nvm,nbdl), INTENT(in)                 :: lignin_struc_below    !! Ratio Lignin content in structural litter,
                                                                                               !! below ground, (0-1, unitless)
    REAL(r_std),DIMENSION (npts), INTENT (in)                         :: floodout              !! flux out of floodplains
    REAL(r_std),DIMENSION (npts,nstm), INTENT (in)                    :: runoff_per_soil       !! Runoff per soil type [mm]
    REAL(r_std),DIMENSION (npts,nstm), INTENT (in)                    :: drainage_per_soil     !! Drainage per soil type [mm]
    REAL(r_std),DIMENSION (npts,nstm), INTENT(in)                     :: wat_flux0             !! Water flux in the first soil layers exported for soil C calculations(mm)
    REAL(r_std),DIMENSION (npts,nslm,nstm), INTENT(in)                :: wat_flux              !! Water flux in the soil layers exported for soil C calculations(mm)
    REAL(r_std),DIMENSION(npts,nvm),INTENT(in)                        :: veget_max             !! PFT "Maximal" coverage fraction of a PFT 
                                                                                               !! defined in the input vegetation map 
                                                                                               !! @tex $(m^2 m^{-2})$ @endtex 
                                                                                               !! The unit is given by m^2 of
                                                                                               !! water @tex $(gC m{-2} of water)$ @endtex
    REAL(r_std),DIMENSION (npts,nbdl,nstm), INTENT(in)                :: soil_mc               !! soil moisture content \f($m^3 \times m^3$)\f
    REAL(r_std),DIMENSION (npts,nstm), INTENT (in)                    :: soiltile              !! Fraction of each soil tile (0-1, unitless)

    REAL(r_std),DIMENSION (npts,nflow), INTENT(in)                    :: DOC_to_topsoil        !! DOC inputs to top of the soil column, from reinfiltration on
                                                                                               !! floodplains and from irrigation
                                                                                               !! @tex $(gC m^{-2} day{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm), INTENT(in)                      :: precip2ground         !! Precipitation not intercepted by vegetation
                                                                                               !! @tex $(mm.dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm), INTENT(in)                      :: precip2canopy         !! Precipitation intercepted by vegetation
                                                                                               !! @tex $(mm.dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm), INTENT(in)                      :: canopy2ground         !! Water flux from canopy to ground
                                                                                               !! @tex $(mm.dt^{-1})$ @endtex 
    REAL(r_std),DIMENSION (npts,nflow), INTENT(in)                    :: DOC_to_subsoil        !! DOC inputs to bottom of the soil column, from returnflow
                                                                                               !! in swamps and lakes
                                                                                               !! @tex $(gC m^{-2} day{-1})$ @endtex
    REAL(r_std), DIMENSION(npts), INTENT(in)                          :: flood_frac            !! Flooded fraction of grid box
                                                                                               !! @tex $(-)$ @endtex
    REAL(r_std), DIMENSION(npts), INTENT(in)                          :: fastr                 !! Fast reservoir (mm)

    !! 0.2 Output variables
    
    REAL(r_std), DIMENSION(npts,nvm), INTENT(out)                     :: resp_hetero_soil      !! Soil heterotrophic respiration \f$(gC m^{-2} (dtradia one_day^{-1})^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm), INTENT(out)                     :: resp_flood_soil       !! Soil heterotrophic respiration when flooded 
                                                                                               !! \f$(gC m^{-2} (dtradia one_\day^{-1})^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm,nexp,npool,nelements), INTENT(out):: DOC_EXP               !! Exported DOC through runoff, drainage, flood,
                                                                                               !! The unit is given by m^2 of
                                                                                               !! ground \f$(gC m^{-2} day^{-1})$\f
    REAL(r_std), DIMENSION(npts,nbdl), INTENT(out)                    :: moist_soil            !! moisture in soil for one pixel (m3/m3)
    !! TF-DOC, maybe put all togetehr into one matrix, note that all Throughfall related fluxes are expressed relative to the total area of the pixel
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(out):: dry_dep_canopy        !! Increase in canopy storage of soluble OC & DOC
                                                                                    !! @tex $(gC.m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(out):: DOC_precip2canopy     !! Wet deposition of DOC onto canopy
                                                                                    !! @tex $(gC.m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(out):: DOC_precip2ground     !! Wet deposition of DOC not intecepted by canopy
                                                                                    !! @tex $(gC.m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(out):: DOC_canopy2ground     !! Wet deposition of DOC not intecepted by canopy
                                                                                    !! @tex $(gC.m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(out):: DOC_infil             !! Wet deposition of DOC infiltrating into the ground
                                                                                    !! @tex $(gC.m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(out):: DOC_noinfil           !! Wet deposition of DOC going into surface runoff
                                                                                    !! @tex $(gC.m^{-2})$ @endtex  
    !! 0.3 Modified variables

    REAL(r_std), DIMENSION(npts,nvm,nbdl,npool,nelements), INTENT(inout) :: soilcarbon_input   !! Amount of carbon going into the carbon pools 
                                                                                               !! from litter decomposition \f$(gC m^{-2} day^{-1})$\f 
    REAL(r_std), DIMENSION(npts,nvm,npool,nelements), INTENT(inout)    :: floodcarbon_input    !! Amount of carbon going directly into the water column
                                                                                          !! from litter decomposition \f$(gC m^{-2} day^{-1})$\f   
    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl), INTENT(inout)                :: carbon   !! Soil carbon pools: active, slow, or passive, \f$(gC m^{2})$\f
    REAL(r_std), DIMENSION(npts,nvm,nbdl,ndoc,npool,nelements), INTENT(inout) :: DOC      !! Dissolved Organic Carbon in soil
                                                                                          !! The unit is given by m^2 of
                                                                                          !! ground @tex $(gC m{-2} of ground)$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nparts,nelements), INTENT(inout) :: biomass           !! Biomass @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements), INTENT(inout):: interception_storage      !! Storage of DOC and solube OC in canopy
                                                                                          !! @tex $(gC.m^{-2})$ @endtex  
    !! 0.4 Local variables
    REAL(r_std), SAVE, DIMENSION(ncarb)                           :: carbon_tau       !! Residence time in carbon pools (years)
!$OMP THREADPRIVATE(carbon_tau)
    REAL(r_std), DIMENSION(npts,ncarb,ncarb)                      :: frac_carb        !! Flux fractions between carbon pools 
                                                                                      !! (second index=origin, third index=destination) 
                                                                                      !! (unitless, 0-1)
    REAL(r_std), DIMENSION(npts,ncarb,nelements,nbdl)             :: fluxtot          !! Total flux out of carbon pools \f$(gC m^{2})$\f
    REAL(r_std), DIMENSION(npts,nvm,nbdl,npool)                   :: fluxtot_DOC      !! Total flux out of dissolved organic carbon pools \f$(gC m^{2}ground)$\f
    REAL(r_std), DIMENSION(npts,ncarb,nelements,nbdl)             :: fluxtot_flood    !! Total flux out of carbon pools \f$(gC m^{2})$\f
    REAL(r_std), DIMENSION(npts,nvm,nbdl,npool)                   :: fluxtot_DOC_flood!! Total flux out of dissolved organic carbon pools \f$(gC m^{2}ground\)$\f
    REAL(r_std), DIMENSION(npts,npool,ncarb,nelements,nbdl)       :: flux             !! Fluxes between carbon pools \f$(gC m^{2})$\f
    CHARACTER(LEN=7), DIMENSION(ncarb)                            :: carbon_str       !! Name of the carbon pools for informative outputs (unitless)
    REAL(r_std), SAVE, DIMENSION(ncarb)                           :: priming_param    !! parmater controlling the sensitvity of priming effect on  each pools to the amount of LOM
!$OMP THREADPRIVATE(priming_param)
    REAL(r_std), SAVE, DIMENSION(npool)                           :: DOC_tau          !! Residence time of DOC (days)
!$OMP THREADPRIVATE(DOC_tau)
    REAL(r_std), DIMENSION(npts,nvm,nbdl)                         :: litter_tot       !! total litter (gC/m**2 of ground)
    REAL(r_std), DIMENSION(npts,nvm,nbdl)                         :: LOM              !! total labile organic matter that could induce the decompostion rate
                                                                                      !! (gC/m**2 of ground)
    INTEGER(i_std)                                                :: k,kk,m,j,l,i,kkk !! Indices (unitless)
    REAL(r_std),DIMENSION(0:nbdl)                                 :: z_soil           !! Soil levels (m)
    REAL(r_std), DIMENSION(npts,nvm, nbdl,npool,nelements)        :: DOC_RE           !! Adsoprtion Desorption flux of DOC (mmol C/kg)
    REAL(r_std), DIMENSION(npts,nvm, nbdl,npool,nelements)        :: DOC_FLUX         !! DOC flux between layers in f$(gC m^{-2} day^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm, nbdl,npool,nelements)        :: DOC_FLUX_DIFF    !! DOC flux by diffusion between layers in f$(gC m^{-2} day^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm, nbdl,npool,nelements)        :: DOC_FLUX_DIFF_old !! DOC flux by diffusion between layers in f$(gC m^{-2} day^{-1})$\f for storage
    REAL(r_std), DIMENSION(npts,nvm,npool,nelements)              :: DOC_RUN          !! DOC lost with runoff in f$(gC m^{-2} day^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm,npool,nelements)              :: DOC_FLOOD        !! DOC lost with flooding in f$(gC m^{-2} day^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm,npool,nelements)              :: DOC_DRAIN        !! DOC lost with drainage in f$(gC m^{-2} day^{-1})$\f
    REAL(r_std), DIMENSION(npts,nvm,nbdl,ndoc,npool,nelements)    :: DOC_old          !! Dissolved Organic Carbon in soil
                                                                                      !! The unit is given by m^2 of
                                                                                      !! ground @tex $(gC m{-2} of ground)$ @endtex variable used for storage
    REAL(r_std), DIMENSION(npts,nvm,nbdl,ndoc,npool,nelements)    :: DOC_old2         !! Dissolved Organic Carbon in soil
                                                                                      !! The unit is given by m^2 of
                                                                                      !! ground @tex $(gC m{-2} of ground)$ @endtex variable used for storage
    REAL(r_std), DIMENSION(npts,nvm,nbdl,ndoc,npool,nelements)    :: DOC_old_buffer   !! Dissolved Organic Carbon in soil
                                                                                      !! The unit is given by m^2 of
                                                                                      !! ground @tex $(gC m{-2} of ground)$ @endtex variable used for storage

    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl)                   :: carbon_old       !! Soil carbon pools: active, slow, or passive, \f$(gC m^{2})$\f used for storage
    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl)                   :: carbon_old_buffer!! Soil carbon pools: active, slow, or passive, \f$(gC m^{2})$\f used for storage
    REAL(r_std), DIMENSION(npts)                                  :: Qmax             !! Maximum adsorption capacity for DOC in the Langmuir equation (mg kg¿1)
    REAL(r_std), DIMENSION(npts,nbdl)                             :: Kd               !! distribution coefficient between DOC adsorbed and DOC free (unitless [0-1])
    REAL(r_std), DIMENSION(npts,nbdl)                             :: b                !! desorption term of the Initial Mass (IM) isotherm (mmol C/kg)
    REAL(r_std), DIMENSION(npts,nbdl)                             :: IM               !! sorption affinity of the Initial Mass (IM) isotherm (unitless)

    REAL(r_std), DIMENSION(npts)                                  :: Dif_coef         !! Diffusion coeficient used for bioturbation (m2 dt-1)

    REAL(r_std), DIMENSION(npts)                                  :: Dif_DOC          !! Diffusion coeficient used for DOC diffusion (m2 dt-1)

    REAL(r_std), DIMENSION(npts)                                  :: CUE_coef         !! Partition coefficient for DOC decomposed between CO2 and POC (unitless, 0-1)
                                                                                    
    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl)                   :: carbon_flux      !! Soil carbon flux within pools towards the soil column\f$(gC m^{2} dt^{-1})$\f
    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl)                   :: carbon_flux_old  !! Soil carbon flux within pools towards the soil column\f$(gC m^{2} dt^{-1})$\f
                                                                                      !! used for storage
    REAL(r_std), DIMENSION(npts,nstm)                             :: soilwater_31mm   !! Soil water content of the first x layers (m)
    REAL(r_std), DIMENSION(npts)                                  :: soilDOC_31mm     !! Soil DOC content of the first x layers (m)
    REAL(r_std), DIMENSION(npts)                                  :: soilDOC_corr     !! Soil DOC content of the first x layers (m)
    REAL(r_std), DIMENSION(npts)                                  :: flux_red         !! Flux reduction in upper meter of the soil
    REAL(r_std), DIMENSION(npts,nvm,nmbcomp,nelements)            :: check_intern     !! Contains the components of the internal
                                                                                      !! mass balance chech for this routine
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements)                    :: closure_intern   !! Check closure of internal mass balance

    REAL(r_std), DIMENSION(npts,nvm,nelements)                    :: pool_start       !! Start and end pool of this routine 
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements)                    :: pool_end         !! Start and end pool of this routine 
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex 
    REAL(r_std), DIMENSION(npts,nvm,nelements)                    :: pool_end_after   !! Start and end pool of this routine after mass balance correction 
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements)                    :: wet_dep_ground    !! total wet deposition of DOC onto the ground
    REAL(r_std), DIMENSION(npts,nvm,nelements)                    :: wet_dep_flood     !! total wet deposition of DOC onto the flooded
    REAL(r_std), DIMENSION(npts)                                  :: bio_frac          !! total wet deposition of DOC onto the flooded
    REAL(r_std), DIMENSION(npts)                                  :: fastr_corr        !! correction factor based on fast reservoir
    REAL(r_std)                                                   :: lat_exp           !! factor used to disable soil-water export of DOC
!_ ================================================================================================================================

    !! bavard is the level of diagnostic information, 0 (none) to 4 (full)
    IF (printlev>=3) WRITE(numout,*) 'Entering soilcarbon' 

!! 1. Initializations
!    clay(:) = 1.E-06
    !! 1.0 Calculation of soil moisture
    IF (lat_exp_doc) THEN
       lat_exp = un
    ELSE
       lat_exp = zero
    ENDIF

    moist_soil(:,:) = zero
      DO l = 1,nbdl
          DO i = 1, nstm
             moist_soil(:,l)= moist_soil(:,l) + soil_mc(:,l,i)*soiltile(:,i)
          ENDDO
      ENDDO
      !! 1.1 soil levels
       z_soil(0) = zero
       z_soil(1:nbdl) = diaglev(1:nbdl)

       !! DOC flux reduction factor, effects of poor soils
       flux_red(:) = flux_red_sro * (un - poor_soils(:)) + poor_soils(:)

       !! 1.2 TF-DOC: Initialize wet and dry deposition of DOC
       DOC_precip2canopy(:,:,icarbon) = zero
       DOC_precip2ground(:,:,icarbon) = zero
       dry_dep_canopy(:,:,icarbon) = zero
       bio_frac(:)=SUM(veget_max(:,2:13),DIM=2)

       IF (ok_TF_DOC) THEN   
          !! Calculate the DOC depoited onto the ground
          !! TF Calculate thoughfall DOC

          !! TF.1 Flux of DOC with precipitation onto the ground and onto the canopy (interception storage)
          DOC_precip2canopy(:,2:13,icarbon) = precip2canopy(:,2:13) * conc_DOC_rain * 1e-3
          DOC_precip2ground(:,2:13,icarbon) = precip2ground(:,2:13) * conc_DOC_rain * 1e-3
          !! TF.2 Daily contirubitoon of dry deposition and exsuadtes and feces

          DO j = 2,9! Loop over PFTs composed of trees
             DO i = 1, npts! Loop over # of pixels
                IF (veget_max(i,j) .GT. 0) THEN
                   dry_dep_canopy(i,j,icarbon) = DOC_incr_per_LEAF_M2 * dt * biomass(i,j,ileaf,icarbon) * veget_max(i,j)
                   !! So far, this is only the parametrisation for PFT=2
                   !! sa the rate is per day, we have to multiply by dt
                ELSE
                   dry_dep_canopy(i,j,icarbon) = zero
                ENDIF
             ENDDO ! Loop over # of pixels
          ENDDO ! # End Loop over # of PFTs
       ENDIF !(ok_TF_DOC)

    !! 1.4 Initialize check for mass balance closure
    !  The mass balance is calculated at the end of this routine
    !  in section 14. Initial biomass and harvest pool and all other
    !  relevant pools were set to zero before calling this routine.
    pool_start(:,:,:) = zero
    IF (ld_doc) THEN
       !Starting carbon pool
       DO m = 1, nvm
          DO i = 1,ncarb
             DO l = 1, nbdl
                pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                     ( carbon(:,i,m,l) ) * veget_max(:,m)
             ENDDO
          ENDDO
          pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + interception_storage(:,m,icarbon)
       ENDDO

       !Starting DOC pool
       DO m = 1, nvm
          DO i = 1,npool
             DO l = 1, nbdl
                DO j = 1, ndoc
                   pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                        ( DOC(:,m,l,j,i,icarbon)) * veget_max(:,m)
                ENDDO
             ENDDO
          ENDDO
       ENDDO


       !Starting litter below pool
       DO m = 1,nvm
          DO l = 1, nbdl
             pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                  (soilcarbon_input(:,m,l,imetbel,icarbon)+soilcarbon_input(:,m,l,istrbel,icarbon)+ &
                  soilcarbon_input(:,m,l,imetabo,icarbon)+soilcarbon_input(:,m,l,istrabo,icarbon)) * &
                  veget_max(:,m) * dt
          ENDDO
       ENDDO

       !Starting litter above pool
       DO m = 1,nvm
          WHERE (veget_max(:,m) .GT. zero)
             pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                  (DOC_to_topsoil(:,idocl) + DOC_to_subsoil(:,idocl) + DOC_to_topsoil(:,idocr) + DOC_to_subsoil(:,idocr) + &
                  floodcarbon_input(:,m,iact,icarbon) + floodcarbon_input(:,m,islo,icarbon))*dt*veget_max(:,m) + &
                  DOC_precip2canopy(:,m,icarbon) + DOC_precip2ground(:,m,icarbon) + dry_dep_canopy(:,m,icarbon)
          ELSEWHERE
             pool_start(:,m,icarbon) = pool_start(:,m,icarbon) 
          ENDWHERE
       ENDDO
    ENDIF

    !! 1.2 Get soil "constants"
    !! 1.2.1 Flux fractions between carbon pools: depend on clay content, recalculated each time
    ! From active pool: depends on clay content
    frac_carb(:,iactive,iactive) = zero
    frac_carb(:,iactive,ipassive) = frac_carb_ap /(un - metabolic_ref_frac + active_to_pass_clay_frac*clay(:))
    frac_carb(:,iactive,islow) = un - frac_carb(:,iactive,ipassive)  
    ! 1.2.1.2 from slow pool
    frac_carb(:,islow,islow) = zero
    frac_carb(:,islow,iactive) = frac_carb_sa
    frac_carb(:,islow,ipassive) = un - frac_carb(:,islow,iactive) 
    ! From passive pool
    frac_carb(:,ipassive,ipassive) = zero
    frac_carb(:,ipassive,iactive) = frac_carb_pa
    frac_carb(:,ipassive,islow) = un - frac_carb(:,ipassive,iactive)

    ! 1.2.1.3 Initialization
    carbon_flux(:,:,:,:) =  zero
    carbon_flux_old(:,:,:,:) =  zero
    IF ( firstcall ) THEN

        !! 1.2.2 Residence times in carbon pools (in year converted in days)
        carbon_tau(iactive) = carbon_tau_iactive * one_year      !residence times when SOM decomposition is function of litter coming from from optimization with the theoretical model SOM-FOM-DEP
        carbon_tau(islow) = carbon_tau_islow * one_year          ! See Guenet et al., (In prep)
        carbon_tau(ipassive) = carbon_tau_ipassive * one_year    

        priming_param(iactive)=priming_param_iactive  !Priming parameter for mineralization obtained from optimization of SOM-FOM-DEP see Guenet et al., (In prep)
        priming_param(islow)=priming_param_islow
        priming_param(ipassive)=priming_param_ipassive

        ! The values of DOC_tau are coming from Kalbitz et al 2003 Geoderma
        ! (already in days) 
        DOC_tau(imetabo) = DOC_tau_labile
        DOC_tau(istrabo) = DOC_tau_labile 
        DOC_tau(imetbel) = DOC_tau_labile  
        DOC_tau(istrbel) = DOC_tau_labile  
        DOC_tau(iact) = DOC_tau_labile  
        DOC_tau(islo) = DOC_tau_stable  
        DOC_tau(ipas) = DOC_tau_stable  

       !! 1.3 soil levels
        !! 1.4 Messages : display the residence times  
        carbon_str(iactive) = 'active'
        carbon_str(islow) = 'slow'
        carbon_str(ipassive) = 'passive'
        
        WRITE(numout,*) 'soilcarbon:'
        
        WRITE(numout,*) '   > minimal carbon residence time in carbon pools (d):'
        DO k = 1, ncarb ! Loop over carbon pools
          WRITE(numout,*) '(1, ::carbon_str(k)):',carbon_str(k),' : (1, ::carbon_tau(k)):',carbon_tau(k)
        ENDDO
        
        WRITE(numout,*) '   > flux fractions between carbon pools: depend on clay content'
        firstcall = .FALSE.
        
    ENDIF

    !! 1.5 Initialization
    !! 1.5.1 Initialization of parameters
        Dif_coef(:) = (Dif/one_year)*dt
        Dif_DOC(:)=D_DOC*dt        !We divided by one_year  
                                   !because the unit of the Dif
                                   !parameter is m2 yr-1 and the time stept of the model
                                   !is dt (1/48, by default, i.e. the fraction
                                   !of 30 min expressed in days) 
        CUE_coef(:)=CUE

    !! 1.5.2 Set soil respiration and DOC fluxes to zero
    resp_hetero_soil(:,:) = zero
    resp_flood_soil(:,:) = zero
    DOC_EXP(:,:,:,:,:) = zero
    DOC_RE(:,:,:,:,:) = zero
    DOC_FLUX(:,:,:,:,:) = zero
    DOC_FLUX_DIFF(:,:,:,:,:) = zero
    DOC_FLUX_DIFF_old(:,:,:,:,:) = zero
    DOC_RUN(:,:,:,:)= zero
    DOC_FLOOD(:,:,:,:)= zero
    DOC_DRAIN(:,:,:,:)= zero
    fluxtot(:,:,:,:)= zero
    fluxtot_DOC(:,:,:,:) = zero
    fluxtot_flood(:,:,:,:) = zero
    fluxtot_DOC_flood(:,:,:,:) = zero
    litter_tot(:,:,:) = zero
    DOC_canopy2ground(:,:,:) = zero
    wet_dep_flood(:,:,:) = zero
    wet_dep_ground(:,:,:) = zero
    LOM(:,:,:)= zero

    ! bulk_density must be in kg m-3 and is generally given in g cm-3
    WHERE (bulk_dens(:) .LT. 500)
    bulk_dens(:) = bulk_dens(:)*1e3
    ENDWHERE
  
    !!1.6 Calculating water content of the first 5 layers
    soilwater_31mm(:,:)= zero
    soilwater_31mm(:,:)= soilwater_31mm(:,:) + (z_soil(1) * soil_mc(:,1,:))   
     IF (sro_bottom .GT. 1) THEN
      DO l=2, sro_bottom
         soilwater_31mm(:,:)= soilwater_31mm(:,:) + ((z_soil(l)-z_soil(l-1)) * soil_mc(:,l,:))
      ENDDO
    ENDIF

    !! 2. Fluxes between carbon reservoirs, and to the atmosphere (respiration) \n
    !! 2.1 calculation of TF fluxes
    !! Calculation of TF-Fluxes

       !! TF.3 Calculate the flow to the ground

       !! Calculate max possible DOC flux, assuming a maximum concentration
       DO j = 2,nvm! Loop over # of PFTs
          DO i = 1, npts! Loop over # of pixels
             IF (veget_max(i,j) > zero) THEN
                DOC_canopy2ground(i,j,icarbon) = min(canopy2ground(i,j)*conc_DOC_max * 1e-3, &
                     &                                              interception_storage(i,j,icarbon))
             ELSE
                DOC_canopy2ground(i,j,icarbon) = zero
             ENDIF
          ENDDO ! Loop over # of pixels
       ENDDO ! # End Loop over # of PFTs

       DO j = 1,nvm! Loop over # of PFTs
          wet_dep_flood(:,j,icarbon)  = (DOC_precip2ground(:,j,icarbon) + DOC_canopy2ground(:,j,icarbon)) * flood_frac(:)
          wet_dep_ground(:,j,icarbon) = (DOC_precip2ground(:,j,icarbon) + DOC_canopy2ground(:,j,icarbon)) * (un-flood_frac(:))
       ENDDO

       IF (.NOT. lat_exp_doc) THEN
          wet_dep_ground = wet_dep_ground + wet_dep_flood
          wet_dep_flood = zero
       ELSE
          !Do nothing
       ENDIF

       !! TF.4 Update the canopy storage of soluble OC and DOC

       DO j = 2,nvm! Loop over # of PFTs
          DO i = 1, npts! Loop over # of pixels
             IF (veget_max(i,j) > zero) THEN
                interception_storage(i,j,icarbon) = interception_storage(i,j,icarbon) + dry_dep_canopy(i,j,icarbon) + &
                     &                          DOC_precip2canopy(i,j,icarbon) - DOC_canopy2ground(i,j,icarbon)
             ELSE
                interception_storage(i,j,icarbon) = zero
             ENDIF
          ENDDO ! Loop over # of pixels
       ENDDO ! # End Loop over # of PFTs

       !! TF.5 Calculate the fraction that is infiltrating into the soil
       DOC_infil = zero
       DOC_noinfil = zero

    !! 2.2 Input from litter to DOC and from reinflitration in floodplains
    !! For the variables DOC_to_topsoil and DOC_to_subsoil we multiplied by
    !veget_max because it is not defined per PFT and therefore we redistribute
    !it following veget_max. Moreover we assume the all the DOC coming from
    !floodplains reinfiltration is going to DOC coming from the active pool.
    DO m = 2, nvm
       DO l = 1,nbdl

          DOC(:,m,l,ifree,imetabo,icarbon) = DOC(:,m,l,ifree,imetabo,icarbon) + soilcarbon_input(:,m,l,imetabo,icarbon)*dt 
          DOC(:,m,l,ifree,istrabo,icarbon) = DOC(:,m,l,ifree,istrabo,icarbon) + soilcarbon_input(:,m,l,istrabo,icarbon)*dt
          DOC(:,m,l,ifree,imetbel,icarbon) = DOC(:,m,l,ifree,imetbel,icarbon) + soilcarbon_input(:,m,l,imetbel,icarbon)*dt
          DOC(:,m,l,ifree,istrbel,icarbon) = DOC(:,m,l,ifree,istrbel,icarbon) + soilcarbon_input(:,m,l,istrbel,icarbon)*dt

       ENDDO
          WHERE (veget_max(:,m) .GT. zero)
          DOC(:,m,1,ifree,iact,icarbon) = DOC(:,m,1,ifree,iact,icarbon) + DOC_to_topsoil(:,idocl)*dt/bio_frac(:) &
               + wet_dep_ground(:,m,icarbon) * undemi/veget_max(:,m)
          DOC(:,m,nbdl,ifree,iact,icarbon) = DOC(:,m,nbdl,ifree,iact,icarbon) + DOC_to_subsoil(:,idocl)*dt/bio_frac(:)
          DOC(:,m,1,ifree,islo,icarbon) = DOC(:,m,1,ifree,islo,icarbon) + DOC_to_topsoil(:,idocr)*dt/bio_frac(:) &
               + wet_dep_ground(:,m,icarbon) * undemi/veget_max(:,m)
          DOC(:,m,nbdl,ifree,islo,icarbon) = DOC(:,m,nbdl,ifree,islo,icarbon) + DOC_to_subsoil(:,idocr)*dt/bio_frac(:)
          ELSEWHERE
          DOC(:,m,1,ifree,iact,icarbon) = DOC(:,m,1,ifree,iact,icarbon) 
          DOC(:,m,nbdl,ifree,iact,icarbon) = DOC(:,m,nbdl,ifree,iact,icarbon)
          ENDWHERE
    ENDDO

    !! 2.3. Calculate fluxes
    DO m = 2,nvm
      !calculate total litter available
      litter_tot(:,m,1)=litter_above(:,imetabolic,m,icarbon)+litter_above(:,istructural,m,icarbon)
    
      DO l = 1, nbdl
      litter_tot(:,m,l)=litter_tot(:,m,l) + litter_below(:,imetabolic,m,l,icarbon)+litter_below(:,istructural,m,l,icarbon)
      ENDDO
   ENDDO

    DO m = 1,nvm ! Loop over # PFTs


      !! 2.3.1. Flux out of pools

      DO k = 1, ncarb ! Loop over carbon pools from which the flux comes

        DO l = 1, nbdl !Loop over soil layers

         ! Determine total flux out of pool
         ! S.L. Piao 2006/05/05 - for crop multiply tillage factor of decomposition
         ! Not crop
         

          IF (k == iactive) THEN
             LOM(:,m,l) = litter_tot(:,m,l) + DOC(:,m,l,ifree,imetabo,icarbon) + DOC(:,m,l,ifree,istrabo,icarbon) + DOC(:,m,l,ifree,imetbel,icarbon) + DOC(:,m,l,ifree,istrbel,icarbon)
          ELSEIF (k == islow) THEN
             LOM(:,m,l) = litter_tot(:,m,l) + carbon(:,iactive,m,l) + DOC(:,m,l,ifree,iact,icarbon) + &
                          DOC(:,m,l,ifree,imetabo,icarbon) + DOC(:,m,l,ifree,istrabo,icarbon) + DOC(:,m,l,ifree,imetbel,icarbon) + DOC(:,m,l,ifree,istrbel,icarbon)
          ELSEIF (k == ipassive) THEN
             LOM(:,m,l) = litter_tot(:,m,l) + carbon(:,iactive,m,l) + carbon(:,islow,m,l) + DOC(:,m,l,ifree,iact,icarbon) + DOC(:,m,l,ifree,islo,icarbon) + &
                          DOC(:,m,l,ifree,imetabo,icarbon) + DOC(:,m,l,ifree,istrabo,icarbon) + DOC(:,m,l,ifree,imetbel,icarbon) + DOC(:,m,l,ifree,istrbel,icarbon)
          ELSE
             WRITE(numout,*) "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"
             WRITE(numout,*) "BE CAREFUL SOMETHING STRANGE HAPPENS"
             WRITE(numout,*) "IN STOMATE_SOILCARBON FOR LOM CALCULATION"
             WRITE(numout,*) "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"

          ENDIF
             
       WHERE ((soil_mc(:,l,pref_soil_veg(m)) .GT. zero) .AND. (natural(m)))
               fluxtot(:,k,icarbon,l) =(dt/(carbon_tau(k)*(un+poor_soils(:))))*(un-exp(-priming_param(k)*LOM(:,m,l))) * carbon(:,k,m,l) * &
                    control_moist_soil(:,l,m) * control_temp_soil(:,l,k+4) * (un-flood_frac(:))
               fluxtot_flood(:,k,icarbon,l) =(dt/(carbon_tau(k)*(un+poor_soils(:))))*(un-exp(-priming_param(k)*LOM(:,m,l))) * carbon(:,k,m,l) * &
                    control_moist_soil(:,l,m) * control_temp_soil(:,l,k+4) * flood_frac(:) * un/trois
            ! C3 crop
       ELSEWHERE ((soil_mc(:,l,pref_soil_veg(m)) .GT. zero) .AND.( (.NOT. natural(m)) .AND. (.NOT. is_c4(m)) ))
               fluxtot(:,k,icarbon,l) =(dt/(carbon_tau(k)*(un+poor_soils(:))))*(un-exp(-priming_param(k)*LOM(:,m,l))) * carbon(:,k,m,l) * &
                    control_moist_soil(:,l,m) * control_temp_soil(:,l,k+4) * flux_tot_coeff(1) * (un-flood_frac(:))
               fluxtot_flood(:,k,icarbon,l) =(dt/(carbon_tau(k)*(un+poor_soils(:))))*(un-exp(-priming_param(k)*LOM(:,m,l))) * carbon(:,k,m,l) * &
                    control_moist_soil(:,l,m) * control_temp_soil(:,l,k+4) * flux_tot_coeff(1) * flood_frac(:) * un/trois
            ! C4 crop
       ELSEWHERE ((soil_mc(:,l,pref_soil_veg(m)) .GT. zero) .AND.( (.NOT. natural(m)) .AND. (is_c4(m)) )) 
               fluxtot(:,k,icarbon,l) =(dt/(carbon_tau(k)*(un+poor_soils(:))))*(un-exp(-priming_param(k)*LOM(:,m,l))) * carbon(:,k,m,l) * &
                    control_moist_soil(:,l,m) * control_temp_soil(:,l,k+4) * flux_tot_coeff(2) * (un-flood_frac(:))
               fluxtot_flood(:,k,icarbon,l) =(dt/(carbon_tau(k)*(un+poor_soils(:))))*(un-exp(-priming_param(k)*LOM(:,m,l))) * carbon(:,k,m,l) * &
                    control_moist_soil(:,l,m) * control_temp_soil(:,l,k+4) * flux_tot_coeff(2) * flood_frac(:) * un/trois
       ELSEWHERE
               fluxtot(:,k,icarbon,l) = zero
               fluxtot_flood(:,k,icarbon,l) = zero
       ENDWHERE

          ! END - S.L. Piao 2006/05/05 - for crop multiply tillage factor of decomposition

          ! Carbon flux from active pools depends on clay content
          IF ( k .EQ. iactive ) THEN
              fluxtot(:,k,icarbon,l) = fluxtot(:,k,icarbon,l) * ( un - flux_tot_coeff(3) * clay(:) )
              fluxtot_flood(:,k,icarbon,l) = fluxtot_flood(:,k,icarbon,l) * ( un - flux_tot_coeff(3) * clay(:) )
          ENDIF
        ENDDO ! End of loop over soil layers

      ENDDO ! End of loop over carbon pools

      IF (.NOT. lat_exp_doc) THEN
         fluxtot = fluxtot + fluxtot_flood
         fluxtot_flood = zero
      ELSE
         !do nothing
      ENDIF
      !! 2.3.2 respiration
      !BE CAREFUL: Here resp_hetero_soil is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_soil is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
      !
      DO l = 1,nbdl !Loop over soil layers
         resp_hetero_soil(:,m) = resp_hetero_soil(:,m)+&
            ( (un-CUE_coef(:)) * fluxtot(:,iactive,icarbon,l) + &
            (un-CUE_coef(:)) * fluxtot(:,islow,icarbon,l) + &
            (un-CUE_coef(:))* fluxtot(:,ipassive,icarbon,l)  ) / dt
         resp_flood_soil(:,m) = resp_flood_soil(:,m)+&
            ( (un-CUE_coef(:)) * fluxtot_flood(:,iactive,icarbon,l) + &
            (un-CUE_coef(:)) * fluxtot_flood(:,islow,icarbon,l) + &
            (un-CUE_coef(:))* fluxtot_flood(:,ipassive,icarbon,l)  ) / dt
      ENDDO !End loop over soil layers

      !! 2.3.3 DOC calculation
      !! 2.3.3.1 DOC decomposition
      ! In the Kalbitz et al., 2003 Geoderma as well
      ! as in Qualls and Haines 1992 SSAJ
      ! the equation are defined with exponential but for the time
      ! step we used we can approximate the values with 1st order kinetics.

     DO j = 1,npool  !Loop over pools
       DO l = 1,nbdl !Loop over soil layers
        WHERE (soil_mc(:,l,pref_soil_veg(m)) .GT. zero) 
                 fluxtot_DOC(:,m,l,j) = (dt/(DOC_tau(j)*(un+poor_soils(:))))* (un - flood_frac(:)) * &
                                         DOC(:,m,l,ifree,j,icarbon)*control_temp_soil(:,l,npool+j) 
                 fluxtot_DOC_flood(:,m,l,j) = (dt/(DOC_tau(j)*(un+poor_soils(:))))* flood_frac(:) * un/trois * &
                                         DOC(:,m,l,ifree,j,icarbon)*control_temp_soil(:,l,npool+j)
         ELSEWHERE
                 fluxtot_DOC(:,m,l,j) = zero
                 fluxtot_DOC_flood(:,m,l,j) = zero
         ENDWHERE

          resp_hetero_soil(:,m) = resp_hetero_soil(:,m)+&
             (un-CUE_coef(:))*fluxtot_DOC(:,m,l,j)/dt

          resp_flood_soil(:,m) = resp_flood_soil(:,m)+&
             (un-CUE_coef(:))*fluxtot_DOC_flood(:,m,l,j)/dt

      ENDDO !End loop over soil layers!
     ENDDO !End loop over soil pool!

      !! 2.3.4 Pools update
      !! 2.3.4.1 For POC
      DO l= 1, nbdl ! Loop over soil layers

       carbon(:,iactive,m,l) = carbon(:,iactive,m,l)+ &
                               frac_carb(:,ipassive,iactive)*(CUE_coef(:))*fluxtot_DOC(:,m,l,ipas) + &
                               frac_carb(:,islow,iactive)*(CUE_coef(:))*fluxtot_DOC(:,m,l,islo) + &
                               (CUE_coef(:))*fluxtot_DOC(:,m,l,imetbel) + &
                               (CUE_coef(:))*fluxtot_DOC(:,m,l,istrbel) * (un - lignin_struc_below(:,m,l)) + &
                               (CUE_coef(:))*fluxtot_DOC(:,m,l,imetabo) + &
                               (CUE_coef(:))*fluxtot_DOC(:,m,l,istrabo) * (un - lignin_struc_above(:,m)) + &
                               frac_carb(:,ipassive,iactive)*(CUE_coef(:))*fluxtot_DOC_flood(:,m,l,ipas) + &
                               frac_carb(:,islow,iactive)*(CUE_coef(:))*fluxtot_DOC_flood(:,m,l,islo) + &
                               (CUE_coef(:))*fluxtot_DOC_flood(:,m,l,imetbel) + &
                               (CUE_coef(:))*fluxtot_DOC_flood(:,m,l,istrbel) * (un - lignin_struc_below(:,m,l)) + &
                               (CUE_coef(:))*fluxtot_DOC_flood(:,m,l,imetabo) + &
                               (CUE_coef(:))*fluxtot_DOC_flood(:,m,l,istrabo) * (un - lignin_struc_above(:,m))

       carbon(:,islow,m,l) = carbon(:,islow,m,l)+ &
                               frac_carb(:,ipassive,islow)*(CUE_coef(:))*fluxtot_DOC(:,m,l,ipas) + &
                               frac_carb(:,iactive,islow)*(CUE_coef(:))*fluxtot_DOC(:,m,l,iact) + &
                               (CUE_coef(:))*fluxtot_DOC(:,m,l,istrbel) *lignin_struc_below(:,m,l) + &
                               (CUE_coef(:))*fluxtot_DOC(:,m,l,istrabo) *lignin_struc_above(:,m) + &
                               frac_carb(:,ipassive,islow)*(CUE_coef(:))*fluxtot_DOC_flood(:,m,l,ipas) + &
                               frac_carb(:,iactive,islow)*(CUE_coef(:))*fluxtot_DOC_flood(:,m,l,iact) + &
                               (CUE_coef(:))*fluxtot_DOC_flood(:,m,l,istrbel) *lignin_struc_below(:,m,l) + &
                               (CUE_coef(:))*fluxtot_DOC_flood(:,m,l,istrabo) *lignin_struc_above(:,m)

       carbon(:,ipassive,m,l) = carbon(:,ipassive,m,l)+ &
                               frac_carb(:,iactive,ipassive)*(CUE_coef(:))*fluxtot_DOC(:,m,l,iact) + &
                               frac_carb(:,islow,ipassive)*(CUE_coef(:))*fluxtot_DOC(:,m,l,islo) + &
                               frac_carb(:,iactive,ipassive)*(CUE_coef(:))*fluxtot_DOC_flood(:,m,l,iact) + &
                               frac_carb(:,islow,ipassive)*(CUE_coef(:))*fluxtot_DOC_flood(:,m,l,islo)
      ENDDO ! End loop over soil layers

      DO k = 1, ncarb ! Loop over carbon pools from which the flux comes
        DO l = 1, nbdl !Loop over soil layers

         carbon(:,k,m,l) = carbon(:,k,m,l) - fluxtot(:,k,icarbon,l) - fluxtot_flood(:,k,icarbon,l)

        ENDDO ! End of loop over soil layers
      ENDDO ! End of loop over carbon pools

      !! 2.3.4.2 For DOC
      DO l = 1,nbdl !Loop over soil layers
         DO kk = 1,nelements !Loop over elements
           IF (l .GT. sro_bottom) THEN
              !For Active pool
              DOC(:,m,l,ifree,iact,kk) = DOC(:,m,l,ifree,iact,kk) + (CUE_coef(:)) * fluxtot(:,iactive,kk,l) + &
                   (CUE_coef(:)) * fluxtot_flood(:,iactive,kk,l)
              
              !For Slow pool    
              DOC(:,m,l,ifree,islo,kk) = DOC(:,m,l,ifree,islo,kk) + (CUE_coef(:)) * fluxtot(:,islow,kk,l) + &
                   (CUE_coef(:)) * fluxtot_flood(:,islow,kk,l)

              !For Passive pool
              DOC(:,m,l,ifree,ipas,kk) = DOC(:,m,l,ifree,ipas,kk) + (CUE_coef(:)) * fluxtot(:,ipassive,kk,l) + &
                   (CUE_coef(:)) * fluxtot_flood(:,ipassive,kk,l)
           ELSE
              !For Active pool
              DOC(:,m,l,ifree,iact,kk) = DOC(:,m,l,ifree,iact,kk) + (CUE_coef(:)) * fluxtot(:,iactive,kk,l)
              !For Slow pool
              DOC(:,m,l,ifree,islo,kk) = DOC(:,m,l,ifree,islo,kk) + (CUE_coef(:)) * fluxtot(:,islow,kk,l) 
              !For Passive pool
              DOC(:,m,l,ifree,ipas,kk) = DOC(:,m,l,ifree,ipas,kk) + (CUE_coef(:)) * fluxtot(:,ipassive,kk,l)
           ENDIF
        ENDDO !End loop over elements
     ENDDO !End loop over soil layers

     DO j = 1,npool  !Loop over pools
       DO l = 1,nbdl !Loop over soil layers

          DOC(:,m,l,ifree,j,icarbon) = DOC(:,m,l,ifree,j,icarbon) - fluxtot_DOC(:,m,l,j) - fluxtot_DOC_flood(:,m,l,j)

       ENDDO !End loop over soil layers!
     ENDDO !End loop over soil pool!

      !! 2.3.5 Adsorption/resorption of DOC using linear Mass isotherm assuming
      !that DOC from native soil is zero (b parameters in Nodvin et al., 1986,
      !Soil Science)
      !For the moment, fixed parameters taken from Kaiser and others,1996, b in mg/g soil
     Kd(:,:)=kd_ads
     Kd(:,1) = dix**(-3.1+0.2*log10(clay(:)*cent)+log10(z_soil(1)*cent/deux))
     DO l = 2,nbdl
        ! IM, b or Kd calculating following the statistical model obtained in
        ! Camino Serrano et al. in prep
        Kd(:,l) = dix**(-3.1+0.2*log10(clay(:)*cent)+log10((z_soil(l)+z_soil(l-1))/deux))
        ! Kd(:,l) = 0.001225841-0.000212346*soil_ph(:)+0.003739918*clay(:)
     ENDDO
     Kd(:,:)=MAX(Kd(:,:),dix**(-3.2))
     Kd(:,:)=MIN(Kd(:,:),dix**(-1.5))

     ! Calculation of amount adsorbed/desorbed based on 
     ! Ota et al., 2013 JGR
     DO l = 1,nbdl !Loop over soil layers
        DO kk = 1,nelements !Loop over elements
           DO kkk = 1,npool !Loop over soil and litter pool
              DOC_RE(:,m,l,kkk,kk) = (Kd(:,l)*bulk_dens(:))/(Kd(:,l)*bulk_dens(:)+un)* &
                   (DOC(:,m,l,ifree,kkk,kk)+DOC(:,m,l,iadsorbed,kkk,kk)) 
           ENDDO !End loop over soil and litter pool
        ENDDO !End loop over elements
     ENDDO !End loop over soil layers

     DOC_old(:,m,:,:,:,:)= zero
     DOC_old(:,m,:,:,:,:)=DOC(:,m,:,:,:,:)
     DO l = 1,nbdl !Loop over soil layers
        DO kk = 1,nelements !Loop over elements
           DO kkk = 1,npool !Loop over soil and litter pool
              WHERE (((DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk)) .GT. zero) .AND. (abs(DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk)) .GT. DOC_old(:,m,l,ifree,kkk,kk)))
                 !To avoid adsorbing more DOC than existing (when there is not enough free carbon)
                 DOC(:,m,l,iadsorbed,kkk,kk)= DOC_old(:,m,l,iadsorbed,kkk,kk)+ DOC_old(:,m,l,ifree,kkk,kk)
                 DOC(:,m,l,ifree,kkk,kk)= zero           
              ELSEWHERE (((DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk)) .LT. zero) .AND. (abs(DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk)) .GT. DOC_old(:,m,l,iadsorbed,kkk,kk)))
               !To avoid desorbing more DOC than existing (when there is not enough adsorbed carbon)
                 DOC(:,m,l,iadsorbed,kkk,kk)= zero 
                 DOC(:,m,l,ifree,kkk,kk)= DOC_old(:,m,l,iadsorbed,kkk,kk)+ DOC_old(:,m,l,ifree,kkk,kk)        
              ELSEWHERE ((DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk)) .NE. zero) 
                !The normal functioning of the distribution coefficient to reach equilibrium between DOC adsorbed and DOC free   
                 DOC(:,m,l,ifree,kkk,kk)=DOC_old(:,m,l,ifree,kkk,kk)-(DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk))  
                 DOC(:,m,l,iadsorbed,kkk,kk)= DOC_old(:,m,l,iadsorbed,kkk,kk)+(DOC_RE(:,m,l,kkk,kk)-DOC_old(:,m,l,iadsorbed,kkk,kk))
              ELSEWHERE
                 DOC(:,m,l,ifree,kkk,kk)=DOC_old(:,m,l,ifree,kkk,kk)
                 DOC(:,m,l,iadsorbed,kkk,kk)= DOC_old(:,m,l,iadsorbed,kkk,kk)
              ENDWHERE
           ENDDO !End loop over soil and litter pool
        ENDDO !End loop over elements
     ENDDO !End loop over soil layers



     !! 2.3.6 Transport of DOC with the water fluxes following Futter et al.,
     ! (2007) Water ressources research. 
     
     !! 2.3.6.1 Change DOC free from gC m-2 of soil to g C m^-3 of water  
     DO kk = 1, nelements !Loop over elements
        
        DO k = 1, npool ! Loop over soil and litter pools
           
           WHERE( soil_mc(:,1,pref_soil_veg(m)) .GT. zero)

              DOC(:,m,1,ifree,k,kk) = DOC(:,m,1,ifree,k,kk) / (z_soil(1)*soil_mc(:,1,pref_soil_veg(m)))

           ELSEWHERE
              DOC(:,m,1,ifree,k,kk) = DOC(:,m,1,ifree,k,kk)
           ENDWHERE

        ENDDO ! End loop over soil and litter pools

     ENDDO !End loop over elements

     DO l = 2,nbdl !Loop over soil layers

        DO kk = 1, nelements !Loop over elements

           DO k = 1, npool ! Loop over soil and litter pools

              WHERE( soil_mc(:,l,pref_soil_veg(m)) .GT. zero)

                 DOC(:,m,l,ifree,k,kk) = DOC(:,m,l,ifree,k,kk) / ((z_soil(l)-z_soil(l-1))*soil_mc(:,l,pref_soil_veg(m)))

              ELSEWHERE
                 DOC(:,m,l,ifree,k,kk) = DOC(:,m,l,ifree,k,kk)
              ENDWHERE

           ENDDO ! End loop over soil and litter pools

        ENDDO !End loop over elements

     ENDDO !End loop over soil layers 
     ! DOC fluxes between layers calculated (we multiplied by
     ! 1E-3 to convert from kg/m2 to m3/m2)
     DO l = 1,nbdl-1 !Loop over soil layers
        DO kk = 1,nelements !Loop over elements
           DO kkk = 1,npool !Loop over soil and litter pool
              WHERE ((wat_flux(:,l,pref_soil_veg(m)) .GT. 0.) .AND. (soil_mc(:,l,pref_soil_veg(m)) .GT. zero))
                 DOC_FLUX(:,m,l,kkk,kk) = DOC(:,m,l,ifree,kkk,kk)*wat_flux(:,l,pref_soil_veg(m))*1E-3*flux_red(:)
              ELSEWHERE ((wat_flux(:,l,pref_soil_veg(m)) .LT. 0.) .AND. (soil_mc(:,l,pref_soil_veg(m)) .GT. zero))
                 DOC_FLUX(:,m,l,kkk,kk) = DOC(:,m,l+1,ifree,kkk,kk)*abs(wat_flux(:,l,pref_soil_veg(m)))*1E-3*flux_red(:)
              ELSEWHERE 
                 DOC_FLUX(:,m,l,kkk,kk) = zero
              ENDWHERE
           ENDDO !End loop over soil and litter pool
        ENDDO !End loop over elements
     ENDDO !End loop over soil layers

     !For the last layer
     DO kk = 1,nelements !Loop over elements
        DO kkk = 1,npool !Loop over soil and litter pool
           WHERE ((wat_flux(:,nbdl,pref_soil_veg(m)) .GT. 0.) .AND. (soil_mc(:,nbdl,pref_soil_veg(m)) .GT. zero))
              DOC_FLUX(:,m,nbdl,kkk,kk) = DOC(:,m,nbdl,ifree,kkk,kk)*wat_flux(:,nbdl,pref_soil_veg(m))*1E-3*flux_red(:)
           ELSEWHERE 
              DOC_FLUX(:,m,nbdl,kkk,kk) = zero
           ENDWHERE
        ENDDO !End loop over soil and litter pool
     ENDDO !End loop over elements

     !! 2.3.7 DOC is reconverted from gC m^⁻3 of water in gC m^-2 of ground to exchange it with the other carbon pools
     !For the first layer

     DO k = 1, npool ! Loop over soil and litter pools
        DO kk = 1,nelements !Loop over elements
           WHERE(soil_mc(:,1,pref_soil_veg(m)) .GT. zero)              
              DOC(:,m,1,ifree,k,kk)=DOC(:,m,1,ifree,k,kk) * soil_mc(:,1,pref_soil_veg(m)) * z_soil(1)
           ELSEWHERE
              DOC(:,m,1,ifree,k,kk)=DOC(:,m,1,ifree,k,kk) 
           ENDWHERE
        ENDDO !End loop over elements
     ENDDO ! End loop over soil and litter pools


     !For the other layers

     DO l= 2, nbdl ! Loop over soil layers
        DO k = 1, npool ! Loop over soil and litter pools
           DO kk = 1,nelements !Loop over elements
              WHERE(soil_mc(:,l,pref_soil_veg(m)) .GT. zero)
                 DOC(:,m,l,ifree,k,kk)=DOC(:,m,l,ifree,k,kk) * soil_mc(:,l,pref_soil_veg(m)) * (z_soil(l)-z_soil(l-1))
              ELSEWHERE
                 DOC(:,m,l,ifree,k,kk)=DOC(:,m,l,ifree,k,kk) 
              ENDWHERE
           ENDDO !End loop over elements
        ENDDO ! End loop over soil and litter pools
     ENDDO ! End loop over soil layers

     ! DOC update
     !For the first layer
     DOC_old(:,m,:,:,:,:)= zero
     DOC_old(:,m,:,:,:,:)=DOC(:,m,:,:,:,:)


     DO kk = 1,nelements !Loop over elements
        DO kkk = 1,npool !Loop over soil and litter pool
              
           WHERE (wat_flux(:,1,pref_soil_veg(m)) .GT. 0.) 
              DOC(:,m,1,ifree,kkk,kk) = DOC_old(:,m,1,ifree,kkk,kk) - MIN(DOC_old(:,m,1,ifree,kkk,kk),DOC_FLUX(:,m,1,kkk,kk)) 
           ELSEWHERE (wat_flux(:,1,pref_soil_veg(m)) .LT. 0.) 
              DOC(:,m,1,ifree,kkk,kk) = DOC_old(:,m,1,ifree,kkk,kk) + MIN(DOC_old(:,m,2,ifree,kkk,kk),DOC_FLUX(:,m,1,kkk,kk))
           ELSEWHERE
              DOC(:,m,1,ifree,kkk,kk) = DOC_old(:,m,1,ifree,kkk,kk)
           ENDWHERE
        ENDDO !End loop over soil and litter pool
     ENDDO !End loop over elements

     ! For the other layers
     DO l = 2,nbdl-1 !Loop over soil layers
        DO kk = 1,nelements !Loop over elements
           DO kkk = 1,npool !Loop over soil and litter pool
              WHERE ((wat_flux(:,l,pref_soil_veg(m)) .GT. 0.) .AND. (wat_flux(:,l-1,pref_soil_veg(m)) .GT. 0.))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old(:,m,l,ifree,kkk,kk) - MIN(DOC_FLUX(:,m,l,kkk,kk),DOC_old(:,m,l,ifree,kkk,kk)) + MIN(DOC_FLUX(:,m,l-1,kkk,kk),DOC_old(:,m,l-1,ifree,kkk,kk))
              ELSEWHERE ((wat_flux(:,l,pref_soil_veg(m)) .LT. 0.) .AND. (wat_flux(:,l-1,pref_soil_veg(m)) .LT. 0.))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old(:,m,l,ifree,kkk,kk) + MIN(DOC_FLUX(:,m,l,kkk,kk),DOC_old(:,m,l+1,ifree,kkk,kk)) - MIN(DOC_FLUX(:,m,l-1,kkk,kk),DOC_old(:,m,l,ifree,kkk,kk))
              ELSEWHERE ((wat_flux(:,l,pref_soil_veg(m)) .GT. 0.) .AND. (wat_flux(:,l-1,pref_soil_veg(m)) .LT. 0.))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old(:,m,l,ifree,kkk,kk) - MIN(DOC_FLUX(:,m,l,kkk,kk),DOC_old(:,m,l,ifree,kkk,kk)) - MIN(DOC_FLUX(:,m,l-1,kkk,kk),DOC_old(:,m,l,ifree,kkk,kk))
              ELSEWHERE ((wat_flux(:,l,pref_soil_veg(m)) .LT. 0.) .AND. (wat_flux(:,l-1,pref_soil_veg(m)) .GT. 0.))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old(:,m,l,ifree,kkk,kk) + MIN(DOC_FLUX(:,m,l,kkk,kk),DOC_old(:,m,l+1,ifree,kkk,kk)) + MIN(DOC_FLUX(:,m,l-1,kkk,kk),DOC_old(:,m,l-1,ifree,kkk,kk))
              ELSEWHERE 
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old(:,m,l,ifree,kkk,kk)
              ENDWHERE
           ENDDO !End loop over soil and litter pool
        ENDDO !End loop over elements
     ENDDO !End loop over soil layers

     !For the last layer
     DO kk = 1,nelements !Loop over elements
        DO kkk = 1,npool !Loop over soil and litter pool
           WHERE (wat_flux(:,nbdl-1,pref_soil_veg(m)) .GT. 0.)
              DOC(:,m,nbdl,ifree,kkk,kk) = DOC_old(:,m,nbdl,ifree,kkk,kk) + MIN(DOC_FLUX(:,m,nbdl-1,kkk,kk),DOC_old(:,m,nbdl-1,ifree,kkk,kk))
           ELSEWHERE (wat_flux(:,nbdl-1,pref_soil_veg(m)) .LT. 0.)
              DOC(:,m,nbdl,ifree,kkk,kk) = DOC_old(:,m,nbdl,ifree,kkk,kk) - MIN(DOC_FLUX(:,m,nbdl-1,kkk,kk),DOC_old(:,m,nbdl,ifree,kkk,kk))
           ELSEWHERE
              DOC(:,m,nbdl,ifree,kkk,kk) = DOC_old(:,m,nbdl,ifree,kkk,kk)
           ENDWHERE
        ENDDO !End loop over soil and litter pool
     ENDDO !End loop over elements

     !! 2.3.8 DOC transport by diffusion
     !! 2.3.8.1 Calculation of the DOC fluxes by diffusion
     ! Here we use the DOC concentration in the soil and not in water to
     ! stabilize the calcul of
     ! water distribution is continuous in the soil column (no dry places)


     DOC_old2(:,m,:,:,:,:)=zero
     DOC_old_buffer(:,m,:,:,:,:)=zero
     DOC_old2(:,m,:,:,:,:)=DOC(:,m,:,:,:,:)

     DO kkk = 1, npool ! Loop over soil and litter pools
        DO kk = 1, nelements ! Loop over elements
               
           DOC_FLUX_DIFF(:,m,1,kkk,kk) = Dif_DOC(:)*ABS(DOC_old2(:,m,1,ifree,kkk,kk)/(z_soil(1))-DOC_old2(:,m,2,ifree,kkk,kk)/(z_soil(2)-z_soil(1)))/(z_soil(2)-z_soil(1))

           DO l= 2, nbdl-1 ! Loop over soil layers
              DOC_FLUX_DIFF(:,m,l,kkk,kk) = Dif_DOC(:)*ABS(DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1))-DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)))/(z_soil(l+1)-z_soil(l))
           ENDDO

       !Below we checked if in case that, in a given layer, you have diffusion
       !in the above and below litters, both fluxes are not higher than the
       !stocks of the given layer.
           DO l =1, nbdl-2
              WHERE (DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .LT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)) .AND. &
                   DOC_old2(:,m,l+2,ifree,kkk,kk)/(z_soil(l+2)-z_soil(l+1)) .LT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)) .AND. &
                   DOC_FLUX_DIFF(:,m,l,kkk,kk) + DOC_FLUX_DIFF(:,m,l+1,kkk,kk) .GT. DOC_old2(:,m,l+1,ifree,kkk,kk))
                 DOC_FLUX_DIFF(:,m,l,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk)*(DOC_FLUX_DIFF(:,m,l,kkk,kk)/(DOC_FLUX_DIFF(:,m,l,kkk,kk)+DOC_FLUX_DIFF(:,m,l+1,kkk,kk)))
                 DOC_FLUX_DIFF(:,m,l+1,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk)*(DOC_FLUX_DIFF(:,m,l+1,kkk,kk)/(DOC_FLUX_DIFF(:,m,l,kkk,kk)+DOC_FLUX_DIFF(:,m,l+1,kkk,kk)))
              ELSEWHERE (DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .GE. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)) .AND. &
                   DOC_old2(:,m,l+2,ifree,kkk,kk)/(z_soil(l+2)-z_soil(l+1)) .LT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)) .AND. &
                   DOC_old2(:,m,l+1,ifree,kkk,kk) + DOC_FLUX_DIFF(:,m,l,kkk,kk) - DOC_FLUX_DIFF(:,m,l+1,kkk,kk) .LE. min_stomate)
                 DOC_FLUX_DIFF(:,m,l,kkk,kk) = DOC_FLUX_DIFF(:,m,l,kkk,kk)
                 DOC_FLUX_DIFF(:,m,l+1,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk) + DOC_FLUX_DIFF(:,m,l,kkk,kk)
              ELSEWHERE (DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .LT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)) .AND. &
                   DOC_old2(:,m,l+2,ifree,kkk,kk)/(z_soil(l+2)-z_soil(l+1)) .GE. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l)) .AND. &
                   DOC_old2(:,m,l+1,ifree,kkk,kk) - DOC_FLUX_DIFF(:,m,l,kkk,kk) + DOC_FLUX_DIFF(:,m,l+1,kkk,kk) .LE. min_stomate)
                 DOC_FLUX_DIFF(:,m,l,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk) + DOC_FLUX_DIFF(:,m,l+1,kkk,kk)
                 DOC_FLUX_DIFF(:,m,l+1,kkk,kk) = DOC_FLUX_DIFF(:,m,l+1,kkk,kk)
              ELSEWHERE
                 DOC_FLUX_DIFF(:,m,l,kkk,kk) = DOC_FLUX_DIFF(:,m,l,kkk,kk)
                 DOC_FLUX_DIFF(:,m,l+1,kkk,kk) = DOC_FLUX_DIFF(:,m,l+1,kkk,kk)
              ENDWHERE
           ENDDO

           DO l= 1, nbdl-1 ! Loop over soil layers
              WHERE ((DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .LT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l))) .AND. &
                   ((DOC_FLUX_DIFF(:,m,l,kkk,kk) - DOC_old2(:,m,l+1,ifree,kkk,kk)) .GE. min_stomate))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old2(:,m,l,ifree,kkk,kk) + DOC_old2(:,m,l+1,ifree,kkk,kk) + DOC_old_buffer(:,m,l,ifree,kkk,kk)
                 DOC_old_buffer(:,m,l+1,ifree,kkk,kk) = - DOC_old2(:,m,l+1,ifree,kkk,kk) 
                 DOC(:,m,l+1,ifree,kkk,kk) = zero
              ELSEWHERE ((DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .LT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l))) .AND. &
                   ((DOC_old2(:,m,l+1,ifree,kkk,kk) - DOC_FLUX_DIFF(:,m,l,kkk,kk)) .GT. min_stomate))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old2(:,m,l,ifree,kkk,kk) + DOC_FLUX_DIFF(:,m,l,kkk,kk) + DOC_old_buffer(:,m,l,ifree,kkk,kk)
                 DOC_old_buffer(:,m,l+1,ifree,kkk,kk) =  - DOC_FLUX_DIFF(:,m,l,kkk,kk)
                 DOC(:,m,l+1,ifree,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk) - DOC_FLUX_DIFF(:,m,l,kkk,kk)
              ELSEWHERE ((DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .GT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l))) .AND. &
                   ((DOC_FLUX_DIFF(:,m,l,kkk,kk) - DOC_old2(:,m,l,ifree,kkk,kk)) .GE. min_stomate))
                 DOC(:,m,l,ifree,kkk,kk) = zero + DOC_old_buffer(:,m,l,ifree,kkk,kk)
                 DOC_old_buffer(:,m,l+1,ifree,kkk,kk) = DOC_old2(:,m,l,ifree,kkk,kk)
                 DOC(:,m,l+1,ifree,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk) + DOC_old2(:,m,l,ifree,kkk,kk)
              ELSEWHERE ((DOC_old2(:,m,l,ifree,kkk,kk)/(z_soil(l)-z_soil(l-1)) .GT. DOC_old2(:,m,l+1,ifree,kkk,kk)/(z_soil(l+1)-z_soil(l))) .AND. &
                   ((DOC_old2(:,m,l,ifree,kkk,kk) - DOC_FLUX_DIFF(:,m,l,kkk,kk)) .GT. min_stomate))
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old2(:,m,l,ifree,kkk,kk) - DOC_FLUX_DIFF(:,m,l,kkk,kk) + DOC_old_buffer(:,m,l,ifree,kkk,kk)
                 DOC_old_buffer(:,m,l+1,ifree,kkk,kk) = DOC_FLUX_DIFF(:,m,l,kkk,kk)
                 DOC(:,m,l+1,ifree,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk) + DOC_FLUX_DIFF(:,m,l,kkk,kk)
              ELSEWHERE
                 DOC(:,m,l,ifree,kkk,kk) = DOC_old2(:,m,l,ifree,kkk,kk)
                 DOC(:,m,l+1,ifree,kkk,kk) = DOC_old2(:,m,l+1,ifree,kkk,kk)
              ENDWHERE
           ENDDO ! End loop over soil layers
        ENDDO ! End loop over elements
     ENDDO ! End loop over DOC pools

        !! 2.3.9 DOC export 
        !! Calculate total DOC going out of the system (the factor 1E-3 is
        !! applied to convert runoff, floodout, drainage from kg/m2 to m3/m2) 
        !! We assume that runoff and floodout affects the first layer of soil
        !! and drainage only the last layer. 
        !! Drainage and runoff are given by soil type by hydrol.f90 but not floodout (it is a weighed mean) explaining why 
        !! the calculation is a bit different between DOC_RUN and DOC_FLOOD. Moreover, floodout is computed as evaporation-precipitation,
        !! so it represent an export to the rivers only if floodout is negative explaing why we have to multiply by -un.
        !! Be careful we do not * by dt because it is already done in hydrol.f90

        !! based on fast reservoir, we calculate a reduction factor. The calculation is based on the assumption that there 
        !! head waters that is the main source of DOC to inland waters, while areas more remote from the head waters have a neglibile 
        !! effect on the DOC in streams. We assume that the export from soils to head waters is proportional to the 
        !! square root of the fast reservoir storage. This is consitnet with empirical finding relating stream width to discharge. As we
        !! do not directly consider a stream surface area, we simply assign a reference fast reservoir for which we assume that
        !! all top soils export DOC into the head waters (which can include streamlets). Note that very high runoff appears where swamps
        !! are pesent.


        fastr_corr(:) = (fastr(:)**undemi)/(fastr_ref**undemi)
        WHERE (fastr_corr(:) .LT. zero)
           fastr_corr(:) = zero
        ELSEWHERE
           fastr_corr(:) = fastr_corr(:)
        ENDWHERE

        !! Based on maximum export concentration, we calculate a correction factor
         soilDOC_corr(:) = zero
         soilDOC_31mm(:) = zero
         DO kk = 1,nelements !Loop over elements
            DO kkk = 1,npool !Loop over soil and litter pool
               soilDOC_31mm(:) = soilDOC_31mm + DOC(:,m,1,ifree,kkk,kk)
               IF (sro_bottom .GT. 1) THEN
                  DO l=2,sro_bottom
                     soilDOC_31mm(:) = soilDOC_31mm(:) + DOC(:,m,l,ifree,kkk,kk)
                  ENDDO !l=2,sro_bottom
               ENDIF !(sro_bottom .GT. 1)
            ENDDO !kkk = 1,npool
         ENDDO !kk = 1,nelements
         !
         WHERE (soilDOC_31mm(:)*flux_red(:)*fastr_corr(:)/soilwater_31mm(:,pref_soil_veg(m)) .GT. DOCexp_max)
            soilDOC_corr(:) = DOCexp_max/(soilDOC_31mm(:)*flux_red(:)*fastr_corr(:)/soilwater_31mm(:,pref_soil_veg(m)))
         ELSEWHERE
            soilDOC_corr(:) = un
         ENDWHERE
         soilDOC_corr(:) = soilDOC_corr(:) * flux_red(:) * fastr_corr(:)
         !
         DO kk = 1,nelements !Loop over elements
            DO kkk = 1,npool !Loop over soil and litter pool
               !DOC lost through runoff for the first layer
               WHERE (soilwater_31mm(:,pref_soil_veg(m)) .GT. zero)
                  DOC_RUN(:,m,kkk,kk) = DOC_RUN(:,m,kkk,kk) + MIN(DOC(:,m,1,ifree,kkk,kk),lat_exp*soilDOC_corr(:) &
                       &              * (runoff_per_soil(:,pref_soil_veg(m))*1E-3/soilwater_31mm(:,pref_soil_veg(m))) &
                       &              * DOC(:,m,1,ifree,kkk,kk))
               ELSEWHERE
                  DOC_RUN(:,m,kkk,kk) = DOC_RUN(:,m,kkk,kk)
               ENDWHERE
               !
               IF (kkk .EQ. imetabo) THEN
                  DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,imetabo,kk) 
                  DOC_RUN(:,m,imetabo,kk) = zero
               ELSEIF (kkk .EQ. istrabo) THEN                  
                  DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,istrabo,kk) * (un - lignin_struc_above(:,m))        
                  DOC_RUN(:,m,islo,kk) = DOC_RUN(:,m,islo,kk) + DOC_RUN(:,m,istrabo,kk) * lignin_struc_above(:,m) 
                  DOC_RUN(:,m,istrabo,kk) = zero
               ELSEIF (kkk .EQ. istrbel) THEN
                  DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,istrbel,kk) * (un - lignin_struc_below(:,m,1))        
                  DOC_RUN(:,m,islo,kk) = DOC_RUN(:,m,islo,kk) + DOC_RUN(:,m,istrbel,kk) * lignin_struc_below(:,m,1) 
                  DOC_RUN(:,m,istrbel,kk) = zero  
               ELSEIF (kkk .EQ. imetbel) THEN              
                  DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,imetbel,kk) 
                  DOC_RUN(:,m,imetbel,kk) = zero
               ELSE
                  DOC_RUN(:,m,kkk,kk) = DOC_RUN(:,m,kkk,kk)
               ENDIF
               !
               !And for the layers 2-? layers
               IF (sro_bottom .GT. 1) THEN
                  DO l=2,sro_bottom
                     WHERE (soilwater_31mm(:,pref_soil_veg(m)) .GT. zero)
                        DOC_RUN(:,m,kkk,kk) = DOC_RUN(:,m,kkk,kk) + MIN(DOC(:,m,l,ifree,kkk,kk), &
                             &      lat_exp * soilDOC_corr(:) * (runoff_per_soil(:,pref_soil_veg(m))*1E-3/soilwater_31mm(:,pref_soil_veg(m))) &
                             &    * DOC(:,m,l,ifree,kkk,kk))
                     ELSEWHERE
                        DOC_RUN(:,m,kkk,kk) = DOC_RUN(:,m,kkk,kk)
                     ENDWHERE
                     !
                     IF (kkk .EQ. imetabo) THEN
                        DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,imetabo,kk) 
                        DOC_RUN(:,m,imetabo,kk) = zero
                     ELSEIF (kkk .EQ. istrabo) THEN                  
                        DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,istrabo,kk) * (un - lignin_struc_above(:,m))        
                        DOC_RUN(:,m,islo,kk) = DOC_RUN(:,m,islo,kk) + DOC_RUN(:,m,istrabo,kk) * lignin_struc_above(:,m) 
                        DOC_RUN(:,m,istrabo,kk) = zero
                     ELSEIF (kkk .EQ. istrbel) THEN
                        DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,istrbel,kk) * (un - lignin_struc_below(:,m,l))        
                        DOC_RUN(:,m,islo,kk) = DOC_RUN(:,m,islo,kk) + DOC_RUN(:,m,istrbel,kk) * lignin_struc_below(:,m,l) 
                        DOC_RUN(:,m,istrbel,kk) = zero  
                     ELSEIF (kkk .EQ. imetbel) THEN              
                        DOC_RUN(:,m,iact,kk) = DOC_RUN(:,m,iact,kk) + DOC_RUN(:,m,imetbel,kk) 
                        DOC_RUN(:,m,imetbel,kk) = zero
                     ELSE
                        DOC_RUN(:,m,kkk,kk) = DOC_RUN(:,m,kkk,kk)
                     ENDIF
                  ENDDO
               ENDIF !sro_bottom .GT. 1
                     !
               WHERE (soil_mc(:,nbdl,pref_soil_veg(m)) .GT. zero)
                  DOC_DRAIN(:,m,kkk,kk) = DOC_DRAIN(:,m,kkk,kk) + MIN(DOC(:,m,nbdl,ifree,kkk,kk),&
                       &                                         lat_exp*DOC(:,m,nbdl,ifree,kkk,kk)*drainage_per_soil(:,pref_soil_veg(m))*1E-3 &
                       &                                         /((z_soil(nbdl)-z_soil(nbdl-1)) * soil_mc(:,nbdl,pref_soil_veg(m))))
               ELSEWHERE
                  DOC_DRAIN(:,m,kkk,kk) = DOC_DRAIN(:,m,kkk,kk)
               ENDWHERE
               !
               IF (kkk .EQ. imetabo) THEN
                  DOC_DRAIN(:,m,iact,kk) = DOC_DRAIN(:,m,iact,kk) + DOC_DRAIN(:,m,imetabo,kk) 
                  DOC_DRAIN(:,m,imetabo,kk) = zero
               ELSEIF (kkk .EQ. istrabo) THEN                  
                  DOC_DRAIN(:,m,iact,kk) = DOC_DRAIN(:,m,iact,kk) + DOC_DRAIN(:,m,istrabo,kk) * (un - lignin_struc_above(:,m))        
                  DOC_DRAIN(:,m,islo,kk) = DOC_DRAIN(:,m,islo,kk) + DOC_DRAIN(:,m,istrabo,kk) * lignin_struc_above(:,m) 
                  DOC_DRAIN(:,m,istrabo,kk) = zero
               ELSEIF (kkk .EQ. istrbel) THEN
                  DOC_DRAIN(:,m,iact,kk) = DOC_DRAIN(:,m,iact,kk) + DOC_DRAIN(:,m,istrbel,kk) * (un - lignin_struc_below(:,m,nbdl))        
                  DOC_DRAIN(:,m,islo,kk) = DOC_DRAIN(:,m,islo,kk) + DOC_DRAIN(:,m,istrbel,kk) * lignin_struc_below(:,m,nbdl) 
                  DOC_DRAIN(:,m,istrbel,kk) = zero  
               ELSEIF (kkk .EQ. imetbel) THEN              
                  DOC_DRAIN(:,m,iact,kk) = DOC_DRAIN(:,m,iact,kk) + DOC_DRAIN(:,m,imetbel,kk) 
                  DOC_DRAIN(:,m,imetbel,kk) = zero
               ELSE
                  DOC_DRAIN(:,m,kkk,kk) = DOC_DRAIN(:,m,kkk,kk)
               ENDIF
               !
            ENDDO !End loop over soil and litter pool          
            !! TF-DOC
            WHERE (veget_max(:,m) .GT. zero)
               DOC_FLOOD(:,m,iact,kk) = DOC_FLOOD(:,m,iact,kk) + wet_dep_flood(:,m,kk)*undemi/veget_max(:,m)
               DOC_FLOOD(:,m,islo,kk) = DOC_FLOOD(:,m,islo,kk) + wet_dep_flood(:,m,kk)*undemi/veget_max(:,m)
               DOC_FLOOD(:,m,iact,kk) = DOC_FLOOD(:,m,iact,kk) + floodcarbon_input(:,m,iact,kk)*dt
               DOC_FLOOD(:,m,islo,kk) = DOC_FLOOD(:,m,islo,kk) + floodcarbon_input(:,m,islo,kk)*dt
            ELSEWHERE
               DOC_FLOOD(:,m,iact,kk) = DOC_FLOOD(:,m,iact,kk)
            ENDWHERE
            !
            DO l=1,sro_bottom
               WHERE (veget_max(:,m) .GT. zero)
                  DOC_FLOOD(:,m,islo,kk) = DOC_FLOOD(:,m,islo,kk) + CUE_coef(:) * fluxtot_flood(:,islow,kk,l)
                  DOC_FLOOD(:,m,ipas,kk) = DOC_FLOOD(:,m,ipas,kk) + CUE_coef(:) * fluxtot_flood(:,ipassive,kk,l)
                  DOC_FLOOD(:,m,iact,kk) = DOC_FLOOD(:,m,iact,kk) + CUE_coef(:) * fluxtot_flood(:,iactive,kk,l)
               ELSEWHERE
                  DOC_FLOOD(:,m,iact,kk) = DOC_FLOOD(:,m,iact,kk)
               ENDWHERE
            ENDDO !l=1,sro_bottom
            !
         ENDDO !End loop over elements


         DO kk = 1,nelements !Loop over elements
            DOC_EXP(:,m,irunoff,:,kk) = DOC_RUN(:,m,:,kk)/dt
            DOC_EXP(:,m,iflooded,:,kk) = DOC_FLOOD(:,m,:,kk)/dt
            DOC_EXP(:,m,idrainage,:,kk) = DOC_DRAIN(:,m,:,kk)/dt
         ENDDO !End loop over elements
         ! We substract runoff, floodout and drainage from DOC pools
         DO kkk = 1,npool !Loop over soil and litter pool
            DO kk = 1, nelements
               
               !For the first layer
               WHERE (soilwater_31mm(:,pref_soil_veg(m)) .GT. zero)
                  DOC(:,m,1,ifree,kkk,kk) = DOC(:,m,1,ifree,kkk,kk) - MIN(DOC(:,m,1,ifree,kkk,kk), &
                       &                    lat_exp * soilDOC_corr(:) * (runoff_per_soil(:,pref_soil_veg(m))*1E-3/soilwater_31mm(:,pref_soil_veg(m))) &
                       &                  * DOC(:,m,1,ifree,kkk,kk))
               ELSEWHERE
                  DOC(:,m,1,ifree,kkk,kk) = DOC(:,m,1,ifree,kkk,kk)
               ENDWHERE
               !And for the layers 2-5
               IF (sro_bottom .GT. 1) THEN
                  DO l=2,sro_bottom
                     WHERE (soilwater_31mm(:,pref_soil_veg(m)) .GT. zero)
                        DOC(:,m,l,ifree,kkk,kk) = DOC(:,m,l,ifree,kkk,kk) - MIN(DOC(:,m,l,ifree,kkk,kk), &
                             &                    lat_exp * soilDOC_corr(:) * (runoff_per_soil(:,pref_soil_veg(m))*1E-3/soilwater_31mm(:,pref_soil_veg(m))) &
                             &                  * DOC(:,m,l,ifree,kkk,kk))
                     ELSEWHERE
                        DOC(:,m,l,ifree,kkk,kk) = DOC(:,m,l,ifree,kkk,kk)
                     ENDWHERE
                  ENDDO
               ENDIF !sro_bottom .GT. 1
               !For the last layer
               WHERE (soil_mc(:,nbdl,pref_soil_veg(m)) .GT. zero)
                  DOC(:,m,nbdl,ifree,kkk,kk) = DOC(:,m,nbdl,ifree,kkk,kk) - &
                       &                       MIN(DOC(:,m,nbdl,ifree,kkk,kk),&
                       &                       lat_exp * DOC(:,m,nbdl,ifree,kkk,kk)*drainage_per_soil(:,pref_soil_veg(m))*1E-3 &
                       &                       /((z_soil(nbdl)-z_soil(nbdl-1)) * soil_mc(:,nbdl,pref_soil_veg(m))))
               ELSEWHERE
                  DOC(:,m,nbdl,ifree,kkk,kk) = DOC(:,m,nbdl,ifree,kkk,kk)
               ENDWHERE
            ENDDO ! End loop over elements
         ENDDO !End loop over soil and litter pool

      !! 2.3.10 Transport of particulate organic C (i.e. active, slow and passive
      !! pools) following the second Fick's Law (O¿Brien and Stout, 1978; Wynn
      !! et al., 2005). Represent the bioturbation and is generally associated
      !! with transport  plant debris (Elzein and Balesdent, 1995; Bruun et al., 
      !! 2007; Braakhekke et al., 2011, Guenet et al., 2013) 

      ! Definition of the diffusion rate from Guenet et al., (2013) BG for the
      ! moment but must depends on clay, ph organic C, or whatever...
      

carbon_old(:,:,m,:)=zero
carbon_old_buffer(:,:,m,:)=zero
carbon_old(:,:,m,:)=carbon(:,:,m,:)

     DO kk = 1, ncarb ! Loop over soil and litter pools

        carbon_flux(:,kk,m,1) = Dif_coef(:)*ABS(carbon_old(:,kk,m,1)/(z_soil(1))-carbon_old(:,kk,m,2)/(z_soil(2)-z_soil(1)))/(z_soil(2)-z_soil(1))

      DO l= 2, nbdl-1 ! Loop over soil layers
        carbon_flux(:,kk,m,l) = Dif_coef(:)*ABS(carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1))-carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)))/(z_soil(l+1)-z_soil(l))
      ENDDO
        carbon_flux_old(:,kk,m,:) = carbon_flux(:,kk,m,:)

       !Below we checked if in case that, in a given layer, you have diffusion
       !in the above and below litters, both fluxes are not higher than the
       !stocks of the given layer.

      DO l =1, nbdl-2
           WHERE (carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .LT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)) .AND. &
                  carbon_old(:,kk,m,l+2)/(z_soil(l+2)-z_soil(l+1)) .LT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)) .AND. &
                  carbon_flux(:,kk,m,l)+carbon_flux(:,kk,m,l+1) .GT. carbon_old(:,kk,m,l+1))
            carbon_flux(:,kk,m,l) = carbon_old(:,kk,m,l+1)*(carbon_flux(:,kk,m,l)/(carbon_flux(:,kk,m,l)+carbon_flux(:,kk,m,l+1)))
            carbon_flux(:,kk,m,l+1) = carbon_old(:,kk,m,l+1)*(carbon_flux(:,kk,m,l+1)/(carbon_flux(:,kk,m,l)+carbon_flux(:,kk,m,l+1)))
           ELSEWHERE (carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .GE. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)) .AND. &
                      carbon_old(:,kk,m,l+2)/(z_soil(l+2)-z_soil(l+1)) .LT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)) .AND. &
                      carbon_old(:,kk,m,l+1) + carbon_flux(:,kk,m,l) - carbon_flux(:,kk,m,l+1) .LE. min_stomate)
            carbon_flux(:,kk,m,l) = carbon_flux(:,kk,m,l)
            carbon_flux(:,kk,m,l+1) = carbon_old(:,kk,m,l+1) + carbon_flux(:,kk,m,l)
           ELSEWHERE (carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .LT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)) .AND. &
                      carbon_old(:,kk,m,l+2)/(z_soil(l+2)-z_soil(l+1)) .GE. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l)) .AND. &
                      carbon_old(:,kk,m,l+1) - carbon_flux(:,kk,m,l) + carbon_flux(:,kk,m,l+1) .LE. min_stomate)
            carbon_flux(:,kk,m,l) = carbon_old(:,kk,m,l+1) + carbon_flux(:,kk,m,l+1)
            carbon_flux(:,kk,m,l+1) = carbon_flux(:,kk,m,l+1)
           ELSEWHERE
            carbon_flux(:,kk,m,l) = carbon_flux(:,kk,m,l)
            carbon_flux(:,kk,m,l+1) = carbon_flux(:,kk,m,l+1)
           ENDWHERE
      ENDDO

      DO l= 1, nbdl-1 ! Loop over soil layers
           WHERE ((carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .LT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l))) .AND. &
                 ((carbon_flux(:,kk,m,l) - carbon_old(:,kk,m,l+1)) .GE. min_stomate))
            carbon(:,kk,m,l) = carbon_old(:,kk,m,l) + carbon_old(:,kk,m,l+1) + carbon_old_buffer(:,kk,m,l)
            carbon_old_buffer(:,kk,m,l+1) = -carbon_old(:,kk,m,l+1) 
            carbon(:,kk,m,l+1) = zero
           ELSEWHERE ((carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .LT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l))) .AND. &
                  ((carbon_old(:,kk,m,l+1) - carbon_flux(:,kk,m,l)) .GT. min_stomate))
            carbon(:,kk,m,l) = carbon_old(:,kk,m,l) + carbon_flux(:,kk,m,l) + carbon_old_buffer(:,kk,m,l)
            carbon_old_buffer(:,kk,m,l+1) =  - carbon_flux(:,kk,m,l)
            carbon(:,kk,m,l+1) = carbon_old(:,kk,m,l+1) - carbon_flux(:,kk,m,l)
           ELSEWHERE ((carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .GT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l))) .AND. &
                  ((carbon_flux(:,kk,m,l) - carbon_old(:,kk,m,l)) .GE. min_stomate))
            carbon(:,kk,m,l) =zero + carbon_old_buffer(:,kk,m,l)
            carbon_old_buffer(:,kk,m,l+1) = carbon_old(:,kk,m,l)
            carbon(:,kk,m,l+1) = carbon_old(:,kk,m,l+1) + carbon_old(:,kk,m,l)
           ELSEWHERE ((carbon_old(:,kk,m,l)/(z_soil(l)-z_soil(l-1)) .GT. carbon_old(:,kk,m,l+1)/(z_soil(l+1)-z_soil(l))) .AND. &
                  ((carbon_old(:,kk,m,l) - carbon_flux(:,kk,m,l)) .GT. min_stomate))
            carbon(:,kk,m,l) = carbon_old(:,kk,m,l) - carbon_flux(:,kk,m,l) + carbon_old_buffer(:,kk,m,l)
            carbon_old_buffer(:,kk,m,l+1) = carbon_flux(:,kk,m,l)
            carbon(:,kk,m,l+1) = carbon_old(:,kk,m,l+1) + carbon_flux(:,kk,m,l)
           ELSEWHERE
            carbon(:,kk,m,l) = carbon_old(:,kk,m,l)
            carbon(:,kk,m,l+1) = carbon_old(:,kk,m,l+1)
           ENDWHERE
        ENDDO ! End loop over soil layers



      ENDDO ! End loop over carbon pools
  ENDDO ! End loop over PFTs

!! 3. Check mass balance closure
IF (ld_doc) THEN    
    !! 3.1 Calculate components of the mass balance
    pool_end(:,:,:) = zero 

!Ending carbon pool
    DO m = 1, nvm
       DO i = 1,ncarb
          DO l = 1, nbdl
                 pool_end(:,m,icarbon) = pool_end(:,m,icarbon) + &
                           ( carbon(:,i,m,l)* veget_max(:,m) ) 
          ENDDO
       ENDDO
    ENDDO

!Ending DOC pool
    DO m = 1, nvm
       DO i = 1,npool
          DO l = 1, nbdl
             DO j = 1, ndoc
                    pool_end(:,m,icarbon) = pool_end(:,m,icarbon) + &
                              ( DOC(:,m,l,j,i,icarbon) * veget_max(:,m)) 
             ENDDO
          ENDDO
       ENDDO
       pool_end(:,m,icarbon) = pool_end(:,m,icarbon) +  interception_storage(:,m,icarbon)
    ENDDO

!WRITE(numout,*) " pool_start(:,m,icarbon) soil=",  pool_start(:,4,icarbon)
!WRITE(numout,*) " pool_end(:,m,icarbon) soil=",  pool_end(:,4,icarbon)

 !! 3.2 Calculate mass balance
    !  Note that soilcarbon is transfered to other pools but that the pool
    !  was not updated. We should not account for it in ::pool_end
    check_intern(:,:,:,:) = zero
    check_intern(:,:,iatm2land,icarbon) = zero

    
    check_intern(:,:,iland2atm,icarbon) = -un * (resp_hetero_soil(:,:)+resp_flood_soil(:,:)) * &
         veget_max(:,:) * dt 
    DO i=1,npool
       check_intern(:,:,ilat2out,icarbon) = check_intern(:,:,ilat2out,icarbon) - un * (DOC_EXP(:,:,irunoff,i,icarbon)+ &
            &         DOC_EXP(:,:,iflooded,i,icarbon)+ DOC_EXP(:,:,idrainage,i,icarbon)) * veget_max(:,:) * dt
    ENDDO !i=1,npool 
    check_intern(:,:,ilat2in,icarbon) = un * zero
    check_intern(:,:,ipoolchange,icarbon) = -un * (pool_end(:,:,icarbon) - &
         pool_start(:,:,icarbon))
    

    closure_intern = zero

    DO i = 1,nmbcomp
       closure_intern(:,:,icarbon) = closure_intern(:,:,icarbon) + &
           check_intern(:,:,i,icarbon)
    ENDDO

!! 3.3 Write verdict
    DO i = 1,npts
       IF (SUM(closure_intern(i,:,icarbon)) .GT. min_stomate .OR. &
            SUM(closure_intern(i,:,icarbon)) .LT. -min_stomate) THEN
          WRITE(numout,*) 'Error: mass balance is not closed in stomate_soilcarbon.f90 at pixel ', i
          WRITE(numout,*) '   Difference is, ', SUM(closure_intern(i,:,icarbon))
          WRITE(numout,*) '   Vegetation is, ', veget_max(i,:)

          WRITE(numout,*) '   deposition, ', SUM(wet_dep_ground(i,:,icarbon))
          WRITE(numout,*) '   deposition flood, ', SUM(wet_dep_flood(i,:,icarbon))

          WRITE(numout,*) '   soilcarbon_input, ', SUM(SUM(soilcarbon_input(i,:,:,:,icarbon),DIM=2),DIM=2)

          WRITE(numout,*) '   DOC_EXP, ',  SUM(SUM(DOC_EXP(i,:,:,:,icarbon),DIM=2),DIM=2)
          WRITE(numout,*) '   FLOOD_carbon_input, ', SUM(floodcarbon_input(i,:,:,icarbon),DIM=2)

          WRITE(numout,*) '   resp_hetero_soil(:,:), ', SUM(resp_hetero_soil(i,:))
          WRITE(numout,*) '   resp_flood_soil(:,:), ', SUM(resp_flood_soil(i,:))
       ENDIF
    ENDDO !i = 1,npts
ENDIF

!! 4 Writing history files

    CALL xios_orchidee_send_field("dry_dep_canopy",dry_dep_canopy(:,:,icarbon)/dt)
    CALL xios_orchidee_send_field("DOC_precip2canopy",DOC_precip2canopy(:,:,icarbon)/dt)
    CALL xios_orchidee_send_field("DOC_precip2ground",DOC_precip2ground(:,:,icarbon)/dt)
    CALL xios_orchidee_send_field("DOC_canopy2ground",DOC_canopy2ground(:,:,icarbon)/dt)
!    CALL xios_orchidee_send_field("EXP_DOC_RUNOFF_ref",DOC_EXP(:,:,irunoff,islo,kk)+DOC_EXP(:,:,irunoff,ipas,kk))
!    CALL xios_orchidee_send_field("EXP_DOC_RUNOFF_lab",DOC_EXP(:,:,irunoff,iact,kk))
!    CALL xios_orchidee_send_field("EXP_DOC_DRAIN_ref",DOC_EXP(:,:,idrainage,islo,kk)+DOC_EXP(:,:,idrainage,ipas,kk))
!    CALL xios_orchidee_send_field("EXP_DOC_DRAIN_lab",DOC_EXP(:,:,idrainage,iact,kk))
!    CALL xios_orchidee_send_field("EXP_DOC_FLOOD_ref",DOC_EXP(:,:,iflooded,islo,kk)+DOC_EXP(:,:,iflooded,ipas,kk))
!    CALL xios_orchidee_send_field("EXP_DOC_FLOOD_lab",DOC_EXP(:,:,iflooded,iact,kk))
    CALL xios_orchidee_send_field("interception_storage",interception_storage(:,:,icarbon))
    CALL xios_orchidee_send_field("precip2canopy",precip2canopy(:,:)/dt)
    CALL xios_orchidee_send_field("precip2ground",precip2ground(:,:)/dt)
    CALL xios_orchidee_send_field("canopy2ground",canopy2ground(:,:)/dt)
    CALL xios_orchidee_send_field("runoff_forest",runoff_per_soil(:,2)/dt)
    CALL xios_orchidee_send_field("CLAY",clay(:))
    CALL xios_orchidee_send_field("SOIL_pH",soil_pH(:))
    CALL xios_orchidee_send_field("KD",Kd(:,:))
    CALL xios_orchidee_send_field("POOR_SOILS",poor_soils(:))

    IF (printlev>=4) WRITE(numout,*) 'Leaving soilcarbon'
    
  END SUBROUTINE soilcarbon

END MODULE stomate_soilcarbon
