! =================================================================================================================================
! MODULE       : stomate_litter
!
! CONTACT      : orchidee-help _at_ ipsl.jussieu.fr
!
! LICENCE      : IPSL (2006)
! This software is governed by the CeCILL licence see ORCHIDEE/ORCHIDEE_CeCILL.LIC
!
!>\BRIEF       Update litter and lignine content after litter fall and 
!! calculating litter decomposition.      
!!
!!\n DESCRIPTION: None
!!
!! RECENT CHANGE(S): None
!!
!! REFERENCE(S)	: None
!!
!! SVN          :
!! $HeadURL: svn://forge.ipsl.jussieu.fr/orchidee/perso/ronny.lauerwald/ORCHIDEE/src_stomate/stomate_litter.f90 $
!! $Date: 2016-11-14 12:56:44 +0100 (Mon, 14 Nov 2016) $
!! $Revision: 3901 $
!! \n
!_ ================================================================================================================================

MODULE stomate_litter

  ! modules used:

  USE ioipsl_para
  USE stomate_data
  USE constantes
  USE constantes_soil
  USE pft_parameters

  IMPLICIT NONE

  ! private & public routines

  PRIVATE
  PUBLIC littercalc,littercalc_clear, deadleaf



  LOGICAL, SAVE                        :: firstcall = .TRUE.       !! first call
!$OMP THREADPRIVATE(firstcall)

CONTAINS

!! ================================================================================================================================
!! SUBROUTINE   : littercalc_calc
!!
!!\BRIEF        Set the flag ::firstcall to .TRUE. and as such activate section
!! 1.1 of the subroutine littercalc (see below).
!!
!! DESCRIPTION  : None
!!
!! RECENT CHANGE(S) : None
!!
!! MAIN OUTPUT VARIABLE(S) : None
!!
!! REFERENCE(S) : None
!!
!! FLOWCHART : None
!! \n
!_ ================================================================================================================================

  SUBROUTINE littercalc_clear
    firstcall =.TRUE.
  END SUBROUTINE littercalc_clear


!! ================================================================================================================================
!! SUBROUTINE   : littercalc
!!
!!\BRIEF        Calculation of the litter decomposition and therefore of the 
!! heterotrophic respiration from litter following Parton et al. (1987).
!!
!! DESCRIPTION  : The littercal routine splits the litter in 4 pools: 
!! aboveground metaboblic, aboveground structural, belowground metabolic and 
!! belowground structural. the fraction (F) of plant material going to metabolic 
!! and structural is defined following Parton et al. (1987). The below ground
!! litter is discretized over 11 layers (the same used by the CWRR hydrology
!! scheme).
!! \latexonly
!! \input{littercalc1.tex}
!! \endlatexonly
!! \n
!! where L is the lignin content of the plant carbon pools considered and CN 
!! its CN ratio. L and CN are fixed parameters for each plant carbon pools,
!! therefore it is the ratio between each plant carbon pool within a PFT, which
!! controlled the part of the total litter, that will be considered as
!! recalcitrant (i.e. structural litter) or labile (i.e. metabolic litter).\n  
!! 
!! The routine calculates the fraction of aboveground litter which is metabolic
!! or structural (the litterpart variable) which is then used in lpj_fire.f90.\n 
!! 
!! In the section 2, the routine calculate the new plant material entering the
!! litter pools by phenological death of plants organs (corresponding to the
!! variable turnover) and by fire, herbivory and others non phenological causes
!! (variable bm_to_litter). This calculation is first done for each PFT and then
!! the values calculated for each PFT are added up. Following the same approach
!! the lignin content of the total structural litter is calculated and will be
!! then used as a factor control of the decomposition of the structural litter
!! (lignin_struc) in the section 5.1.2. A test is performed to avoid that we add
!! more lignin than structural litter. Finally, the variable litterpart is
!! updated.\n
!! 
!! In the section 3 and 4 the temperature and the moisture controlling the
!! decomposition are calculated for above and belowground. For aboveground
!! litter, air temperature and litter moisture are calculated in sechiba and used 
!! directly. For belowground, soil temperature and moisture are also calculated 
!! in sechiba but are modulated as a function of the soil depth. The modulation 
!! is a multiplying factor exponentially distributed between 0 (in depth) and 1
!! in surface.\n  
!! 
!! Then, in the section 5 and 6, the routine calculates the structural litter decomposition 
!! (C) following first order kinetics following Parton et al. (1987).
!! \latexonly
!! \input{littercalc2.tex}
!! \endlatexonly
!! \n
!! with k the decomposition rate of the structural litter. 
!! k corresponds to
!! \latexonly
!! \input{littercalc3.tex}
!! \endlatexonly
!! \n
!! with littertau the turnover rate, T a function of the temperature and M a function of
!! the moisture described below.\n
!!  
!! Then, the fraction of dead leaves (DL) composed by aboveground structural litter is
!! calculated as following
!! \latexonly
!! \input{littercalc4.tex}
!! \endlatexonly
!! \n
!! with k the decomposition rate of the structural litter previously
!! described.\n
!!
!! In the section 5.1 and 6.1, the fraction of decomposed structural litter
!! incorporated to the soil (Input) and its associated heterotrophic respiration are
!! calculated. For structural litter, the C decomposed could go in the active
!! soil carbon pool or in the slow carbon, as described in 
!! stomate_soilcarbon.f90.\n
!! \latexonly
!! \input{littercalc5.tex}
!! \endlatexonly
!! \n
!! with f a parameter describing the fraction of structural litter incorporated
!! into the considered soil carbon pool, C the amount of litter decomposed and L 
!! the amount of lignin in the litter. The litter decomposed which is not
!! incorporated into the soil is respired.\n
!!
!! In the section 5.2 and 6.2, the fraction of decomposed metabolic litter
!! incorporated to the soil and its associated heterotrophic respiration are
!! calculated with the same approaches presented for 5.1 but no control factor
!! depending on the lignin content are used.\n
!! 
!! In the section 7 the dead leaf cover is calculated through a call to the 
!! deadleaf subroutine presented below.\n
!!
!! MAIN OUTPUT VARIABLES: ::deadleaf_cover, ::resp_hetero_litter, 
!! ::control_temp, ::control_moist
!!
!! REFERENCES:
!! - Parton, WJ, Schimel, DS, Cole, CV, and Ojima, DS. 1987. Analysis
!! of factors controlling soil organic matter levels in Great Plains
!! grasslands. Soil Science Society of America journal (USA)
!! (51):1173-1179.
!! - Lardy, R, et al., A new method to determine soil organic carbon equilibrium,
!! Environmental Modelling & Software (2011), doi:10.1016|j.envsoft.2011.05.016
!!
!! FLOWCHART    :
!! \latexonly
!! \includegraphics(scale=0.5){littercalcflow.jpg}
!! \endlatexonly
!! \n
!_ ================================================================================================================================

  SUBROUTINE littercalc (npts, dt, &
       turnover, bm_to_litter, &
       veget_max, tsurf, tsoil, soilhum, litterhum, rprof, &
       litterpart, litter_above, litter_below, dead_leaves, &
       lignin_struc_above, lignin_struc_below, &
       deadleaf_cover, resp_hetero_litter, resp_hetero_flood, &
       control_temp_above, control_temp_soil, &
       control_moist_above, control_moist_soil, &
       litter_mc,soilcarbon_input, floodcarbon_input, soil_mc, soiltile, &
       clay, bulk_dens, soil_ph, poor_soils, carbon, flood_frac)

    !! 0. Variable and parameter declaration
    
    !! 0.1 Input variables

    INTEGER(i_std), INTENT(in)                                  :: npts                !! Domain size - number of grid pixels
    REAL(r_std), INTENT(in)                                     :: dt                  !! Time step of the simulations for stomate
                                                                                       !! @tex $(dtradia one\_day^{-1})$ @endtex 
    REAL(r_std), DIMENSION(npts,nvm,nparts,nelements), INTENT(in) :: turnover          !! Turnover rates of plant biomass 
                                                                                       !! @tex $(gC m^{-2} dt\_slow^{-1})$ @endtex 
    REAL(r_std), DIMENSION(npts,nvm,nparts,nelements), INTENT(in) :: bm_to_litter      !! Conversion of biomass to litter 
                                                                                       !! @tex $(gC m^{-2} dt\_slow^{-1})$ @endtex 
    REAL(r_std),DIMENSION(npts,nvm),INTENT(in)                   :: veget_max          !! PFT "Maximal" coverage fraction of a PFT 
                                                                                       !! defined in the input vegetation map 
                                                                                       !! @tex $(m^2 m^{-2})$ @endtex 
    REAL(r_std), DIMENSION(npts), INTENT(in)                     :: tsurf              !! Temperature (K) at the surface
    REAL(r_std), DIMENSION(npts,nbdl), INTENT(in)                :: tsoil              !! Soil temperature (K)
    REAL(r_std), DIMENSION(npts,nbdl), INTENT(in)                :: soilhum            !! Daily soil humidity of each soil layer 
                                                                                       !! (unitless)
    REAL(r_std), DIMENSION(npts), INTENT(in)                     :: litterhum          !! Daily litter humidity (unitless)
    REAL(r_std), DIMENSION(npts,nvm), INTENT(in)                 :: rprof              !! PFT root depth as calculated in stomate.f90 from parameter 
                                                                                       !! humcste which is root profile for different PFTs 
                                                                                       !! in slowproc.f90 (m)
    REAL(r_std),DIMENSION (npts,nstm), INTENT(in)                :: litter_mc          !! soil moisture content \f($m^3 \times m^3$)\f
    REAL(r_std),DIMENSION (npts,nbdl,nstm), INTENT(in)           :: soil_mc            !! soil moisture content \f($m^3 \times m^3$)\f
    REAL(r_std),DIMENSION (npts,nstm), INTENT (in)               :: soiltile           !! Fraction of each soil tile (0-1, unitless)
    REAL(r_std), DIMENSION(npts), INTENT(in)                     :: clay               !! Clay fraction (unitless, 0-1) 
    REAL(r_std), DIMENSION(npts), INTENT(inout)                  :: bulk_dens          !! Variable local of bulk density for the moment must change in the futur (kg m-3)

    REAL(r_std), DIMENSION(npts), INTENT(in)                     :: soil_ph            !! soil pH (pH unit, 0-14)
    REAL(r_std), DIMENSION(npts), INTENT(in)                     :: poor_soils         !! Fraction of poor soils (0-1)
    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl), INTENT(in)      :: carbon             !! Soil carbon pools: active, slow, or passive, \f$(gC m^{2})$\f
    REAL(r_std), DIMENSION(npts), INTENT(in)                     :: flood_frac         !! flooded fraction, needed to calculate heterotrophic respiration input to floodplain
    !! 0.2 Output variables
    
    REAL(r_std), DIMENSION(npts), INTENT(out)                    :: deadleaf_cover     !! Fraction of soil covered by dead leaves 
                                                                                       !! over all PFTs (0-1, unitless)
    REAL(r_std), DIMENSION(npts,nvm,nlevs), INTENT(out)          :: resp_hetero_litter !! Litter heterotrophic respiration. The unit
                                                                                       !! is given by m^2 of ground.  
                                                                                       !! @tex $(gC dtradia one\_day^{-1}) m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm), INTENT(out)                :: resp_hetero_flood  !! Litter heterotrophic respiration in flooded areas.
                                                                                       !! The unit is given by m^2 of ground.
                                                                                       !! @tex $(gC dtradia one\_day^{-1}) m^{-2})$ @endtex 
    REAL(r_std), DIMENSION(npts,nbdl,npool*2), INTENT(out)       :: control_temp_soil  !! Temperature control of heterotrophic 
                                                                                       !! respiration belowground,(0-1, unitless)
    REAL(r_std), DIMENSION(npts,nbdl,nvm), INTENT(out)           :: control_moist_soil !! Moisture control of heterotrophic 
                                                                                       !! respiration aboveground(0.25-1,unitless)
    REAL(r_std), DIMENSION(npts,nlitt), INTENT(out)              :: control_temp_above !! Temperature control of heterotrophic  
                                                                                       !! respiration, above (0-1,  
                                                                                       !! unitless)
    REAL(r_std), DIMENSION(npts,nvm), INTENT(out)                :: control_moist_above!! Moisture control of heterotrophic  
                                                                                       !! respiration aboveground(0.25-1, unitless)

    !! 0.3 Modified variables
    
    REAL(r_std), DIMENSION(npts,nvm,nlitt), INTENT(inout)       :: litterpart         !! Fraction of litter above the ground 
                                                                                      !! belonging to the different PFTs (0-1, 
                                                                                      !! unitless)
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nelements), INTENT(inout) :: litter_above   !! Metabolic and structural litter,above ground
                                                                                      !! The unit is given by m^2 of 
                                                                                      !! ground @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements), INTENT(inout) ::litter_below  !! Metabolic and structural litter, below ground
                                                                                      !! The unit is given by m^2 of
                                                                                      !! ground @tex $(gC m^{-2} $ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nlitt), INTENT(inout)       :: dead_leaves        !! Dead leaves per ground unit area, per PFT,
                                                                                      !! metabolic and structural in 
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm), INTENT(inout)            :: lignin_struc_above  !! Ratio Lignin content in structural litter,
                                                                                      !! above ground, (0-1, unitless)
    REAL(r_std), DIMENSION(npts,nvm,nbdl), INTENT(inout)       :: lignin_struc_below  !! Ratio Lignin content in structural litter,
                                                                                      !! below ground, (0-1, unitless)
    REAL(r_std), DIMENSION(npts,nvm,nbdl,npool,nelements), INTENT(inout) :: soilcarbon_input !! Dissolved Organic Carbon input from litter decomposition
                                                                                      !! The unit is given by m^2
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,npool,nelements), INTENT(inout) :: floodcarbon_input !! Dissolved Organic Carbon input from litter
                                                                                      !! decomposition in flooded areas
                                                                                      !! The unit is given by m^2 
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    !! 0.4 Local variables

    REAL(r_std), DIMENSION(npts,nvm)                            :: control_flood_above!! Moisture control of heterotrophic
                                                                                      !! respiration aboveground(0.25-1, unitless)
!$OMP THREADPRIVATE(control_flood_above) 
    REAL(r_std), SAVE, DIMENSION(nparts,nlitt)                  :: litterfrac         !! The fraction of leaves, wood, etc. that 
                                                                                      !! goes into metabolic and structural 
                                                                                      !! litterpools (0-1, unitless)
    REAL(r_std), DIMENSION(npts,nparts,nlitt)                   :: litterfrac_pxl     !! The fraction of leaves, wood, etc. that
                                                                                      !! goes into metabolic and structural
                                                                                      !! litterpools (0-1, unitless)
!$OMP THREADPRIVATE(litterfrac)
    REAL(r_std),DIMENSION(0:nbdl)                               :: z_soil             !! Soil levels (m)
!$OMP THREADPRIVATE(z_soil)
                                                                                      !! profiles (unitless)
    REAL(r_std), SAVE, DIMENSION(nlitt)                         :: litter_tau         !! Turnover time in litter pools (days)
!$OMP THREADPRIVATE(litter_tau)
    REAL(r_std), SAVE, DIMENSION(npool)                         :: pool_tau         !! Turnover time in litter and soil carbon pools (days)
!$OMP THREADPRIVATE(pool_tau)
    REAL(r_std), SAVE, DIMENSION(npool)                         :: DOC_tau          !! Residence time of DOC (days)    
!$OMP THREADPRIVATE(DOC_tau)
    REAL(r_std), SAVE, DIMENSION(nlitt,ncarb,nlevs)             :: frac_soil          !! Fraction of litter that goes into soil 
                                                                                      !! (litter -> carbon, above and below). The
                                                                                      !! remaining part goes to the atmosphere
!$OMP THREADPRIVATE(frac_soil)
    REAL(r_std), DIMENSION(npts)                                :: tsoil_decomp       !! Temperature used for decompostition in 
                                                                                      !! soil (K)
    REAL(r_std), DIMENSION(npts)                                :: soilhum_decomp     !! Humidity used for decompostition in soil
                                                                                      !! (unitless)
 !   REAL(r_std), SAVE, DIMENSION(npts)                          :: z_lit              !! Thickness of the above ground litter layer (mm)
!!$OMP THREADPRIVATE(z_lit)

    REAL(r_std), DIMENSION(npts)                                :: fd                 !! Fraction of structural or metabolic litter
                                                                                      !! decomposed (unitless)
    REAL(r_std), DIMENSION(npts,nelements)                      :: qd                 !! Quantity of structural or metabolic litter
                                                                                      !! decomposed @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nelements)                      :: qd_flood           !! Quantity of structural or metabolic litter
                                                                                      !! decomposed in flooded areas @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm)                            :: old_struc_above    !! Old structural litter, above ground 
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nbdl)                       :: old_struc_below    !! Old structural litter, below ground 
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nlitt, nelements)           :: litter_inc_PFT_above  !! Increase of litter, per PFT, metabolic and
                                                                                      !! structural, above ground. The 
                                                                                      !! unit is given by m^2 of ground.  
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nlitt,nbdl,nelements)      :: litter_inc_PFT_below     !! Increase of litter, per PFT, metabolic and
                                                                                      !! structural, above ground. The 
                                                                                      !!unit is given by m^2 of ground.
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nelements)            :: litter_inc_above   !! Increase of metabolic and structural 
                                                                                      !! litter, above ground. The unit 
                                                                                      !! is given by m^2 of ground. 
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements)       :: litter_inc_below   !! Increase of metabolic and structural 
                                                                                      !! litter below ground. The unit
                                                                                      !! is given by m^2 of ground
                                                                                      !! @tex $(gC m^{-2})$ @endtex

    REAL(r_std), DIMENSION(npts,nvm)                            :: lignin_struc_inc_above !! Lignin increase in structural litter, 
                                                                                      !! above ground. The unit is given 
                                                                                      !! by m^2 of ground. 
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nbdl)                       :: lignin_struc_inc_below   !! Lignin increase in structural litter, 
                                                                                      !! below ground. The unit is given
                                                                                      !! by m^2 of ground
                                                                                      !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nlitt,nelements)            :: litter_pft         !! Metabolic and structural litter above the 
                                                                                      !! ground per PFT
    REAL(r_std), DIMENSION(npts)                                :: zdiff_min          !! Intermediate field for looking for minimum
                                                                                      !! of what? this is not used in the code. 
                                                                                      !! [??CHECK] could we delete it?
    CHARACTER(LEN=10), DIMENSION(nlitt)                         :: litter_str         !! Messages to write output information about
                                                                                      !! the litter
    CHARACTER(LEN=22), DIMENSION(nparts)                        :: part_str           !! Messages to write output information about
                                                                                      !! the plant
    CHARACTER(LEN=7), DIMENSION(ncarb)                          :: carbon_str         !! Messages to write output information about
                                                                                      !! the soil carbon
    CHARACTER(LEN=5), DIMENSION(nlevs)                          :: level_str          !! Messages to write output information about
                                                                                      !! the level (aboveground or belowground litter)
    INTEGER(i_std)                                              :: i,j,k,l,m          !! Indices (unitless)
    REAL(r_std), DIMENSION(npts)                                :: Dif_coef           !! Diffusion coeficient used for bioturbation (m2 days-1)
!!$OMP THREADPRIVATE(Dif_coef)

    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements)       :: litter_flux        !! Belowground litter carbon flux within pools towards the soil column\f$(gC m^{2} dt^{-1})$\f
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements)       :: litter_flux_old    !! Belowground litter carbon flux within pools towards the soil column\f$(gC m^{2} dt^{-1})$\f
                                                                                      !! used for storage
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nelements)            :: litter_above_flux  !! Above litter carbon flux within pools towards the soil column\f$(gC m^{2} dt^{-1})$\f 
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nelements)            :: litter_above_flux_old  !! Above litter carbon flux within pools towards the soil column\f$(gC m^{2} dt^{-1})$\f
                                                                                      !! used for storage
    REAL(r_std), DIMENSION(npts)                                :: rpc                !! Scaling factor for integrating vertical soil 
                                                                                      !! profiles (unitless)
    REAL(r_std), DIMENSION(npts,nvm,nmbcomp,nelements)     :: check_intern       !! Contains the components of the internal
                                                                                      !! mass balance chech for this routine
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements)             :: closure_intern     !! Check closure of internal mass balance

    REAL(r_std), DIMENSION(npts,nvm,nelements)             :: pool_start         !! Start and end pool of this routine 
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex
    REAL(r_std), DIMENSION(npts,nvm,nelements)             :: pool_end           !! End pool of this routine not corrected
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex 
    REAL(r_std), DIMENSION(npts,nvm,nelements)             :: pool_end_after           !! Start and end pool of this routine 
                                                                                      !! @tex $(gC pixel^{-1} dt^{-1})$ @endtex 
    REAL(r_std), DIMENSION(npts,nbdl)                      :: moist_soil         !! moisture in soil for one pixel (m3/m3)

   ! REAL(r_std), DIMENSION(npts,nvm,nbdl)                       :: resp_hetero_litter_layer !! Litter heterotrophic respiration per layer. The unit
                                                                                      !! is given by m^2 of ground by z.  
                                                                                      !! @tex $(gC dtradia one\_day^{-1}) m^{-2} z^{-1})$ @endtex 
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements)  ::litter_below_old         !! Metabolic and structural litter, below ground
                                                                                      !! The unit is given by m^2 of
                                                                                      !! ground @tex $(gC m^{-2} $ @endtex used for storage
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nbdl,nelements)  ::litter_below_old_buffer  !! Metabolic and structural litter, below ground
                                                                                      !! The unit is given by m^2 of
                                                                                      !! ground @tex $(gC m^{-2} $ @endtex used for storage
    REAL(r_std), DIMENSION(npts,nlitt,nvm,nelements)       :: litter_above_old        !! Metabolic and structural litter,above ground
                                                                                      !! The unit is given by m^2 of 
                                                                                      !! ground @tex $(gC m^{-2})$ @endtex used for storage
    REAL(r_std), DIMENSION(npts)                           :: one_array 
!_ ================================================================================================================================

    IF (printlev>=3) WRITE(numout,*) 'Entering littercalc'

  !! 1. Initialisations of the different fields during the first call of the routine 
    !! 1.0 Calculation of soil moisture
    moist_soil(:,:) = zero
      DO l = 1,nbdl
          DO i = 1, nstm
             moist_soil(:,l)= moist_soil(:,l) + soil_mc(:,l,i)*soiltile(:,i)
          ENDDO
      ENDDO

    WHERE (bulk_dens(:) .LT. 500)
    bulk_dens(:) = bulk_dens(:)*1e3
    ENDWHERE

    z_soil(0) = zero
    z_soil(1:nbdl) = diaglev(1:nbdl)

    !! 1.1 Initialize check for mass balance closure
    !  The mass balance is calculated at the end of this routine
    !  in section 14. Initial biomass and harvest pool and all other
    !  relevant pools were set to zero before calling this routine.
    pool_start(:,:,:) = zero
    IF (ld_doc) THEN
       DO m = 2, nvm
          DO i = 1,nlitt
             DO l = 1, nbdl
                pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                     ( litter_below(:,i,m,l,icarbon) ) * veget_max(:,m)
             ENDDO
          ENDDO
       ENDDO

       DO m = 2, nvm
          DO i = 1,nlitt
             pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                  ( litter_above(:,i,m,icarbon) ) * veget_max(:,m)
          ENDDO
       ENDDO

       DO m = 2, nvm
          DO i = 1, nparts
             pool_start(:,m,icarbon) = pool_start(:,m,icarbon) + &
                  (turnover(:,m,i,icarbon) + bm_to_litter(:,m,i,icarbon)) * veget_max(:,m)
          ENDDO
       ENDDO
    ENDIF
    IF ( firstcall ) THEN

       !! 1.2.1 litter fractions:
       !!   what fraction of leaves, wood, etc. goes into metabolic and structural litterpools
       DO k = 1, nparts

          litterfrac(k,imetabolic) = metabolic_ref_frac - metabolic_LN_ratio * LC(k) * CN(k)
          litterfrac(k,istructural) = un - litterfrac(k,imetabolic)
    
       ENDDO

       !! 1.2.2 residence times in litter pools (days)
       litter_tau(imetabolic) = tau_metabolic * one_year      ! .5 years
       litter_tau(istructural) = tau_struct * one_year     ! 3 years

       pool_tau(imetabo) = tau_metabolic 
       pool_tau(istrabo) = tau_struct 
       pool_tau(imetbel) = tau_metabolic 
       pool_tau(istrbel) = tau_struct 
       pool_tau(iact) = carbon_tau_iactive 
       pool_tau(islo) = carbon_tau_islow 
       pool_tau(ipas) = carbon_tau_ipassive 
       DOC_tau(imetabo) = DOC_tau_labile / one_year
       DOC_tau(istrabo) = DOC_tau_labile / one_year
       DOC_tau(imetbel) = DOC_tau_labile / one_year
       DOC_tau(istrbel) = DOC_tau_labile / one_year
       DOC_tau(iact) = DOC_tau_labile / one_year
       DOC_tau(islo) = DOC_tau_stable / one_year
       DOC_tau(ipas) = DOC_tau_stable / one_year

       !! 1.2.3 decomposition flux fraction that goes into soil
       !       (litter -> carbon, above and below)
       !       1-frac_soil goes into atmosphere
       frac_soil(:,:,:) = zero

       ! structural litter: lignin fraction goes into slow pool + respiration,
       !                    rest into active pool + respiration
       frac_soil(istructural,iactive,iabove) = frac_soil_struct_aa
       frac_soil(istructural,iactive,ibelow) = frac_soil_struct_ab
       frac_soil(istructural,islow,iabove) = frac_soil_struct_sa
       frac_soil(istructural,islow,ibelow) = frac_soil_struct_sb

       ! metabolic litter: all goes into active pool + respiration.
       !   Nothing into slow or passive pool.
       frac_soil(imetabolic,iactive,iabove) = frac_soil_metab_aa
       frac_soil(imetabolic,iactive,ibelow) = frac_soil_metab_ab
       
 
       !! 1.4 messages
       litter_str(imetabolic) = 'metabolic'
       litter_str(istructural) = 'structural'

       carbon_str(iactive) = 'active'
       carbon_str(islow) = 'slow'
       carbon_str(ipassive) = 'passive'

       level_str(iabove) = 'above'
       level_str(ibelow) = 'below'

       part_str(ileaf) = 'leaves'
       part_str(isapabove) = 'sap above ground'
       part_str(isapbelow) = 'sap below ground'
       part_str(iheartabove) = 'heartwood above ground'
       part_str(iheartbelow) = 'heartwood below ground'
       part_str(iroot) = 'roots'
       part_str(ifruit) = 'fruits'
       part_str(icarbres) = 'carbohydrate reserve'

       WRITE(numout,*) 'litter:'

       WRITE(numout,*) '   > C/N ratios: '
       DO k = 1, nparts
          WRITE(numout,*) '       ', part_str(k), ': ',CN(k)
       ENDDO

       WRITE(numout,*) '   > Lignine/C ratios: '
       DO k = 1, nparts
          WRITE(numout,*) '       ', part_str(k), ': ',LC(k)
       ENDDO

       WRITE(numout,*) '   > fraction of compartment that goes into litter: '
       DO k = 1, nparts
          DO m = 1, nlitt
             WRITE(numout,*) '       ', part_str(k), '-> ',litter_str(m), ':',litterfrac(k,m)
          ENDDO
       ENDDO

       WRITE(numout,*) '   > minimal carbon residence time in litter pools (d):'
       DO m = 1, nlitt
          WRITE(numout,*) '       ',litter_str(m),':',litter_tau(m)
       ENDDO

       WRITE(numout,*) '   > litter decomposition flux fraction that really goes '
       WRITE(numout,*) '     into carbon pools (rest into the atmosphere):'
       DO m = 1, nlitt
          DO l = 1, nlevs
             DO k = 1, ncarb
                WRITE(numout,*) '       ',litter_str(m),' ',level_str(l),' -> ',&
                     carbon_str(k),':', frac_soil(m,k,l)
             ENDDO
          ENDDO
       ENDDO

       firstcall = .FALSE.

    ENDIF

    !! 1.3 Initialization

       DO k = 1, nparts

          litterfrac_pxl(:,k,istructural) = metabolic_ref_frac - metabolic_LN_ratio * LC(k) * CN(k)! * (un + poor_soils(:))
          litterfrac_pxl(:,k,istructural) = un - litterfrac_pxl(:,k,imetabolic)

       ENDDO

    litter_above_flux(:,:,:,:) = zero
    litter_flux(:,:,:,:,:) = zero
    litter_flux_old(:,:,:,:,:) = zero
    old_struc_above(:,:) = zero
    old_struc_below(:,:,:) = zero
    litter_pft(:,:,:,:) = zero
    Dif_coef(:) = (Dif/one_year)*dt 

    
    !! 1.3 litter above the ground per PFT.
    DO j = 2, nvm ! Loop over # PFTs
       DO k = 1, nlitt !Loop over litter pool
          litter_pft(:,j,k,icarbon) = litterpart(:,j,k) * litter_above(:,k,j,icarbon)
       ENDDO
    ENDDO

    
    !! 1.4 set output to zero
    deadleaf_cover(:) = zero
    resp_hetero_litter(:,:,:) = zero
    resp_hetero_flood(:,:) = zero
    soilcarbon_input(:,:,:,:,:)= zero
    floodcarbon_input(:,:,:,:)= zero
    !resp_hetero_litter_layer(:,:,:) = zero

  !! 2. Add biomass to different litterpools (per m^2 of ground)
    
    !! 2.1 first, save old structural litter (needed for lignin fractions).
    !     above
       DO m = 2,nvm !Loop over PFTs

          old_struc_above(:,m) = litter_above(:,istructural,m,icarbon)

       ENDDO

    !below
    DO l = 1, nbdl !Loop over soil levels (below ground)
       DO m = 2,nvm !Loop over PFTs

          old_struc_below(:,m,l) = litter_below(:,istructural,m,l,icarbon)

       ENDDO
    ENDDO
    
    !! 2.2 update litter, dead leaves, and lignin content in structural litter
    litter_inc_above(:,:,:,:) = zero
    lignin_struc_inc_above(:,:) = zero
    litter_inc_below(:,:,:,:,:) = zero
    lignin_struc_inc_below(:,:,:) = zero
    litter_inc_PFT_above(:,:,:,:) = zero
    litter_inc_PFT_below(:,:,:,:,:) = zero


    DO j = 2,nvm !Loop over PFTs

       !! 2.2.1 litter
       DO k = 1, nlitt    !Loop over litter pools (metabolic and structural)

          !! 2.2.2 calculate litter increase (per m^2 of ground).
          !       Only a given fracion of fruit turnover is directly coverted into litter.
          !       Litter increase for each PFT, structural and metabolic, above/below. It must
          !       be noted that the distribution of above ground litter is done within the soil layers
          !       as a proportion of the layer size. But for the below ground layer it follows the root profile

          ! For the first litter layer
          litter_inc_PFT_above(:,j,k,icarbon) = &
               litterfrac_pxl(:,ileaf,k) * bm_to_litter(:,j,ileaf,icarbon) + &
               litterfrac_pxl(:,isapabove,k) * bm_to_litter(:,j,isapabove,icarbon) + &
               litterfrac_pxl(:,iheartabove,k) * bm_to_litter(:,j,iheartabove,icarbon) + &
               litterfrac_pxl(:,ifruit,k) * bm_to_litter(:,j,ifruit,icarbon) + &
               litterfrac_pxl(:,icarbres,k) * bm_to_litter(:,j,icarbres,icarbon) + &
               litterfrac_pxl(:,ileaf,k) * turnover(:,j,ileaf,icarbon) + &
               litterfrac_pxl(:,isapabove,k) * turnover(:,j,isapabove,icarbon) + &
               litterfrac_pxl(:,iheartabove,k) * turnover(:,j,iheartabove,icarbon) + &
               litterfrac_pxl(:,ifruit,k) * turnover(:,j,ifruit,icarbon) + &
               litterfrac_pxl(:,icarbres,k) * turnover(:,j,icarbres,icarbon) 

             ! litter increase, met/struct, above
             litter_inc_above(:,k,j,icarbon) = litter_inc_above(:,k,j,icarbon) + litter_inc_PFT_above(:,j,k,icarbon)

          !! The function used for root profile is coming from stomate_npp.f90
       rpc(:) = un /( ( un - EXP( -z_soil(nbdl) / rprof(:,j) )) ) 

          DO l = 1, nbdl

             litter_inc_PFT_below(:,j,k,l,icarbon) = &
                  litterfrac_pxl(:,isapbelow,k) * bm_to_litter(:,j,isapbelow,icarbon) *  &
               rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  litterfrac_pxl(:,iheartbelow,k) * bm_to_litter(:,j,iheartbelow,icarbon) * &
               rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  litterfrac_pxl(:,iroot,k) * bm_to_litter(:,j,iroot,icarbon) * &
               rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  litterfrac_pxl(:,isapbelow,k) * turnover(:,j,isapbelow,icarbon) * &
               rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  litterfrac_pxl(:,iheartbelow,k) * turnover(:,j,iheartbelow,icarbon) * &
               rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  litterfrac_pxl(:,iroot,k) * turnover(:,j,iroot,icarbon) * &
               rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) )

             ! litter increase, met/struct, below
             litter_inc_below(:,k,j,l,icarbon) = litter_inc_below(:,k,j,l,icarbon) + litter_inc_PFT_below(:,j,k,l,icarbon)
          ENDDO
          !! 2.2.3 dead leaves, for soil cover.
          dead_leaves(:,j,k) = &
               dead_leaves(:,j,k) + &
               litterfrac_pxl(:,ileaf,k) * ( bm_to_litter(:,j,ileaf,icarbon) + turnover(:,j,ileaf,icarbon) )

          !! 2.2.4 lignin increase in structural litter
          IF ( k .EQ. istructural ) THEN

             lignin_struc_inc_above(:,j) = &
                  lignin_struc_inc_above(:,j) + &
                  LC(ileaf) * bm_to_litter(:,j,ileaf,icarbon) + &
                  LC(isapabove) * bm_to_litter(:,j,isapabove,icarbon) + &
                  LC(iheartabove) * bm_to_litter(:,j,iheartabove,icarbon) + &
                  LC(ifruit) * bm_to_litter(:,j,ifruit,icarbon) + &
                  LC(icarbres) * bm_to_litter(:,j,icarbres,icarbon) + &
                  LC(ileaf) * turnover(:,j,ileaf,icarbon) + &
                  LC(isapabove) * turnover(:,j,isapabove,icarbon) + &
                  LC(iheartabove) * turnover(:,j,iheartabove,icarbon) + &
                  LC(ifruit) * turnover(:,j,ifruit,icarbon) + &
                  LC(icarbres) * turnover(:,j,icarbres,icarbon) 

            !! The function used for root profile is coming from stomate_npp.f90
            DO l = 1, nbdl

               lignin_struc_inc_below(:,j,l) = &
                  lignin_struc_inc_below(:,j,l) + &
                  LC(isapbelow) * bm_to_litter(:,j,isapbelow,icarbon) * &
                  rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  LC(iheartbelow) * bm_to_litter(:,j,iheartbelow,icarbon) * &
                  rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  LC(iroot) * bm_to_litter(:,j,iroot,icarbon) * &
                  rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  LC(isapbelow)*turnover(:,j,isapbelow,icarbon) * &
                  rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  LC(iheartbelow)*turnover(:,j,iheartbelow,icarbon) * &
                  rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) ) + &
                  LC(iroot)*turnover(:,j,iroot,icarbon) * &
                  rpc(:) * ( EXP( -z_soil(l-1)/rprof(:,j) ) - EXP( -z_soil(l)/rprof(:,j) ) )
            ENDDO

          ENDIF
       ENDDO
     ENDDO



    !! 2.2.5 add new litter (struct/met, above/below)
         litter_above(:,:,:,:) = litter_above(:,:,:,:) + litter_inc_above(:,:,:,:)
         litter_below(:,:,:,:,:) = litter_below(:,:,:,:,:) + litter_inc_below(:,:,:,:,:)

    !! 2.2.6 for security: can't add more lignin than structural litter (above/below)
       DO m = 2,nvm !Lopp over PFTs

          lignin_struc_inc_above(:,m) = &
               MIN( lignin_struc_inc_above(:,m), litter_inc_above(:,istructural,m,icarbon) )

       ENDDO

    DO l = 1, nbdl !Loop over soil levels (below ground)
       DO m = 2,nvm !Lopp over PFTs

          lignin_struc_inc_below(:,m,l) = &
               MIN( lignin_struc_inc_below(:,m,l), litter_inc_below(:,istructural,m,l,icarbon) )

       ENDDO
    ENDDO

    !! 2.2.7.1 new lignin content: add old lignin and lignin increase, divide by 
    !!       total structural litter (above)

       DO m = 2,nvm !Loop over PFTs
          WHERE( litter_above(:,istructural,m,icarbon) .GT. min_stomate )

       !MM : Soenke modif
       ! Best vectorization ?
!!$       lignin_struc(:,:,:) = &
!!$            ( lignin_struc(:,:,:)*old_struc(:,:,:) + lignin_struc_inc(:,:,:) ) / &
!!$            litter(:,istructural,:,:,icarbon)

             lignin_struc_above(:,m) = lignin_struc_above(:,m) * old_struc_above(:,m)
             lignin_struc_above(:,m) = lignin_struc_above(:,m) + lignin_struc_inc_above(:,m)
             lignin_struc_above(:,m) = lignin_struc_above(:,m) / litter_above(:,istructural,m,icarbon)
          ELSEWHERE
             lignin_struc_above(:,m) = zero
          ENDWHERE
       ENDDO

    !! 2.2.7.2 new lignin content: add old lignin and lignin increase, divide by 
    !!       total structural litter (below)
    DO l = 1, nbdl !Loop over soil levels (below ground)
       DO m = 2,nvm !Loop over PFTs
          WHERE( litter_below(:,istructural,m,l,icarbon) .GT. min_stomate )

       !MM : Soenke modif
       ! Best vectorization ?
!!$       lignin_struc(:,:,:) = &
!!$            ( lignin_struc(:,:,:)*old_struc(:,:,:) + lignin_struc_inc(:,:,:)) / &
!!$            litter(:,istructural,:,:,icarbon)

             lignin_struc_below(:,m,l) = lignin_struc_below(:,m,l) * old_struc_below(:,m,l)
             lignin_struc_below(:,m,l) = lignin_struc_below(:,m,l) + lignin_struc_inc_below(:,m,l)
             lignin_struc_below(:,m,l) = lignin_struc_below(:,m,l) /litter_below(:,istructural,m,l,icarbon)
          ELSEWHERE
             lignin_struc_below(:,m,l) = zero
          ENDWHERE
       ENDDO
    ENDDO

    
    !! 2.3 new litter fraction per PFT (for structural and metabolic litter, above
    !!       the ground).
       DO j = 2,nvm !Loop over PFTs
         DO l =1,nlitt 
         WHERE ( litter_above(:,l,j,icarbon) .GT. min_stomate )

          litterpart(:,j,l) = &
               ( litter_pft(:,j,l,icarbon) + litter_inc_PFT_above(:,j,l,icarbon) ) / litter_above(:,l,j,icarbon)

         ELSEWHERE

          litterpart(:,j,l) = zero

         ENDWHERE
         ENDDO
       ENDDO
  !! 3. Temperature control on decay: Factor between 0 and 1

    !! 3.1 above: surface temperature 
    DO j=1,nlitt
       control_temp_above(:,j) = control_temp_func (npts, tsurf,litter_tau(j)*(un+poor_soils(:))) 
    ENDDO         
    !! 3.2 below: convolution of temperature and decomposer profiles
    !!            (exponential decomposer profile supposed)
   
    !! 3.2.1 integrate over the nbdl levels
    tsoil_decomp(:) = zero

    DO l = 1, nbdl
       DO j = 1,npool

          tsoil_decomp(:) = tsoil(:,l) 

          control_temp_soil(:,l,j) = control_temp_func (npts,tsoil_decomp,pool_tau(j)*(un+poor_soils(:)))

       ENDDO
    ENDDO

    DO l = 1, nbdl
       DO j = 1,npool

          tsoil_decomp(:) = tsoil(:,l)

          control_temp_soil(:,l,npool+j) = control_temp_func(npts,tsoil_decomp,DOC_tau(j)*(un+poor_soils(:)))

       ENDDO
    ENDDO

  !! 4. Moisture control. Factor between 0 and 1

    !! 4.1 above the ground: litter humidity 

    one_array(:) = un
    soilhum_decomp(:) = zero
    DO m= 1,nvm
       !We used the sum over the 4 first layer as it is done to
       !compute litterhum in hydrol.f90
         soilhum_decomp(:) =soil_mc(:,1,pref_soil_veg(m))*(z_soil(1)/z_soil(4)) + &
                            soil_mc(:,2,pref_soil_veg(m))*((z_soil(2)-z_soil(1))/z_soil(4)) + &
                            soil_mc(:,3,pref_soil_veg(m))*((z_soil(3)-z_soil(2))/z_soil(4)) + &
                            soil_mc(:,4,pref_soil_veg(m))*((z_soil(4)-z_soil(3))/z_soil(4))
         !
       IF (moist_func_Moyano) THEN
        control_moist_above(:,m) = control_moist_func_moyano (npts, soilhum_decomp,bulk_dens, &
                                                    clay, carbon, veget_max)
       ELSE 
         control_moist_above(:,m) = control_moist_func (npts, soilhum_decomp)
       ENDIF
    ENDDO
    !! 4.2 below: soil humidity for each soil layers 

    !! 4.2.1 integrate over the nbdl levels
    soilhum_decomp(:) = zero

    DO l = 1, nbdl !Loop over soil levels
      DO m =1,nvm
         soilhum_decomp(:) =soil_mc(:,l,pref_soil_veg(m))
         IF (moist_func_Moyano) THEN
            control_moist_soil(:,l,m) = control_moist_func_moyano (npts, soilhum_decomp,bulk_dens, &
                 clay, carbon, veget_max)
         ELSE
            control_moist_soil(:,l,m) = control_moist_func (npts, soilhum_decomp)
         ENDIF
      ENDDO
    ENDDO

  !! 5. fluxes from above ground litter to carbon pools and respiration
      DO m = 2,nvm !Loop over PFTs

          !! 5.1 structural litter: goes into active and slow carbon pools + respiration

          !! 5.1.1 total quantity of above ground structural litter which is decomposed

      WHERE (soil_mc(:,1,pref_soil_veg(m)) .GT. zero)
          fd(:) = dt/(litter_tau(istructural)*(un+poor_soils(:))) * &
             control_temp_above(:,istructural) * control_moist_above(:,m) * exp( -litter_struct_coef * lignin_struc_above(:,m) )
      ENDWHERE
          DO k = 1,nelements
             qd(:,k) = litter_above(:,istructural,m,k) * fd(:) * (un - flood_frac)
             qd_flood(:,k) = litter_above(:,istructural,m,k) * fd(:) * flood_frac/trois
          ENDDO
          IF (.NOT. lat_exp_doc) THEN
             qd = qd + qd_flood
             qd_flood = zero
          ELSE
             !Do nothing
          ENDIF

          !! 5.1.2 decompose same fraction of structural part of dead leaves. Not exact
          !!       as lignine content is not the same as that of the total structural litter.
          dead_leaves(:,m,istructural) = dead_leaves(:,m,istructural) * (un - fd(:)*(un - flood_frac(:)) - fd(:)*flood_frac(:)/trois)

      !BE CAREFUL: Here resp_hetero_litter is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_litter is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
          resp_hetero_litter(:,m,iabove) = resp_hetero_litter(:,m,iabove) + &
               ( 1. - frac_soil(istructural,iactive,iabove) ) * qd(:,icarbon) * &
               ( 1. - lignin_struc_above(:,m) ) / dt
          resp_hetero_flood(:,m) = resp_hetero_flood(:,m) + &
               ( 1. - frac_soil(istructural,iactive,iabove) ) * qd_flood(:,icarbon) * &
               ( 1. - lignin_struc_above(:,m) ) / dt

      !BE CAREFUL: Here resp_hetero_litter is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_litter is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
          resp_hetero_litter(:,m,iabove) = resp_hetero_litter(:,m,iabove) + &
               ( 1. - frac_soil(istructural,islow,iabove) ) * qd(:,icarbon) * lignin_struc_above(:,m) / dt
          resp_hetero_flood(:,m) = resp_hetero_flood(:,m) + &
               ( 1. - frac_soil(istructural,islow,iabove) ) * qd_flood(:,icarbon) * lignin_struc_above(:,m) / dt

         ! resp_hetero_litter_layer(:,m,1) = resp_hetero_litter_layer(:,m,1) + &
         !      ( 1. - frac_soil(istructural,islow,iabove) ) * qd(:,icarbon) *lignin_struc_above(:,m) / dt

          !! 5.1.3 Calculation of the soilcarbon_input in gC m^-2 of water

          DO k = 1, nelements
             soilcarbon_input(:,m,1,istrabo,k) = (frac_soil(istructural,iactive,iabove) * ( 1. - lignin_struc_above(:,m) ) * qd(:,k) + &
                                  frac_soil(istructural,islow,iabove) *  qd(:,k) * lignin_struc_above(:,m) ) /dt 
             floodcarbon_input(:,m,iact,k) = frac_soil(istructural,iactive,iabove) * ( 1. - lignin_struc_above(:,m) ) * qd_flood(:,k) / dt
             floodcarbon_input(:,m,islo,k) = frac_soil(istructural,islow,iabove) * qd_flood(:,k) * lignin_struc_above(:,m) /dt
         ENDDO

          litter_above(:,istructural,m,:) = litter_above(:,istructural,m,:) - qd(:,:) - qd_flood(:,:)
          
          !! 5.2 above ground metabolic litter goes into active carbon pool + respiration
         
          !! 5.2.1 total quantity of aboveground metabolic litter that is decomposed
      WHERE (soil_mc(:,1,pref_soil_veg(m)) .GT. zero)
          fd(:) = dt/(litter_tau(imetabolic)*(un+poor_soils(:))) * control_temp_above(:,imetabolic) * control_moist_above(:,m)
      ELSEWHERE
          fd(:) = zero
      ENDWHERE

          DO k = 1,nelements          
             qd(:,k) = litter_above(:,imetabolic,m,k) * fd(:) * (un - flood_frac)
             qd_flood(:,k) = litter_above(:,imetabolic,m,k) * fd(:) * flood_frac/trois
          ENDDO
          !
          IF (.NOT. lat_exp_doc) THEN
             qd = qd + qd_flood
             qd_flood = zero
          ELSE
             !Do nothing
          ENDIF

          !! 5.2.2 decompose same fraction of metabolic part of dead leaves.
          dead_leaves(:,m,imetabolic) = dead_leaves(:,m,imetabolic) * (un - fd(:)*(un - flood_frac(:)) - fd(:)*flood_frac(:)/trois)

      !BE CAREFUL: Here resp_hetero_litter is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_litter is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
          resp_hetero_litter(:,m,iabove) = resp_hetero_litter(:,m,iabove) + &
               ( 1. - frac_soil(imetabolic,iactive,iabove) ) * qd(:,icarbon) / dt
          resp_hetero_flood(:,m) = resp_hetero_flood(:,m) + &
               ( 1. - frac_soil(imetabolic,iactive,iabove) ) * qd_flood(:,icarbon) / dt

          !! 5.2.3 Calculation of the soilcarbon_input in gC m^-2 of water
             ! For above ground litter we assume that the soilcarbon_input coming from AB
             ! litter decomposition is directly incorporated into the 1st soil
             ! layers

          DO k = 1, nelements
             soilcarbon_input(:,m,1,imetabo,k) = frac_soil(imetabolic,iactive,iabove) *  qd(:,k) / dt              
             floodcarbon_input(:,m,iact,k) = floodcarbon_input(:,m,iact,k) + frac_soil(imetabolic,iactive,iabove) *  qd_flood(:,k) / dt
         ENDDO

          litter_above(:,imetabolic,m,:) = litter_above(:,imetabolic,m,:) - qd(:,:) - qd_flood(:,:)

      ENDDO

!! 6. fluxes from below ground litter to carbon pools and respiration
   DO l = 1,nbdl
       DO m = 2,nvm !Loop over PFTs

          !! 6.1 structural litter: goes into active and slow carbon pools respiration

          !! 6.1.1 total quantity of below ground structural litter which is decomposed
          WHERE (soil_mc(:,l,pref_soil_veg(m)) .GT. zero)
             fd(:) = dt/(litter_tau(istructural)*(un+poor_soils(:))) * &
                  control_temp_soil(:,l,istrbel) * control_moist_soil(:,l,m) * exp(-litter_struct_coef * lignin_struc_below(:,m,l) )
          ELSEWHERE
             fd(:) = zero
          ENDWHERE

          DO k = 1,nelements
             qd(:,k) = litter_below(:,istructural,m,l,k) * fd(:) * (un - flood_frac)
             qd_flood(:,k) = litter_below(:,istructural,m,l,k) * fd(:) * flood_frac/trois
          ENDDO
          !
          IF (.NOT. lat_exp_doc) THEN
             qd = qd + qd_flood
             qd_flood = zero
          ELSE
             !Do nothing
          ENDIF


      !BE CAREFUL: Here resp_hetero_litter is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_litter is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
          resp_hetero_litter(:,m,ibelow) = resp_hetero_litter(:,m,ibelow) + &
               ( 1. - frac_soil(istructural,iactive,ibelow) ) * qd(:,icarbon) * &
               ( 1. - lignin_struc_below(:,m,l) ) / dt
          resp_hetero_flood(:,m) = resp_hetero_flood(:,m) + &
               ( 1. - frac_soil(istructural,iactive,ibelow) ) * qd_flood(:,icarbon) * &
               ( 1. - lignin_struc_below(:,m,l) ) / dt

      
      !BE CAREFUL: Here resp_hetero_litter is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_litter is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
          resp_hetero_litter(:,m,ibelow) = resp_hetero_litter(:,m,ibelow) + &
               ( 1. - frac_soil(istructural,islow,ibelow) ) * qd(:,icarbon) * lignin_struc_below(:,m,l) / dt
          resp_hetero_flood(:,m) = resp_hetero_flood(:,m) + &
               ( 1. - frac_soil(istructural,islow,ibelow) ) * qd_flood(:,icarbon) * lignin_struc_below(:,m,l) / dt

          !! 6.1 Calculation of the soilcarbon_input in gC m^-2

          DO k = 1, nelements
             IF (l .GT. sro_bottom) THEN
                  soilcarbon_input(:,m,l,istrbel,k) = (frac_soil(istructural,iactive,ibelow) * ( 1. - lignin_struc_below(:,m,l) ) &
                       * (qd(:,k)+qd_flood(:,k)) + & 
                         (frac_soil(istructural,islow,ibelow) * (qd(:,k)+qd_flood(:,k)) * lignin_struc_below(:,m,l) )) / dt
             ELSE
                soilcarbon_input(:,m,l,istrbel,k) = (frac_soil(istructural,iactive,ibelow) * ( 1. - lignin_struc_below(:,m,l) ) &
                       * (qd(:,k)) + &
                         (frac_soil(istructural,islow,ibelow) * (qd(:,k)) * lignin_struc_below(:,m,l) )) / dt
                floodcarbon_input(:,m,iact,k) = floodcarbon_input(:,m,iact,k) + &
                     frac_soil(istructural,iactive,ibelow) * ( 1. - lignin_struc_below(:,m,l) ) * qd_flood(:,k) /dt
                floodcarbon_input(:,m,islo,k) = floodcarbon_input(:,m,islo,k) + (frac_soil(istructural,islow,ibelow) &
                     * qd_flood(:,k) * lignin_struc_below(:,m,l)) / dt
             ENDIF
          ENDDO

          litter_below(:,istructural,m,l,:) = litter_below(:,istructural,m,l,:) - qd(:,:) - qd_flood(:,:)

          !! 6.2 below ground metabolic litter goes into active carbon pool + respiration

          !! 6.2.1 total quantity of belowground metabolic litter that is decomposed
          WHERE (soil_mc(:,l,pref_soil_veg(m)) .GT. zero)
             fd(:) = dt/(litter_tau(imetabolic)*(un+poor_soils(:))) * control_temp_soil(:,l,imetbel) * control_moist_soil(:,l,m)
          ELSEWHERE
             fd(:) = zero
          ENDWHERE

          DO k = 1,nelements
             qd(:,k) = litter_below(:,imetabolic,m,l,k) * fd(:) * (un-flood_frac(:))
             qd_flood(:,k) = litter_below(:,imetabolic,m,l,k) * fd(:) * flood_frac(:)/trois
          ENDDO
          !
          IF (.NOT. lat_exp_doc) THEN
             qd = qd + qd_flood
             qd_flood = zero
          ELSE
             !Do nothing
          ENDIF


      !BE CAREFUL: Here resp_hetero_litter is divided by dt to have a value which corresponds to
      ! the sechiba time step but then in stomate.f90 resp_hetero_litter is multiplied by dt.
      ! Perhaps it could be simplified. Moreover, we must totally adapt the routines to the dtradia/one_day
      ! time step and avoid some constructions that could create bug during future developments.
          resp_hetero_litter(:,m,ibelow) = resp_hetero_litter(:,m,ibelow) + &
               ( 1. - frac_soil(imetabolic,iactive,ibelow) ) * qd(:,icarbon) /dt
          resp_hetero_flood(:,m) = resp_hetero_flood(:,m) + &
               ( 1. - frac_soil(imetabolic,iactive,ibelow) ) * qd_flood(:,icarbon) /dt
 
          !! 6.2.3 Calculation of the soilcarbon_input in gC m^-2 of water

          DO k = 1, nelements
             IF (l .GT. sro_bottom) THEN
                soilcarbon_input(:,m,l,imetbel,k) = frac_soil(imetabolic,iactive,ibelow) * (qd(:,k)+qd_flood(:,k))/ dt 
             ELSE
                soilcarbon_input(:,m,l,imetbel,k) = frac_soil(imetabolic,iactive,ibelow) * qd(:,k)/ dt
                floodcarbon_input(:,m,iact,k) = floodcarbon_input(:,m,iact,k) + frac_soil(imetabolic,iactive,ibelow) * qd_flood(:,k)/ dt
             ENDIF
          ENDDO
         !
          litter_below(:,imetabolic,m,l,:) = litter_below(:,imetabolic,m,l,:) - qd(:,:) - qd_flood(:,:)
         !
       ENDDO
   ENDDO


   DO m =2,nvm

      litter_below_old(:,:,m,:,:)=zero
      litter_below_old_buffer(:,:,m,:,:)=zero
      litter_below_old(:,:,m,:,:)=litter_below(:,:,m,:,:)
      litter_above_old(:,:,m,:)=zero
      litter_above_old(:,:,m,:)=litter_above(:,:,m,:)
      DO k = 1, nelements ! Loop over elements
         DO j = 1,nlitt ! Loop over litter pools

            litter_flux(:,j,m,1,k) = Dif_coef(:)*dt*ABS(litter_below_old(:,j,m,1,k)/(z_soil(1))-litter_below_old(:,j,m,2,k)/(z_soil(2)-z_soil(1)))/(z_soil(2)-z_soil(1))
            litter_above_flux(:,j,m,k) = Dif_coef(:)*dt*ABS(litter_above(:,j,m,k)/(z_litter*1e-3)-litter_below(:,j,m,1,k)/(z_litter*1e-3-z_soil(1)))
            ! BE CAREFUL: Here very important assumption we assume the above ground litter layers to be 10mm thick by default.

            DO l= 2, nbdl-1 ! Loop over soil layers
               litter_flux(:,j,m,l,k) = Dif_coef(:)*dt*ABS(litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1))-litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)))/(z_soil(l+1)-z_soil(l))
            ENDDO
            litter_flux_old(:,j,m,:,k) = litter_flux(:,j,m,:,k)
            litter_above_flux_old(:,j,m,k) = litter_above_flux(:,j,m,k)

            !Below we checked if in case that, in a given layer, you have diffusion in the above and below litters, both fluxes are not higher than the stocks of the given layer.
            WHERE (litter_above_old(:,j,m,k)/(z_litter*1e-3) .LT. litter_below_old(:,j,m,1,k)/(z_soil(1)) .AND. &
                 litter_below_old(:,j,m,2,k)/(z_soil(2)-z_soil(1)) .LT. litter_below_old(:,j,m,1,k)/(z_soil(1)) .AND. &
                 litter_above_flux_old(:,j,m,k)+litter_flux_old(:,j,m,1,k) .GT. litter_below_old(:,j,m,1,k))
               litter_above_flux(:,j,m,k) = litter_below_old(:,j,m,1,k)*(litter_above_flux_old(:,j,m,k)/(litter_above_flux_old(:,j,m,k)+litter_flux_old(:,j,m,1,k)))
               litter_flux(:,j,m,1,k) = litter_below_old(:,j,m,1,k)*(litter_flux_old(:,j,m,1,k)/(litter_above_flux_old(:,j,m,k)+litter_flux_old(:,j,m,1,k)))
            ELSEWHERE (litter_above_old(:,j,m,k)/(z_litter*1e-3) .GE. litter_below_old(:,j,m,1,k)/(z_soil(1)) .AND. &
                 litter_below_old(:,j,m,2,k)/(z_soil(2)-z_soil(1)) .LT. litter_below_old(:,j,m,1,k)/(z_soil(1)) .AND. &
                 litter_below_old(:,j,m,1,k) + litter_above_flux_old(:,j,m,k) - litter_flux_old(:,j,m,1,k) .LE. min_stomate)
               litter_above_flux(:,j,m,k) = litter_above_flux_old(:,j,m,k)
               litter_flux(:,j,m,1,k) = litter_below_old(:,j,m,1,k) + litter_above_flux_old(:,j,m,k)
            ELSEWHERE (litter_above_old(:,j,m,k)/(z_litter*1e-3) .LT. litter_below_old(:,j,m,1,k)/(z_soil(1)) .AND. &
                 litter_below_old(:,j,m,2,k)/(z_soil(2)-z_soil(1)) .GE. litter_below_old(:,j,m,1,k)/(z_soil(1)) .AND. &
                 litter_below_old(:,j,m,1,k) - litter_above_flux_old(:,j,m,k) + litter_flux_old(:,j,m,1,k) .LE. min_stomate)
               litter_above_flux(:,j,m,k) = litter_below_old(:,j,m,1,k) + litter_flux_old(:,j,m,1,k) 
               litter_flux(:,j,m,1,k) = litter_flux_old(:,j,m,1,k)
            ELSEWHERE
               litter_above_flux(:,j,m,k) = litter_above_flux_old(:,j,m,k)
               litter_flux(:,j,m,1,k) = litter_flux_old(:,j,m,1,k)
            ENDWHERE


            DO l =1, nbdl-2
               WHERE (litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .LT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)) .AND. &
                    litter_below_old(:,j,m,l+2,k)/(z_soil(l+2)-z_soil(l+1)) .LT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)) .AND. &
                    litter_flux_old(:,j,m,l,k)+litter_flux_old(:,j,m,l+1,k) .GT. litter_below_old(:,j,m,l+1,k))
                  litter_flux(:,j,m,l,k) = litter_below_old(:,j,m,l+1,k)*(litter_flux_old(:,j,m,l,k)/(litter_flux_old(:,j,m,l,k)+litter_flux_old(:,j,m,l+1,k)))
                  litter_flux(:,j,m,l+1,k) = litter_below_old(:,j,m,l+1,k)*(litter_flux_old(:,j,m,l+1,k)/(litter_flux_old(:,j,m,l,k)+litter_flux_old(:,j,m,l+1,k)))
               ELSEWHERE (litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .GE. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)) .AND. &
                    litter_below_old(:,j,m,l+2,k)/(z_soil(l+2)-z_soil(l+1)) .LT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)) .AND. &
                    litter_below_old(:,j,m,l+1,k) + litter_flux_old(:,j,m,l,k) - litter_flux_old(:,j,m,l+1,k) .LE. min_stomate)
                  litter_flux(:,j,m,l,k) = litter_flux_old(:,j,m,l,k)
                  litter_flux(:,j,m,l+1,k) = litter_below_old(:,j,m,l+1,k) + litter_flux_old(:,j,m,l,k)
               ELSEWHERE (litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .LT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)) .AND. &
                    litter_below_old(:,j,m,l+2,k)/(z_soil(l+2)-z_soil(l+1)) .GE. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l)) .AND. &
                    litter_below_old(:,j,m,l+1,k) - litter_flux_old(:,j,m,l,k) + litter_flux_old(:,j,m,l+1,k) .LE. min_stomate)
                  litter_flux(:,j,m,l,k) = litter_below_old(:,j,m,l+1,k) + litter_flux_old(:,j,m,l+1,k)
                  litter_flux(:,j,m,l+1,k) = litter_flux_old(:,j,m,l+1,k)
               ELSEWHERE
                  litter_flux(:,j,m,l,k) = litter_flux_old(:,j,m,l,k)
                  litter_flux(:,j,m,l+1,k) = litter_flux_old(:,j,m,l+1,k)
               ENDWHERE
            ENDDO

            WHERE ((litter_above_old(:,j,m,k)/(z_litter*1e-3) .LT. litter_below_old(:,j,m,1,k)/(z_soil(1))) .AND. &
                 ((litter_above_flux(:,j,m,k) - litter_below_old(:,j,m,1,k)) .GE. min_stomate))
               litter_above(:,j,m,k) = litter_above_old(:,j,m,k) + litter_below_old(:,j,m,1,k) 
               litter_below_old_buffer(:,j,m,1,k) = -litter_below_old(:,j,m,1,k) 
               litter_below(:,j,m,1,k) = zero
            ELSEWHERE ((litter_above_old(:,j,m,k)/(z_litter*1e-3) .LT. litter_below_old(:,j,m,1,k)/(z_soil(1))) .AND. &
                 ((litter_below_old(:,j,m,1,k) - litter_above_flux(:,j,m,k)) .GT. min_stomate))
               litter_above(:,j,m,k) = litter_above_old(:,j,m,k) + litter_above_flux(:,j,m,k) 
               litter_below_old_buffer(:,j,m,1,k) =  - litter_above_flux(:,j,m,k)
               litter_below(:,j,m,1,k) = litter_below_old(:,j,m,1,k) - litter_above_flux(:,j,m,k)
            ELSEWHERE ((litter_above_old(:,j,m,k)/(z_litter*1e-3) .GT. litter_below_old(:,j,m,1,k)/(z_soil(1))) .AND. &
                 ((litter_above_flux(:,j,m,k) - litter_above_old(:,j,m,k)) .GE. min_stomate))
               litter_above(:,j,m,k) = zero 
               litter_below_old_buffer(:,j,m,1,k) = litter_above_old(:,j,m,k)
               litter_below(:,j,m,1,k) = litter_below_old(:,j,m,1,k) + litter_above_old(:,j,m,k)
            ELSEWHERE ((litter_above_old(:,j,m,k)/(z_litter*1e-3) .GT. litter_below_old(:,j,m,1,k)/(z_soil(1))) .AND. &
                 ((litter_above_old(:,j,m,k) - litter_above_flux(:,j,m,k)) .GT. min_stomate))
               litter_above(:,j,m,k) = litter_above_old(:,j,m,k) - litter_above_flux(:,j,m,k)
               litter_below_old_buffer(:,j,m,1,k) = litter_above_flux(:,j,m,k)
               litter_below(:,j,m,1,k) = litter_below_old(:,j,m,1,k) + litter_above_flux(:,j,m,k)
            ELSEWHERE
               litter_above(:,j,m,k) = litter_above_old(:,j,m,k)
               litter_below(:,j,m,1,k) = litter_below_old(:,j,m,1,k)
            ENDWHERE

            DO l= 1, nbdl-1 ! Loop over soil layers
               WHERE ((litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .LT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l))) .AND. &
                    ((litter_flux(:,j,m,l,k) - litter_below_old(:,j,m,l+1,k)) .GE. min_stomate))
                  litter_below(:,j,m,l,k) = litter_below_old(:,j,m,l,k) + litter_below_old(:,j,m,l+1,k) + litter_below_old_buffer(:,j,m,l,k)
                  litter_below_old_buffer(:,j,m,l+1,k) = -litter_below_old(:,j,m,l+1,k)
                  litter_below(:,j,m,l+1,k) = zero
               ELSEWHERE ((litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .LT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l))) .AND. &
                    ((litter_below_old(:,j,m,l+1,k) - litter_flux(:,j,m,l,k)) .GT. min_stomate))
                  litter_below(:,j,m,l,k) = litter_below_old(:,j,m,l,k) + litter_flux(:,j,m,l,k) + litter_below_old_buffer(:,j,m,l,k)
                  litter_below_old_buffer(:,j,m,l+1,k) =  - litter_flux(:,j,m,l,k)
                  litter_below(:,j,m,l+1,k) = litter_below_old(:,j,m,l+1,k) - litter_flux(:,j,m,l,k)
               ELSEWHERE ((litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .GT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l))) .AND. &
                    ((litter_flux(:,j,m,l,k) - litter_below_old(:,j,m,l,k)) .GE. min_stomate))
                  litter_below(:,j,m,l,k) = zero + litter_below_old_buffer(:,j,m,l,k)
                  litter_below_old_buffer(:,j,m,l+1,k) = litter_below_old(:,j,m,l,k)
                  litter_below(:,j,m,l+1,k) = litter_below_old(:,j,m,l+1,k) + litter_below_old(:,j,m,l,k)
               ELSEWHERE ((litter_below_old(:,j,m,l,k)/(z_soil(l)-z_soil(l-1)) .GT. litter_below_old(:,j,m,l+1,k)/(z_soil(l+1)-z_soil(l))) .AND. &
                    ((litter_below_old(:,j,m,l,k) - litter_flux(:,j,m,l,k)) .GT. min_stomate))
                  litter_below(:,j,m,l,k) = litter_below_old(:,j,m,l,k) - litter_flux(:,j,m,l,k) + litter_below_old_buffer(:,j,m,l,k)
                  litter_below_old_buffer(:,j,m,l+1,k) = litter_flux(:,j,m,l,k)
                  litter_below(:,j,m,l+1,k) = litter_below_old(:,j,m,l+1,k) + litter_flux(:,j,m,l,k)
               ELSEWHERE
                  litter_below(:,j,m,l,k) = litter_below_old(:,j,m,l,k)
                  litter_below(:,j,m,l+1,k) = litter_below_old(:,j,m,l+1,k)
               ENDWHERE
            ENDDO ! End loop over soil layers
         ENDDO ! End loop over litter pools
      ENDDO ! End loop over elements
   ENDDO ! End loop over PFT

  !! 8. calculate fraction of total soil covered by dead leaves

   CALL deadleaf (npts, veget_max, dead_leaves, deadleaf_cover)

   !! 9. Check mass balance closure

   !! 9.1 Calculate components of the mass balance
   pool_end_after(:,:,:) = zero
   pool_end(:,:,:) = zero
   IF (ld_doc) THEN
      DO m = 2, nvm
         DO i = 1,nlitt
            DO l = 1, nbdl
               pool_end(:,m,icarbon) = pool_end(:,m,icarbon) + &
                    ( litter_below(:,i,m,l,icarbon) ) * veget_max(:,m)
            ENDDO
         ENDDO
      ENDDO

      DO m = 2, nvm
         DO i = 1,nlitt
            pool_end(:,m,icarbon) = pool_end(:,m,icarbon) + &
                 ( litter_above(:,i,m,icarbon) ) * veget_max(:,m)
         ENDDO
      ENDDO


    !! 9.2 Calculate mass balance
    !  Note that soilcarbon is transfered to other pools but that the pool
    !  was not updated. We should not account for it in ::pool_end

      check_intern(:,:,:,:) = zero
      check_intern(:,:,iatm2land,icarbon) = zero 

    
      check_intern(:,:,iland2atm,icarbon) = -un * (resp_hetero_litter(:,:,iabove)+resp_hetero_litter(:,:,ibelow)+resp_hetero_flood(:,:)) * &
           veget_max(:,:) * dt
    

      DO m = 1,nvm
         DO l = 1, nbdl
            check_intern(:,m,ilat2out,icarbon) = check_intern(:,m,ilat2out,icarbon) -un * &
                 (soilcarbon_input(:,m,l,imetbel,icarbon)+soilcarbon_input(:,m,l,istrbel,icarbon)) * &
                 veget_max(:,m) * dt
         ENDDO
      ENDDO
      DO m = 1,nvm
         check_intern(:,m,ilat2out,icarbon) = check_intern(:,m,ilat2out,icarbon) -un * &
              (soilcarbon_input(:,m,1,imetabo,icarbon)+soilcarbon_input(:,m,1,istrabo,icarbon)    &
              +floodcarbon_input(:,m,iact,icarbon)+floodcarbon_input(:,m,islo,icarbon)) * &
              veget_max(:,m) * dt
      ENDDO

      check_intern(:,:,ilat2in,icarbon) = zero
    
      check_intern(:,:,ipoolchange,icarbon) = -un * (pool_end(:,:,icarbon) - &
           pool_start(:,:,icarbon))
    
      closure_intern = zero
      DO i = 1,nmbcomp
         closure_intern(:,:,icarbon) = closure_intern(:,:,icarbon) + &
              check_intern(:,:,i,icarbon)
      ENDDO

    !! 9.3 Write verdict

      IF (SUM(SUM(closure_intern(:,:,icarbon),2),1) .LT. min_stomate .AND. &
           SUM(SUM(closure_intern(:,:,icarbon),2),1) .GT. -min_stomate) THEN
         WRITE(numout,*) 'Mass balance closure in stomate_litter.f90'
      ELSE
         WRITE(numout,*) 'Error: mass balance is not closed in stomate_litter.f90'
         WRITE(numout,*) '   Difference is, ', SUM(SUM(closure_intern(:,:,icarbon),2),1)
         WRITE(numout,*) '   Difference is, ', closure_intern(:,:,icarbon)
      ENDIF
   ENDIF


   IF (printlev>=4) WRITE(numout,*) 'Leaving littercalc'

 END SUBROUTINE littercalc


!! ==============================================================================================================================\n
!! SUBROUTINE   : deadleaf
!!
!>\BRIEF        This routine calculates the deadleafcover. 
!!
!! DESCRIPTION  : It first calculates the lai corresponding to the dead leaves (LAI) using 
!! the dead leaves carbon content (DL) the specific leaf area (sla) and the 
!! maximal coverage fraction of a PFT (vegetmax) using the following equations:
!! \latexonly
!! \input{deadleaf1.tex}
!! \endlatexonly
!! \n
!! Then, the dead leaf cover (DLC) is calculated as following:\n
!! \latexonly
!! \input{deadleaf2.tex}
!! \endlatexonly
!! \n
!! 
!! RECENT CHANGE(S) : None
!!
!! MAIN OUTPUT VARIABLE: ::deadleaf_cover
!! 
!! REFERENCE(S) : None
!!
!! FLOWCHART : None
!! \n
!_ ================================================================================================================================

  SUBROUTINE deadleaf (npts, veget_max, dead_leaves, deadleaf_cover)

  !! 0. Variable and parameter declaration
    
    !! 0.1 Input variables

    INTEGER(i_std), INTENT(in)                          :: npts           !! Domain size - number of grid pixels (unitless)
    REAL(r_std), DIMENSION(npts,nvm,nlitt), INTENT(in)  :: dead_leaves    !! Dead leaves per ground unit area, per PFT, 
                                                                          !! metabolic and structural  
                                                                          !! @tex $(gC m^{-2})$ @endtex
    REAL(r_std),DIMENSION(npts,nvm),INTENT(in)          :: veget_max      !! PFT "Maximal" coverage fraction of a PFT defined in 
                                                                          !! the input vegetation map 
                                                                          !! @tex $(m^2 m^{-2})$ @endtex 
    
    !! 0.2 Output variables
    
    REAL(r_std), DIMENSION(npts), INTENT(out)           :: deadleaf_cover !! Fraction of soil covered by dead leaves over all PFTs
                                                                          !! (0-1, unitless)

    !! 0.3 Modified variables

    !! 0.4 Local variables

    REAL(r_std), DIMENSION(npts)                        :: dead_lai       !! LAI of dead leaves @tex $(m^2 m^{-2})$ @endtex
    INTEGER(i_std)                                      :: j              !! Index (unitless)
!_ ================================================================================================================================
    
  !! 1. LAI of dead leaves
  
    dead_lai(:) = zero

    DO j = 2,nvm !Loop over PFTs
       dead_lai(:) = dead_lai(:) + ( dead_leaves(:,j,imetabolic) + dead_leaves(:,j,istructural) ) * sla(j) &
            * veget_max(:,j)
    ENDDO

  !! 2. fraction of soil covered by dead leaves

    deadleaf_cover(:) = un - exp( - 0.5 * dead_lai(:) )

    IF (printlev>=4) WRITE(numout,*) 'Leaving deadleaf'

  END SUBROUTINE deadleaf


!! ================================================================================================================================
!! FUNCTION     : control_moist_func_moyano
!!
!>\BRIEF        Calculate moisture control for litter and soild C decomposition
!!
!! DESCRIPTION  : Calculate moisture control factor applied
!! to litter decomposition and to soil carbon decomposition in
!! stomate_soilcarbon.f90 using the following equation: \n
!! \latexonly
!! \input{control_moist_func1.tex}
!! \endlatexonly
!! \n
!! with M the moisture control factor and soilmoisutre, the soil moisture 
!! calculated in sechiba.
!! Then, the function is ranged between 0.25 and 1:\n
!! \latexonly
!! \input{control_moist_func2.tex}
!! \endlatexonly
!! \n
!! RECENT CHANGE(S) : None
!!
!! RETURN VALUE : ::moistfunc_result
!! 
!! REFERENCE(S) : None
!!
!! FLOWCHART : None
!! \n
!_ ================================================================================================================================
  
  FUNCTION control_moist_func_moyano (npts, moist_in,bulk_dens, clay, carbon,veget_max) RESULT (moistfunc_result)

  !! 0. Variable and parameter declaration

    !! 0.1 Input variables

    INTEGER(i_std), INTENT(in)               :: npts                !! Domain size - number of grid pixel (unitless)
    REAL(r_std), DIMENSION(npts), INTENT(in) :: moist_in            !! relative humidity (unitless)
    REAL(r_std), DIMENSION(npts), INTENT(in)                    :: clay               !! Clay fraction (unitless, 0-1) 
    REAL(r_std), DIMENSION(npts), INTENT(in)                    :: bulk_dens          !! Variable local of bulk density for the moment must change in the futur (kg m-3)
    REAL(r_std), DIMENSION(npts,ncarb,nvm,nbdl), INTENT(in)     :: carbon             !! Soil carbon pools: active, slow, or passive, \f$(gC m^{2})$\f
    REAL(r_std),DIMENSION(npts,nvm),INTENT(in)                  :: veget_max          !! Maximum fraction of vegetation type including 
                                                                                      !! non-biological fraction (unitless)

    !! 0.2 Output variables

    REAL(r_std), DIMENSION(npts)          :: moistfunc_result    !! Moisture control factor (0-1, unitless)

    !! 0.3 Modified variables

    !! 0.4 Local variables
   REAL(r_std), DIMENSION(npts)           :: total_soc              !! Total soil organic carbon for a grid pixel (g-C g-1 soil)
   INTEGER(i_std)                         :: k,i,j                  !! Indices
   INTEGER, PARAMETER                     :: nummoist = 120         !! We used numoist then divived by 100. because to calculate the function
                                                                    !! we need to predict PRsr for all the possible soil moisture values
   REAL(r_std), DIMENSION(npts,nummoist)  :: PRsr                   !! Proportional response or soil respiration at moist_in 
   REAL(r_std), DIMENSION(npts,nummoist)  :: SR                     !! Soil respiration at moist_in to facilitate the understanding regarding 
                                                                    !! the Moyano et al paper we keep the same nomenclature than in the paper
                                                                    !! but **BE CAREFUL** it does not correspond to the variable resp_hetero_soil
                                                                    !! or resp_hetero_litter in ORCHIDEE
   REAL(r_std), DIMENSION(0:nbdl)   :: z_soil                 !! Soil levels (m)
!$OMP THREADPRIVATE(z_soil)
   REAL(r_std), DIMENSION(npts,nummoist)  :: moistfunc              !! An intermediate variable to calculate the Moisture control factor
   INTEGER(i_std)                         :: ind                    !! A local variable to know what is the index of the max value for the statistical function
   INTEGER(i_std), DIMENSION(npts)        :: ind_i                  !! A local variable to know what is the index of the max value for the 
                                                                    !! statistical function at each points
   REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:)    :: moistfunc_below_1 !! A allocatable variable to store the value of moistfunc that are below the max value
                                                                       !! to then rescale at is it done in Moyano et al. 2012 BG
!$OMP THREADPRIVATE(moistfunc_below_1)
   INTEGER(i_std)                         :: ier                    !! Check errors in netcdf call (unitless)
   LOGICAL                                :: l_error                !! Check errors in netcdf call

!_ ================================================================================================================================

   !! 1.3 soil levels
   z_soil(0) = zero
   z_soil(1:nbdl) = diaglev(1:nbdl)

   ! In Moyano et al. 2012 BG, the function obtained is based on soil 
   ! incubations were litter is generally manually removed, therefore we used 
   ! only the variable carbon corresponding to the soil organic carbon even
   ! if some litter (metabolic litter) is not easy to remove manually since it
   ! corresponds to roots exudates. Therefore, we probably underestimate a bit
   ! the soil organic carbon compared to what is used in Moyano et al., 2012
   ! BG.

   PRsr(:,:) = zero
   SR(:,:) = zero
   moistfunc(:,:) = zero
   l_error = .FALSE.

   total_soc(:) = zero
   DO k = 1,nvm
      DO j = 1,nbdl
         total_soc(:) =  total_soc(:) + &
              ((carbon(:,iactive,k,j)*veget_max(:,k) + &
              carbon(:,islow,k,j)*veget_max(:,k) + &
              carbon(:,ipassive,k,j)*veget_max(:,k))* &
              ((z_soil(j)-z_soil(j-1))/z_soil(nbdl))) / &
              (bulk_dens(:) ) 
      ENDDO
   ENDDO

   DO k = 1,nummoist

      PRsr(:,k) = 1.11066-0.83344*(k/100.) + 1.48095*((k/100.)**2) - 1.02959*((k/100.)**3) + 0.07995*clay(:) + 1.27892*total_soc(:)
      
   ENDDO

   SR(:,1)=PRsr(:,1)
   DO k=2,nummoist
      SR(:,k)= SR(:,k-1)* PRsr(:,k)
   ENDDO

   DO i = 1,npts
      DO k= 1, nummoist
         moistfunc(i,k) = SR(i,k)/MAXVAL(SR(i,:))
      ENDDO
   ENDDO

   ind_i(:) = MAXLOC(moistfunc(:,:),DIM=2)
   
   DO i = 1,npts
      ind=ind_i(i)

      ALLOCATE(moistfunc_below_1(npts,ind),stat=ier)
      l_error = l_error .OR. (ier /= 0)
      IF (l_error) THEN
         WRITE(numout,*) 'Memory allocation error for moistfunc_below_1. '
      ENDIF

      DO k= 1,ind
         moistfunc_below_1(i,k)= moistfunc(i,k)
      ENDDO

      DO k= 1,ind           
         moistfunc(i,k)=moistfunc(i,k) - MINVAL(moistfunc_below_1(i,:))
      ENDDO

      DO k= 1,ind
         moistfunc_below_1(i,k)= moistfunc(i,k)
      ENDDO

      DO k=1,ind
         moistfunc(i,k)=moistfunc(i,k)/MAXVAL(moistfunc_below_1(i,:))
      ENDDO

      IF (ALLOCATED(moistfunc_below_1)) DEALLOCATE(moistfunc_below_1)

   ENDDO
   DO i=1,npts
      IF (NINT(moist_in(i)*100.) .GT. zero) THEN
         moistfunc_result(i) = moistfunc(i,NINT(moist_in(i)*100.))
      ELSE
         moistfunc_result(i) = zero
      ENDIF
   ENDDO

 END FUNCTION control_moist_func_moyano

!! ================================================================================================================================
!! FUNCTION     : control_moist_func
!!
!>\BRIEF        Calculate moisture control for litter and soild C decomposition
!!
!! DESCRIPTION  : Calculate moisture control factor applied
!! to litter decomposition and to soil carbon decomposition in
!! stomate_soilcarbon.f90 using the following equation: \n
!! \latexonly
!! \input{control_moist_func1.tex}
!! \endlatexonly
!! \n
!! with M the moisture control factor and soilmoisutre, the soil moisture 
!! calculated in sechiba.
!! Then, the function is ranged between 0.25 and 1:\n
!! \latexonly
!! \input{control_moist_func2.tex}
!! \endlatexonly
!! \n
!! RECENT CHANGE(S) : None
!!
!! RETURN VALUE : ::moistfunc_result
!! 
!! REFERENCE(S) : None
!!
!! FLOWCHART : None
!! \n
!_ ================================================================================================================================

  FUNCTION control_moist_func (npts, moist_in) RESULT (moistfunc_result)

  !! 0. Variable and parameter declaration

    !! 0.1 Input variables

    INTEGER(i_std), INTENT(in)               :: npts                !! Domain size - number of grid pixel (unitless)
    REAL(r_std), DIMENSION(npts), INTENT(in) :: moist_in            !! relative humidity (unitless)


    !! 0.2 Output variables

    REAL(r_std), DIMENSION(npts)          :: moistfunc_result    !! Moisture control factor (0-1, unitless)

    !! 0.3 Modified variables


!_ ================================================================================================================================


          moistfunc_result(:) = -moist_coeff(1) * moist_in(:) * moist_in(:) + moist_coeff(2)* moist_in(:) - moist_coeff(3)
          moistfunc_result(:) = MAX( 0.25_r_std, MIN( un, moistfunc_result(:) ) )
    
  END FUNCTION control_moist_func

!! ================================================================================================================================
!! FUNCTION     : control_temp_func
!!
!>\BRIEF        Calculate temperature control for litter and soild C decomposition
!!
!! DESCRIPTION  : Calculate temperature control factor applied
!! to litter decomposition and to soil carbon decomposition in
!! stomate_soilcarbon.f90 using the following equation: \n
!! \latexonly
!! \input{control_temp_func1.tex}
!! \endlatexonly
!! \n
!! with T the temperature control factor, temp the temperature in Kelvin of 
!! the air (for aboveground litter) or of the soil (for belowground litter 
!! and soil)
!! Then, the function is limited in its maximal range to 1:\n
!! \latexonly
!! \input{control_temp_func2.tex}
!! \endlatexonly
!! \n
!! RECENT CHANGE(S) : None
!!
!! RETURN VALUE: ::tempfunc_result
!!
!! REFERENCE(S) : None
!!
!! FLOWCHART : None
!! \n
!_ ================================================================================================================================

  FUNCTION control_temp_func (npts, temp_in,tau) RESULT (tempfunc_result)

  !! 0. Variable and parameter declaration
    
    !! 0.1 Input variables
    INTEGER(i_std), INTENT(in)                 :: npts            !! Domain size - number of land pixels (unitless)
    REAL(r_std), DIMENSION(npts), INTENT(in)   :: temp_in         !! Temperature (K)
    REAL(r_std), DIMENSION(npts), INTENT(in)   :: tau             !! residence time of the pool considered (days)
    !! 0.2 Output variables
    REAL(r_std), DIMENSION(npts)               :: tempfunc_result !! Temperature control factor (0-1, unitless)

    !! 0.3 Modified variables

    !! 0.4 Local variables
    REAL(r_std), DIMENSION(npts)               :: Ea              !! Activation Energy 
    REAL(r_std), DIMENSION(npts)               :: Q10_soil        !! Q10 values calculated based on Ea
!_ ================================================================================================================================

    ! This relationship is based on a reanalysis of the data published by Lefèvre et al. 2014 in global change biology 
    ! We use MAX(cent,(temp_in(:)-ZeroCelsius)*(temp_in(:)-ZeroCelsius)))) because for values between -10 and 10 degC
    ! the Q10 values are close to infinity 
    Ea(:)=a_term_Q10_soil*tau + b_term_Q10_soil
    Q10_soil(:)=exp((dix*Ea(:))/(RR*MAX(cent,(temp_in(:)-ZeroCelsius)*(temp_in(:)-ZeroCelsius))))
    tempfunc_result(:) = exp( log(Q10_soil(:)) * ( temp_in(:) - (ZeroCelsius+tsoil_ref)) / Q10 )
    tempfunc_result(:) = MIN( un, tempfunc_result(:) )
    WHERE (temp_in(:) .LT. zero) 
        tempfunc_result(:) = zero
    ENDWHERE
  END FUNCTION control_temp_func

END MODULE stomate_litter
