










! =================================================================================================================================
! PROGRAM       : forcesoil
!
! CONTACT	: orchidee-help _at_ ipsl.jussieu.fr
!
! LICENCE      	: IPSL (2006). This software is governed by the CeCILL licence see ORCHIDEE/ORCHIDEE_CeCILL.LIC
!
!>\BRIEF        This subroutine runs the soilcarbon submodel using specific initial conditions 
!! and driving variables in order to obtain soil carbon stocks closed to the steady-state values 
!! quicker than when using the ''full'' ORCHIDEE.  
!!	
!!\n DESCRIPTION: None
!! This subroutine computes the soil carbon stocks by calling the soilcarbon routine at each time step. \n
!! The aim is to obtain soil carbon stocks closed to the steady-state values and ultimately to create  
!! an updated stomate restart file for the stomate component. The state variables of the subsystem are the clay content 
!! (fixed value) and the soil carbon stocks. Initial conditions for the state variables are read in an  
!! input stomate restart file. Driving variables are Soil carbon input, Water and Temperature stresses on 
!! Organic Matter decomposition. Driving variables are read from a specific forcing file produced by a former run of ORCHIDEE
!! (SECHIBA+STOMATE). \n 
!! The FORCESOIL program first consists in reading a set of input files, allocating variables and 
!! preparing output stomate restart file. \n                                                             
!! Then, a loop over time is performed in which the soilcarbon routine is called at each time step. \n
!! Last, final values of the soil carbon stocks are written into the output stomate restart file. \n
!! No flag is associated with the use of the FORCESOIL program. \n
!!
!! RECENT CHANGE(S): None
!!
!! REFERENCE(S) : None    
!!
!! FLOWCHART    : None
!!
!! SVN		:
!! $HeadURL: $ 
!! $Date: $
!! $Revision: $
!! \n
!_ =================================================================================================================================

PROGRAM forcesoil
 
  USE netcdf
  !-
  USE defprec
  USE constantes
  USE constantes_soil
  USE constantes_mtc
!  USE pft_parameters 
  USE stomate_data
  USE ioipsl_para
  USE mod_orchidee_para
  USE stomate_soilcarbon
  USE vertical_soil
  USE control
!  USE constantes_var  

!-
  IMPLICIT NONE
  !-
  CHARACTER(LEN=80)                          :: sto_restname_in,sto_restname_out
  INTEGER(i_std)                             :: iim,jjm                 !! Indices (unitless)

  INTEGER(i_std),PARAMETER                   :: llm = 1                 !! Vertical Layers (requested by restini routine) (unitless)
  INTEGER(i_std)                             :: kjpindex                !! Domain size (unitless)

  INTEGER(i_std)                             :: itau_dep,itau_len       !! Time step read in the restart file (?) 
                                                                        !! and number of time steps of the simulation (unitless) 
  CHARACTER(LEN=30)                          :: time_str                !! Length of the simulation (year)
  REAL(r_std)                                :: dt_files                !! time step between two successive itaus (?) 
                                                                        !! (requested by restini routine) (seconds)
  REAL(r_std)                                :: date0                   !! Time at which itau = 0 (requested by restini routine) (?)
  INTEGER(i_std)                             :: rest_id_sto             !! ID of the input restart file (unitless)
  CHARACTER(LEN=20), SAVE                    :: thecalendar = 'noleap'  !! Type of calendar defined in the input restart file 
                                                                        !! (unitless)
  !-
  CHARACTER(LEN=100)                         :: Cforcing_name           !! Name of the forcing file (unitless)
  INTEGER                                    :: Cforcing_id             !! ID of the forcing file (unitless)
  INTEGER                                    :: v_id                    !! ID of the variable 'Index' stored in the forcing file 
                                                                        !! (unitless)
  REAL(r_std)                                :: dt_forcesoil            !! Time step at which soilcarbon routine is called (days) 
  INTEGER                                    :: nparan                  !! Number of values stored per year in the forcing file 
                                                                        !! (unitless)
  INTEGER                                    :: nbyear
  INTEGER(i_std),DIMENSION(:),ALLOCATABLE    :: indices                 !! Grid Point Index used per processor (unitless)
  INTEGER(i_std),DIMENSION(:),ALLOCATABLE    :: indices_g               !! Grid Point Index for all processor (unitless)
  REAL(r_std),DIMENSION(:),ALLOCATABLE       :: x_indices_g             !! Grid Point Index for all processor (unitless)
  REAL(r_std),DIMENSION(:,:),ALLOCATABLE     :: lon, lat                !! Longitude and Latitude of each grid point defined 
                                                                        !! in lat/lon (2D) (degrees)
  REAL(r_std),DIMENSION(llm)                 :: lev                     !! Number of level (requested by restini routine) (unitless)


  INTEGER                                    :: i,k,j,l,m,iatt,iv,iyear !! counters (unitless)
  REAL(r_std)                                :: soil_depth              !! Soil depth (m)
  CHARACTER(LEN=80)                          :: var_name
  CHARACTER(LEN=8000)                        :: taboo_vars              !! string used for storing the name of the variables 
                                                                        !! of the stomate restart file that are not automatically 
                                                                        !! duplicated from input to output restart file (unitless)
  REAL(r_std),DIMENSION(1)                   :: xtmp                    !! scalar read/written in restget/restput routines (unitless)
  INTEGER(i_std),PARAMETER                   :: nbvarmax=800            !! maximum # of variables assumed in the stomate restart file 
                                                                        !! (unitless)
  INTEGER(i_std)                             :: nbvar                   !! # of variables effectively present 
                                                                        !! in the stomate restart file (unitless)
  CHARACTER(LEN=50),DIMENSION(nbvarmax)      :: varnames                !! list of the names of the variables stored 
                                                                        !! in the stomate restart file (unitless)
  INTEGER(i_std)                             :: varnbdim                !! # of dimensions of a given variable 
                                                                        !! of the stomate restart file
  INTEGER(i_std),PARAMETER                   :: varnbdim_max=20         !! maximal # of dimensions assumed for any variable 
                                                                        !! of the stomate restart file 
  INTEGER,DIMENSION(varnbdim_max)            :: vardims                 !! length of each dimension of a given variable 
                                                                        !! of the stomate restart file
  LOGICAL                                    :: l1d                     !! boolean : TRUE if all dimensions of a given variable 
                                                                        !! of the stomate restart file are of length 1 (ie scalar) 
                                                                        !! (unitless)
  REAL(r_std),DIMENSION(:,:),ALLOCATABLE     :: var_3d                  !! matrix read/written in restget/restput routines (unitless)
  REAL(r_std)                                :: x_tmp                   !! temporary variable used to store return value 
                                                                        !! from nf90_get_att (unitless)
  CHARACTER(LEN=10)  :: part_str                                        !! string suffix indicating the index of a PFT 

  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:)  :: soil_ph_g               !! Soil pH (0-14, pH unit)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:)  :: clay_g                  !! clay fraction (nbpglo) (unitless)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:)  :: bulk_dens_g             !! soil bulk density (nbpglo) (g cm-3)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:):: soiltile_g              !! Fraction of each soil tile (0-1, unitless)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:):: veget_max_g             !! PFT "Maximal" coverage fraction of a PFT defined in 
                                                                        !! the input vegetation map 
                                                                        !! @tex $(m^2 m^{-2})$ @endtex, parallel computing  
  REAL(r_std),DIMENSION(:,:,:,:,:,:),ALLOCATABLE :: soilcarbon_input_g  !! soil carbon input (nbpglob,npool,nvm,time) 
                                                                        !! (\f$gC m^{-2} dt_forcesoil^{-1}\f$) 
  REAL(r_std),DIMENSION(:,:,:,:,:),ALLOCATABLE :: floodcarbon_input_g  !! soil carbon input (nbpglob,npool,nvm,time)
                                                                        !! (\f$gC m^{-2} dt_forcesoil^{-1}\f$)
  REAL(r_std),DIMENSION(:,:,:,:),ALLOCATABLE   :: control_temp_soil_g     !! Temperature control (nbp_glo,nbdl,time) on OM decomposition 
                                                                        !! (unitless)
  REAL(r_std),DIMENSION(:,:,:,:),ALLOCATABLE   :: control_moist_soil_g    !! Moisture control (nbp_glo,nbdl,time) on OM decomposition 
                                                                        !! ?? Should be defined per PFT as well (unitless)
  REAL(r_std),DIMENSION (:,:,:), ALLOCATABLE   :: moist_soil_g          !! soil moisture content \f($m^3 \times m^3$)\f
  REAL(r_std),DIMENSION (:,:,:,:), ALLOCATABLE :: soil_mc_g             !! soil moisture content per soil type \f($m^3 \times m^3$)\f
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:) :: floodout_g             !! flux out of floodplains
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: wat_flux0_g          !! Water flux in the first soil layers exported for soil C calculations
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:,:):: wat_flux_g          !! Water flux in the soil layers exported for soil C calculations
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) ::runoff_per_soil_g     !! Runoff per soil type [mm]
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) ::drainage_per_soil_g   !! Drainage per soil type [mm]
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: DOC_to_topsoil_g     !! DOC inputs to top of the soil column, from reinfiltration on
                                                                        !! floodplains and from irrigation
                                                                        !! @tex $(gC m^{-2} day{-1})$ @endtex
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: DOC_to_subsoil_g     !! DOC inputs to bottom of the soil column, from returnflow
                                                                        !! in swamps and lakes
                                                                        !! @tex $(gC m^{-2} day{-1})$ @endtex
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:)   :: flood_frac_g         !! DOC inputs to bottom of the soil column, from returnflow
!! TF-DOC
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: precip2canopy_g      !! Precipitation onto the canopy
!$OMP THREADPRIVATE(precip2canopy_g)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: precip2ground_g      !! Precipitation not intercepted by canopy
!$OMP THREADPRIVATE(precip2ground_g)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: canopy2ground_g      !! Water flux from canopy to the ground
!$OMP THREADPRIVATE(canopy2ground_g)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: interception_storage_g !! Storage of DOC and soluble OC in canopy
!$OMP THREADPRIVATE(interception_storage_g)                                  

  REAL(r_std),DIMENSION(:,:,:,:),ALLOCATABLE :: carbon_g                !! Soil carbon stocks (nbp_glo,ncarb,nvm) (\f$gC m^{-2}\f$)
  REAL(r_std), DIMENSION(:,:,:,:,:,:), ALLOCATABLE :: DOC_g             !! Dissolved Organic Carbon in soil
                                                                        !! The unit is given by m^3 of
                                                                        !! ground @tex $(gC m{-2}) $ @endtex
  REAL(r_std), ALLOCATABLE ::litter_above_g(:,:,:,:,:)                  !! Metabolic and structural litter, below ground
                                                                        !! The unit is given by m^2 of
                                                                        !! ground @tex $(gCi m{-2})$ @endtex
  REAL(r_std), ALLOCATABLE ::litter_below_g(:,:,:,:,:,:)                !! Metabolic and structural litter, below ground
                                                                        !! The unit is given by m^2 of
                                                                        !! ground @tex $(gC m^{-2})$ @endtex
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: lignin_struc_above_g !! Ratio Lignine/Carbon in structural litter for above
                                                                        !! ground compartments (unitless)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:,:):: lignin_struc_below_g   !! Ratio Lignine/Carbon in structural litter for below
                                                                        !! ground compartments (unitless)
  REAL(r_std),ALLOCATABLE :: clay(:)                                    !! clay fraction (nbp_loc) (unitless)
  REAL(r_std),ALLOCATABLE :: soil_ph(:)                                 !! Soil pH (0-14 pH units)
  REAL(r_std),ALLOCATABLE :: poor_soils(:)                              !! Poor soils (0-1)
  REAL(r_std),ALLOCATABLE :: bulk_dens(:)                               !! soil bulk density (nbpglo) (g cm-3)
  REAL(r_std),ALLOCATABLE :: soilcarbon_input(:,:,:,:,:,:)              !! soil carbon input (nbp_loc,npool,nvm,time) 
                                                                        !! water @tex $(gC m{-2} of water)$ @endte

  REAL(r_std),ALLOCATABLE :: floodcarbon_input(:,:,:,:,:)               !! soil carbon input (nbp_loc,npool,nvm,time)
                                                                        !! water @tex $(gC m{-2} of water)$ @endte
  REAL(r_std),ALLOCATABLE :: control_temp_soil(:,:,:,:)                 !! Temperature control (nbp_loc,nbdl,time) on OM decomposition 
                                                                        !! (unitless)
  REAL(r_std),ALLOCATABLE :: control_moist_soil(:,:,:,:)                 !! Moisture control (nbp_loc,nbdl,time) on OM decomposition 
                                                                        !! ?? Should be defined per PFT as well (unitless)

  REAL(r_std),ALLOCATABLE :: carbon(:,:,:,:)                            !! Soil carbon stocks (nbp_loc,ncarb,nvm) (\f$gC m^{-2}\f$)
  REAL(r_std), ALLOCATABLE ::litter_above(:,:,:,:,:)                    !! Metabolic and structural litter, below ground
                                                                        !! The unit is given by m^2 of
                                                                        !! ground @tex $(gCi m{-2})$ @endtex
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: lignin_struc_above   !! Ratio Lignine/Carbon in structural litter for above
                                                                        !! ground compartments (unitless)
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:,:):: lignin_struc_below   !! Ratio Lignine/Carbon in structural litter for below
                                                                        !! ground compartments (unitless)
  REAL(r_std),ALLOCATABLE :: veget_max(:,:)                             !! PFT "Maximal" coverage fraction of a PFT defined in 
                                                                        !! the input vegetation map 
                                                                        !! @tex $(m^2 m^{-2})$ @endtex  
  REAL(r_std), ALLOCATABLE ::litter_below(:,:,:,:,:,:)                  !! Metabolic and structural litter, below ground
                                                                        !! The unit is given by m^2 of
                                                                        !! ground @tex $(gC m^{-2})$ @endtex
  REAL(r_std),DIMENSION(:,:),ALLOCATABLE     :: resp_hetero_soil        !! Heterotrophic respiration (\f$gC m^{-2} dt_forcesoil^{-1}\f$) 
                                                                        !! (requested by soilcarbon routine but not used here)
  REAL(r_std),DIMENSION(:,:),ALLOCATABLE     :: resp_flood_soil         !! Heterotrophic respiration (\f$gC m^{-2} dt_forcesoil^{-1}\f$)
                                                                        !! (requested by soilcarbon routine but not used here) 
  REAL(r_std), DIMENSION(:,:), ALLOCATABLE   :: soilhum                 !! Daily soil humidity of each soil layer 
                                                                        !! (unitless) 
  REAL(r_std),DIMENSION (:,:,:), ALLOCATABLE   :: moist_soil            !! soil moisture content \f($m^3 \times m^3$)\f
  REAL(r_std),DIMENSION (:,:,:,:), ALLOCATABLE :: soil_mc               !! soil moisture content \f($m^3 \times m^3$)\f
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:) :: floodout               !! flux out of floodplains
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: wat_flux0            !! Water flux in the first soil layers exported for soil C calculations
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:,:):: wat_flux            !! Water flux in the soil layers exported for soil C calculations
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) ::runoff_per_soil       !! Runoff per soil type [mm]
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) ::drainage_per_soil     !! Drainage per soil type [mm]
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: DOC_to_topsoil       !! DOC inputs to top of the soil column, from reinfiltration on
                                                                        !! floodplains and from irrigation
                                                                        !! @tex $(gC m^{-2} day{-1})$ @endtex
  REAL(r_std),ALLOCATABLE,SAVE,DIMENSION(:,:,:) :: DOC_to_subsoil       !! DOC inputs to bottom of the soil column, from returnflow
                                                                        !! in swamps and lakes
                                                                        !! @tex $(gC m^{-2} day{-1})$ @endtex
  !! TF-DOC, maybe put all togetehr into one matrix
  REAL(r_std), DIMENSION(:,:,:,:), ALLOCATABLE :: dry_dep_canopy        !! Increase in canopy storage of soluble OC & DOC
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std), DIMENSION(:,:,:,:), ALLOCATABLE :: DOC_precip2canopy     !! Wet deposition of DOC onto canopy
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std), DIMENSION(:,:,:,:), ALLOCATABLE :: DOC_precip2ground     !! Wet deposition of DOC not intecepted by canopy
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std), DIMENSION(:,:,:,:), ALLOCATABLE :: DOC_canopy2ground     !! Wet deposition of DOC not intecepted by canopy
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std), DIMENSION(:,:,:,:), ALLOCATABLE :: DOC_infil             !! Wet deposition of DOC infiltrating into the ground
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std), DIMENSION(:,:,:,:), ALLOCATABLE :: DOC_noinfil           !! Wet deposition of DOC going into surface runoff
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std), DIMENSION(:,:,:), ALLOCATABLE :: interception_storage    !! Storage of soluble OC attached to canopy
                                                                        !! @tex $(gC.m^{-2})$ @endtex
  REAL(r_std),  DIMENSION(:,:,:), ALLOCATABLE  :: precip2canopy         !! Precipitation onto the canopy
                                                                        !! @tex $(mm.dt^{-1})$ @endtex
  REAL(r_std),  DIMENSION(:,:,:), ALLOCATABLE  :: precip2ground         !! Precipitation not intercepted by canopy
                                                                        !! @tex $(mm.dt^{-1})$ @endtex
  REAL(r_std),  DIMENSION(:,:,:), ALLOCATABLE  :: canopy2ground         !! Water flux from canopy to the ground
                                                                        !! @tex $(mm.dt^{-1})$ @endtex
  REAL(r_std),  DIMENSION(:,:,:,:,:), ALLOCATABLE  :: biomass           !! Biomass in
                                                                        !! @tex $(kg.m^{-2})$ @endtex
  REAL(r_std),DIMENSION (:,:), ALLOCATABLE    :: soiltile               !! Fraction of each soil tile (0-1, unitless)
  REAL(r_std), DIMENSION(:,:,:,:,:,:), ALLOCATABLE :: DOC               !! Dissolved Organic Carbon in soil
                                                                        !! The unit is given by m^3 of
                                                                        !! ground @tex $(gC m{-2}) $ @endtex
  REAL(r_std), DIMENSION(:,:,:,:,:),   ALLOCATABLE :: DOC_EXP           !! Dissolved Organic Carbon in soil
                                                                        !! The unit is given by m^3 of
                                                                        !! water @tex $(gCm {-3}) $ @endtex
  REAL(r_std),  DIMENSION(:,:), ALLOCATABLE    :: flood_frac            !! Flooded fraction of grid box      

  INTEGER(i_std)                             :: ier,iret                !! Used for hangling errors 

  CHARACTER(LEN=30) :: temp_name 
  LOGICAL :: debug                                                      !! boolean used for printing messages
  LOGICAL :: l_error                                                    !! boolean for memory allocation

!_ =================================================================================================================================
 
  CALL Init_orchidee_para
  CALL init_timer

!-
  debug = .FALSE.
  CALL getin_p('DEBUG_INFO',debug)
  !!- 
  !! 1. Initialisation stage
  !! Reading a set of input files, allocating variables and preparing output restart file.     
  !!-
  ! Define restart file name
  ! for reading initial conditions (sto_restname_in var) and for writting final conditions (sto_restname_out var). 
  ! User values are used if present in the .def file.
  ! If not present, default values (stomate_start.nc and stomate_rest_out.c) are used.
  !-
  IF (is_root_prc) THEN
     sto_restname_in = 'stomate_start.nc'
     CALL getin ('STOMATE_RESTART_FILEIN',sto_restname_in)
     WRITE(numout,*) 'STOMATE INPUT RESTART_FILE: ',TRIM(sto_restname_in)
     sto_restname_out = 'stomate_rest_out.nc'
     CALL getin ('STOMATE_RESTART_FILEOUT',sto_restname_out)
     WRITE(numout,*) 'STOMATE OUTPUT RESTART_FILE: ',TRIM(sto_restname_out)
     !-
     ! Open the input file and Get some Dimension and Attributes ID's 
     !-
     iret = NF90_OPEN (sto_restname_in, NF90_NOWRITE, rest_id_sto)
     iret = NF90_INQUIRE_DIMENSION (rest_id_sto,1,len=iim_g)
     iret = NF90_INQUIRE_DIMENSION (rest_id_sto,2,len=jjm_g)
     iret = NF90_INQ_VARID (rest_id_sto, "time", iv)
     iret = NF90_GET_ATT (rest_id_sto, iv, 'calendar',thecalendar)
     iret = NF90_CLOSE (rest_id_sto)
     i=INDEX(thecalendar,ACHAR(0))
     IF ( i > 0 ) THEN
        thecalendar(i:20)=' '
     ENDIF
     !-
     ! Allocate longitudes and latitudes
     !-
     ALLOCATE (lon(iim_g,jjm_g))
     ALLOCATE (lat(iim_g,jjm_g))
     lon(:,:) = zero
     lat(:,:) = zero
     lev(1)   = zero
     !-
     CALL restini &
          & (sto_restname_in, iim_g, jjm_g, lon, lat, llm, lev, &
          &  sto_restname_out, itau_dep, date0, dt_files, rest_id_sto)
  ENDIF
  CALL control_initialize(dt_files)
  CALL bcast(date0)
  CALL bcast(thecalendar)
  WRITE(numout,*) "calendar = ",thecalendar
  !-
  ! calendar
  !-
  CALL ioconf_calendar (thecalendar)
  CALL ioget_calendar  (one_year,one_day)
  CALL ioconf_startdate(date0)
  !
  !! For master process only
  !
  IF (is_root_prc) THEN
     !-
     ! define forcing file's name (Cforcing_name var)
     ! User value is used if present in the .def file
     ! If not, default (NONE) is used
     !-
     Cforcing_name = 'NONE'
     CALL getin ('STOMATE_CFORCING_NAME',Cforcing_name)
     !-
     ! Open FORCESOIL's forcing file to read some basic info (dimensions, variable ID's)
     ! and allocate variables.
     !-
     iret = NF90_OPEN (TRIM(Cforcing_name),NF90_NOWRITE,Cforcing_id)
     IF (iret /= NF90_NOERR) THEN
        CALL ipslerr (3,'forcesoil', &
             &        'Could not open file : ', &
             &          Cforcing_name,'(Do you have forget it ?)')
     ENDIF
     !-
     ! Total Domain size is stored in nbp_glo variable
     !-
     ier = NF90_GET_ATT (Cforcing_id,NF90_GLOBAL,'kjpindex',x_tmp)
     nbp_glo = NINT(x_tmp)
     !-
     ! Number of values stored per year in the forcing file is stored in nparan var.
     !-
     ier = NF90_GET_ATT (Cforcing_id,NF90_GLOBAL,'nparan',x_tmp)
     nparan = NINT(x_tmp)
     ier = NF90_GET_ATT (Cforcing_id,NF90_GLOBAL,'nbyear',x_tmp)
     nbyear = NINT(x_tmp)
     !-
     ALLOCATE (indices_g(nbp_glo))
     ALLOCATE (clay_g(nbp_glo))
     ALLOCATE (soil_ph_g(nbp_glo))
     ALLOCATE (bulk_dens_g(nbp_glo))
     ALLOCATE (soiltile_g(nbp_glo,nstm))
     ALLOCATE (veget_max_g(nbp_glo,nvm))
     !-
     ALLOCATE (x_indices_g(nbp_glo),stat=ier)
     ier = NF90_INQ_VARID (Cforcing_id,'index',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,x_indices_g)
     indices_g(:) = NINT(x_indices_g(:))
     WRITE(numout,*) mpi_rank,"indices globaux : ",indices_g
     DEALLOCATE (x_indices_g)
     !-
     ier = NF90_INQ_VARID (Cforcing_id,'clay',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,clay_g)
     ier = NF90_INQ_VARID (Cforcing_id,'soil_ph',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,soil_ph_g)
     ier = NF90_INQ_VARID (Cforcing_id,'bulk_dens',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,bulk_dens_g)
     ier = NF90_INQ_VARID (Cforcing_id,'soiltile',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,soiltile_g)
     ier = NF90_INQ_VARID (Cforcing_id,'veget_max',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,veget_max_g)
     !-
     ! time step of forcesoil program (in days)
     !-
     dt_forcesoil = one_year / FLOAT(nparan)
     WRITE(numout,*) 'time step (d): ',dt_forcesoil
     WRITE(numout,*) 'nparan: ',nparan
     WRITE(numout,*) 'nbyear: ',nbyear    
     !-
     ! read and write the variables in the output restart file we do not modify within the Forcesoil program
     ! ie all variables stored in the input restart file except those stored in taboo_vars
     !-
     taboo_vars ='$lon$ $lat$ $lev$ $nav_lon$ $nav_lat$ $nav_lev$ $time$ $time_steps$ '// &
          &             '$dt_days$ $date$'
     !     &             '$day_counter$ $dt_days$ $date$ '
     !-
     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z01_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z02_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z03_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z04_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z05_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z06_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z07_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z08_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z09_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z10_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,nvm
        WRITE(part_str,'(I2)') m
        IF (m < 10) part_str(1:1) = '0'
        temp_name = '$carbon_z11_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z1_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z2_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z3_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z4_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z5_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z6_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z7_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z8_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z9_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z10_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$freedoc_z11_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z1_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z2_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z3_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z4_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z5_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z6_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z7_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z8_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z9_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z10_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     DO m = 1,npool
        WRITE(part_str,'(I1)') m
        temp_name = '$adsdoc_z11_'//part_str(1:LEN_TRIM(part_str))//'$'
        taboo_vars = TRIM(taboo_vars)//' '//TRIM(temp_name)
     ENDDO

     !-
     CALL ioget_vname(rest_id_sto, nbvar, varnames)
     !-
     ! read and write some special variables (1D or variables that we need)
     !-
     !var_name = 'day_counter'
     !CALL restget (rest_id_sto, var_name, 1, 1, 1, itau_dep, .TRUE., xtmp)
     !CALL restput (rest_id_sto, var_name, 1, 1, 1, itau_dep, xtmp)
     !-
     var_name = 'dt_days'
     CALL restget (rest_id_sto, var_name, 1, 1, 1, itau_dep, .TRUE., xtmp)
     CALL restput (rest_id_sto, var_name, 1, 1, 1, itau_dep, xtmp)
     !-
     var_name = 'date'
     CALL restget (rest_id_sto, var_name, 1, 1, 1, itau_dep, .TRUE., xtmp)
     CALL restput (rest_id_sto, var_name, 1, 1, 1, itau_dep, xtmp)
     !-

     DO iv=1,nbvar
        !-- check if the variable is to be written here
        IF (INDEX(taboo_vars,'$'//TRIM(varnames(iv))//'$') == 0 ) THEN
           !---- get variable dimensions, especially 3rd dimension
           CALL ioget_vdim &
                &      (rest_id_sto, varnames(iv), varnbdim_max, varnbdim, vardims)
           l1d = ALL(vardims(1:varnbdim) == 1)
           !---- read it
           IF (l1d) THEN
              CALL restget &
                   &        (rest_id_sto, TRIM(varnames(iv)), 1, vardims(3), &
                   &         1, itau_dep, .TRUE., xtmp)
           ELSE
              ALLOCATE( var_3d(nbp_glo,vardims(3)), stat=ier)
              IF (ier /= 0) STOP 'ALLOCATION PROBLEM'
              !----

              CALL restget &
                   &        (rest_id_sto, TRIM(varnames(iv)), nbp_glo, vardims(3), &
                   &         1, itau_dep, .TRUE., var_3d, 'gather', nbp_glo, indices_g)
           ENDIF
           !---- write it
           IF (l1d) THEN
              CALL restput &
                   &        (rest_id_sto, TRIM(varnames(iv)), 1, vardims(3), &
                   &         1, itau_dep, xtmp)
           ELSE
              CALL restput &
                   &        (rest_id_sto, TRIM(varnames(iv)), nbp_glo, vardims(3), &
                   &         1, itau_dep, var_3d, 'scatter',  nbp_glo, indices_g)
              !----
              DEALLOCATE(var_3d)
           ENDIF
        ENDIF
     ENDDO
     !-
     ! read soil carbon stocks values stored in the input restart file
     !-
     ALLOCATE(carbon_g(nbp_glo,ncarb,nvm,nbdl),stat=ier)
  IF (ier /= 0) THEN
     WRITE(numout,*) 'Allocatoin error carbon_g ier = ',ier 
     STOP 
  END IF
     carbon_g(:,:,:,:) = val_exp
     ALLOCATE(DOC_g(nbp_glo,nvm,nbdl,ndoc,npool,nelements),stat=ier)
  IF (ier /= 0) THEN
     WRITE(numout,*) 'Allocatoin error DOC_g ier = ',ier 
     STOP 
  END IF

     DOC_g(:,:,:,:,:,:) = val_exp
    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z01_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,1), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,1) == val_exp)) carbon_g(:,:,m,1) = zero
    ENDDO
    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z02_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,2), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,2) == val_exp)) carbon_g(:,:,m,2) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z03_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,3), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,3) == val_exp)) carbon_g(:,:,m,3) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z04_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,4), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,4) == val_exp)) carbon_g(:,:,m,4) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z05_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,5), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,5) == val_exp)) carbon_g(:,:,m,5) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z06_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,6), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,6) == val_exp)) carbon_g(:,:,m,6) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z07_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,7), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,7) == val_exp)) carbon_g(:,:,m,7) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z08_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,8), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,8) == val_exp)) carbon_g(:,:,m,8) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z09_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,9), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,9) == val_exp)) carbon_g(:,:,m,9) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z10_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,10), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,10) == val_exp)) carbon_g(:,:,m,10) = zero
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z11_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, ncarb , 1, itau_dep, &
              &                   .TRUE., carbon_g(:,:,m,11), 'gather', nbp_glo, indices_g)
         IF (ALL(carbon_g(:,:,m,11) == val_exp)) carbon_g(:,:,m,11) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z1_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,1,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,1,ifree,m,icarbon) == val_exp)) DOC_g(:,:,1,ifree,m,icarbon) = zero
    ENDDO
    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z2_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,2,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,2,ifree,m,icarbon) == val_exp)) DOC_g(:,:,2,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z3_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,3,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,3,ifree,m,icarbon) == val_exp)) DOC_g(:,:,3,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z4_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,4,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,4,ifree,m,icarbon) == val_exp)) DOC_g(:,:,4,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z5_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,5,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,5,ifree,m,icarbon) == val_exp)) DOC_g(:,:,5,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z6_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,6,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,6,ifree,m,icarbon) == val_exp)) DOC_g(:,:,6,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z7_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,7,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,7,ifree,m,icarbon) == val_exp)) DOC_g(:,:,7,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z8_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,8,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,8,ifree,m,icarbon) == val_exp)) DOC_g(:,:,8,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z9_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,9,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,9,ifree,m,icarbon) == val_exp)) DOC_g(:,:,9,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z10_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,10,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,10,ifree,m,icarbon) == val_exp)) DOC_g(:,:,10,ifree,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z11_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,11,ifree,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,11,ifree,m,icarbon) == val_exp)) DOC_g(:,:,11,ifree,m,icarbon) = zero
    ENDDO
    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z1_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,1,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,1,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,1,iadsorbed,m,icarbon) = zero
    ENDDO
    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z2_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,2,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,2,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,2,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z3_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,3,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,3,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,3,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z4_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,4,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,4,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,4,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z5_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,5,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,5,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,5,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z6_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,6,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,6,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,6,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z7_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,7,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,7,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,7,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z8_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,8,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,8,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,8,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z9_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,9,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,9,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,9,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z10_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,10,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,10,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,10,iadsorbed,m,icarbon) = zero
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z11_'//part_str(1:LEN_TRIM(part_str))
         CALL restget (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   .TRUE., DOC_g(:,:,11,iadsorbed,m,icarbon), 'gather', nbp_glo, indices_g)
         IF (ALL(DOC_g(:,:,11,iadsorbed,m,icarbon) == val_exp)) DOC_g(:,:,11,iadsorbed,m,icarbon) = zero
    ENDDO

    ! TF-DOC
    ALLOCATE(interception_storage_g(nbp_glo,nvm,nelements))
    interception_storage_g(:,:,:) = val_exp
    CALL restget (rest_id_sto, 'interception_storage_DOC', nbp_glo, nvm , 1, itau_dep, &
         &              &                   .TRUE., interception_storage_g(:,:,icarbon), 'gather', nbp_glo, indices_g)
    IF (ALL(interception_storage_g(:,:,icarbon) == val_exp)) interception_storage_g(:,:,icarbon) = zero
    !
    WRITE(numout,*) "date0 : ",date0, itau_dep
     !-
     ! Analytical spinup is set to false
     !
     spinup_analytic = .FALSE.

     ! Length of the run (in Years)
     ! User value is used if present in the .def file
     ! If not, default value (10000 Years) is used
     !-
     WRITE(time_str,'(a)') '10000Y'
     CALL getin('TIME_LENGTH', time_str)
     write(numout,*) 'Number of years for carbon spinup : ',time_str
     ! transform into itau
     CALL tlen2itau(time_str, dt_forcesoil*one_day, date0, itau_len)
     write(numout,*) 'Number of time steps to do: ',itau_len
     !-
     ! read soil carbon inputs, water and temperature stresses on OM decomposition 
     ! into the forcing file - We read an average year.
     !-
     ALLOCATE(soilcarbon_input_g(nbp_glo,nvm,nbdl,npool,nelements,nparan*nbyear))
     ALLOCATE(floodcarbon_input_g(nbp_glo,nvm,npool,nelements,nparan*nbyear))
     ALLOCATE(litter_above_g(nbp_glo,nlitt,nvm,nelements,nparan*nbyear))
     ALLOCATE(litter_below_g(nbp_glo,nlitt,nvm,nbdl,nelements,nparan*nbyear))
     ALLOCATE(lignin_struc_above_g(nbp_glo,nvm,nparan*nbyear))
     ALLOCATE(lignin_struc_below_g(nbp_glo,nvm,nbdl,nparan*nbyear))
     !-
     ALLOCATE(control_temp_soil_g(nbp_glo,nbdl,npool*2,nparan*nbyear))
     ALLOCATE(control_moist_soil_g(nbp_glo,nbdl,nvm,nparan*nbyear))
     ALLOCATE(moist_soil_g(nbp_glo,nbdl,nparan*nbyear))
     ALLOCATE(soil_mc_g(nbp_glo,nbdl,nstm,nparan*nbyear))
     ALLOCATE(floodout_g(nbp_glo,nparan*nbyear))
     ALLOCATE(wat_flux0_g(nbp_glo,nstm,nparan*nbyear))
     ALLOCATE(wat_flux_g(nbp_glo,nbdl,nstm,nparan*nbyear))
     ALLOCATE(runoff_per_soil_g(nbp_glo,nstm,nparan*nbyear))
     ALLOCATE(drainage_per_soil_g(nbp_glo,nstm,nparan*nbyear))
     ALLOCATE(DOC_to_topsoil_g(nbp_glo,nflow,nparan*nbyear))
     ALLOCATE(DOC_to_subsoil_g(nbp_glo,nflow,nparan*nbyear))
     ALLOCATE(flood_frac_g(nbp_glo,nparan*nbyear))
     !! TF-DOC
     ALLOCATE(precip2canopy_g(nbp_glo,nvm,nparan*nbyear))
     ALLOCATE(precip2ground_g(nbp_glo,nvm,nparan*nbyear))
     ALLOCATE(canopy2ground_g(nbp_glo,nvm,nparan*nbyear)) 
     !-
     ier = NF90_INQ_VARID (Cforcing_id,'soilcarbon_input',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,soilcarbon_input_g)
     ier = NF90_INQ_VARID (Cforcing_id,'floodcarbon_input',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,soilcarbon_input_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'control_moist_soil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,control_moist_soil_g)
     ier = NF90_INQ_VARID (Cforcing_id,    'control_temp_soil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,control_temp_soil_g)
     ier = NF90_INQ_VARID (Cforcing_id,    'litter_above',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,litter_above_g)
     ier = NF90_INQ_VARID (Cforcing_id,    'litter_below',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,litter_below_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'moist_soil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,moist_soil_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'soil_mc',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,soil_mc_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'floodout',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,floodout_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'wat_flux0',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,wat_flux0_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'wat_flux',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,wat_flux_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'runoff_per_soil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,runoff_per_soil_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'drainage_per_soil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,drainage_per_soil_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'DOC_to_topsoil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,DOC_to_topsoil_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'DOC_to_subsoil',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,DOC_to_subsoil_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'flood_frac',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,flood_frac_g)
     !! TF-DOC
     ier = NF90_INQ_VARID (Cforcing_id,   'precip2canopy',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,precip2canopy_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'precip2ground',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,precip2ground_g)
     ier = NF90_INQ_VARID (Cforcing_id,   'canopy2ground',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,canopy2ground_g)
     !-
     ier = NF90_INQ_VARID (Cforcing_id,    'lignin_struc_above',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,lignin_struc_above_g)
     ier = NF90_INQ_VARID (Cforcing_id,    'lignin_struc_below',v_id)
     ier = NF90_GET_VAR   (Cforcing_id,v_id,lignin_struc_below_g)
     !-
     ier = NF90_CLOSE (Cforcing_id)
     !-
  ENDIF
  CALL bcast(nparan)
  CALL bcast(nbyear)
  CALL bcast(dt_forcesoil)
  CALL bcast(iim_g)
  CALL bcast(jjm_g)
  CALL bcast(nbp_glo)
  CALL bcast(itau_dep)
  CALL bcast(itau_len)
  IF (.NOT. ALLOCATED(indices_g)) ALLOCATE (indices_g(nbp_glo))
  CALL bcast(indices_g)
 
  !
  ! We must initialize data_para :
  CALL init_orchidee_data_para_driver(nbp_glo,indices_g)

  kjpindex=nbp_loc
  jjm=jj_nb
  iim=iim_g
  IF (debug) WRITE(numout,*) "Local grid : ",kjpindex,iim,jjm

  !---
  !--- Create the index table
  !---
  !--- This job returns a LOCAL kindex.
  !---
  ALLOCATE (indices(kjpindex),stat=ier)
  !
  !! scattering to all processes in parallel mode
  !
  CALL scatter(indices_g,indices)
  indices(1:kjpindex)=indices(1:kjpindex)-(jj_begin-1)*iim_g
  IF (debug) WRITE(numout,*) mpi_rank,"indices locaux = ",indices(1:kjpindex)
  !-
  ! Allocation of the variables for a processor
  !-
  ALLOCATE(clay(kjpindex))
  ALLOCATE(soil_ph(kjpindex))
  ALLOCATE(poor_soils(kjpindex))
  ALLOCATE(bulk_dens(kjpindex))
  ALLOCATE(soiltile(kjpindex,nstm))
  ALLOCATE(veget_max(kjpindex,nvm))
  ALLOCATE(soilcarbon_input(kjpindex,nvm,nbdl,npool,nelements,nparan*nbyear))
  ALLOCATE(floodcarbon_input(kjpindex,nvm,npool,nelements,nparan*nbyear))
  ALLOCATE(control_temp_soil(kjpindex,nbdl,npool*2,nparan*nbyear))
  ALLOCATE(control_moist_soil(kjpindex,nbdl,nvm,nparan*nbyear))
  ALLOCATE(carbon(kjpindex,ncarb,nvm,nbdl))
  ALLOCATE(litter_above(kjpindex,nlitt,nvm,nelements,nparan*nbyear))
  ALLOCATE(litter_below(kjpindex,nlitt,nvm,nbdl,nelements,nparan*nbyear))
  ALLOCATE(resp_hetero_soil(kjpindex,nvm))
  ALLOCATE(resp_flood_soil(kjpindex,nvm))
  ALLOCATE(soilhum(kjpindex,nbdl))
  ALLOCATE(moist_soil(kjpindex,nbdl,nparan*nbyear))
  ALLOCATE(soil_mc(kjpindex,nbdl,nstm,nparan*nbyear))
  ALLOCATE(floodout(kjpindex,nparan*nbyear))
  ALLOCATE(wat_flux0(kjpindex,nstm,nparan*nbyear))
  ALLOCATE(wat_flux(kjpindex,nbdl,nstm,nparan*nbyear))
  ALLOCATE(runoff_per_soil(kjpindex,nstm,nparan*nbyear))
  ALLOCATE(drainage_per_soil(kjpindex,nstm,nparan*nbyear))
  ALLOCATE(DOC_to_topsoil(kjpindex,nflow,nparan*nbyear))
  ALLOCATE(DOC_to_subsoil(kjpindex,nflow,nparan*nbyear))
  ALLOCATE(flood_frac(kjpindex,nparan*nbyear))
  ALLOCATE(DOC(kjpindex,nvm,nbdl,ndoc,npool,nelements))
  ALLOCATE(DOC_EXP(kjpindex,nvm,nexp,npool,nelements))
  ALLOCATE(lignin_struc_above(kjpindex,nvm,nparan*nbyear))
  ALLOCATE(lignin_struc_below(kjpindex,nvm,nbdl,nparan*nbyear))
  ! TF-DOC
  ALLOCATE(dry_dep_canopy(kjpindex,nvm,nelements,nparan*nbyear))
  ALLOCATE(DOC_precip2canopy(kjpindex,nvm,nelements,nparan*nbyear))
  ALLOCATE(DOC_precip2ground(kjpindex,nvm,nelements,nparan*nbyear))
  ALLOCATE(DOC_canopy2ground(kjpindex,nvm,nelements,nparan*nbyear))
  ALLOCATE(DOC_infil(kjpindex,nvm,nelements,nparan*nbyear))
  ALLOCATE(DOC_noinfil(kjpindex,nvm,nelements,nparan*nbyear))
  ALLOCATE(interception_storage(kjpindex,nvm,nelements))
  ALLOCATE(precip2canopy(kjpindex,nvm,nparan*nbyear))
  ALLOCATE(precip2ground(kjpindex,nvm,nparan*nbyear))
  ALLOCATE(canopy2ground(kjpindex,nvm,nparan*nbyear))
  ALLOCATE(biomass(kjpindex,nvm,nparts,nelements,nparan*nbyear))
  flood_frac = zero
  iatt = 0
  !-
  ! Initialization of the variables for a processor
  !-
  CALL Scatter(clay_g,clay)
  CALL Scatter(soil_ph_g,soil_ph)
  CALL Scatter(bulk_dens_g,bulk_dens)
  CALL Scatter(soiltile_g,soiltile)
  CALL Scatter(veget_max_g,veget_max)
          DO m =1,nvm
             DO l=1,nbdl
                DO i=1,npool
  CALL Scatter(soilcarbon_input_g(:,m,l,i,:,:),soilcarbon_input(:,m,l,i,:,:))
                ENDDO
             ENDDO
          ENDDO
          DO m =1,nvm
             DO i=1,npool
                CALL Scatter(floodcarbon_input_g(:,m,i,:,:),floodcarbon_input(:,m,i,:,:))
             ENDDO
          ENDDO
          DO i =1,nlitt
             DO j=1,nvm
  CALL Scatter(litter_above_g(:,i,j,:,:),litter_above(:,i,j,:,:))
             ENDDO
          ENDDO

          DO i =1,nlitt
             DO j=1,nvm
                DO k = 1,nbdl
  CALL Scatter(litter_below_g(:,i,j,k,:,:),litter_below(:,i,j,k,:,:))
                ENDDO
             ENDDO
          ENDDO
  CALL Scatter(control_temp_soil_g,control_temp_soil)
  CALL Scatter(control_moist_soil_g,control_moist_soil)
  CALL Scatter(moist_soil_g,moist_soil)
  CALL Scatter(soil_mc_g,soil_mc)
  CALL Scatter(floodout_g,floodout)
  CALL Scatter(wat_flux0_g,wat_flux0)
  CALL Scatter(wat_flux_g,wat_flux)
  CALL Scatter(runoff_per_soil_g,runoff_per_soil)
  CALL Scatter(drainage_per_soil_g,drainage_per_soil)
  CALL Scatter(DOC_to_topsoil_g,DOC_to_topsoil)
  CALL Scatter(DOC_to_subsoil_g,DOC_to_subsoil)
  CALL Scatter(flood_frac_g,flood_frac)
  !TF-DOC
  CALL Scatter(precip2canopy_g,precip2canopy)
  CALL Scatter(precip2ground_g,precip2ground)
  CALL Scatter(canopy2ground_g,canopy2ground)
  CALL Scatter(interception_storage_g,interception_storage)  
          DO m =1,nvm
             DO l=1,nbdl
  CALL Scatter(carbon_g(:,:,m,l),carbon(:,:,m,l))  
             ENDDO
          ENDDO
          DO m =1,nbdl
             DO l=1,ndoc
                DO i= 1,npool
  CALL Scatter(DOC_g(:,:,m,l,i,icarbon),DOC(:,:,m,l,i,icarbon))
                ENDDO
             ENDDO
          ENDDO
          DO j=1,nvm
  CALL Scatter(lignin_struc_above_g(:,j,:),lignin_struc_above(:,j,:))
          ENDDO
          DO j=1,nvm
             DO k = 1,nbdl
  CALL Scatter(lignin_struc_below_g(:,j,k,:),lignin_struc_below(:,j,k,:))
             ENDDO
          ENDDO
!-
! Configu
!-

  !Config Key   = FRAC_CARB_AP
  !Config Desc  = frac carb coefficients from active pool: depends on clay content
  !Config if    = OK_STOMATE 
  !Config Def   = 0.004
  !Config Help  = fraction of the active pool going to the passive pool
  !Config Units = [-]
  CALL getin_p('FRAC_CARB_AP',frac_carb_ap)  
  !
  !Config Key   = FRAC_CARB_SA
  !Config Desc  = frac_carb_coefficients from slow pool
  !Config if    = OK_STOMATE 
  !Config Def   = 0.42
  !Config Help  = fraction of the slow pool going to the active pool
  !Config Units = [-] 
  CALL getin_p('FRAC_CARB_SA',frac_carb_sa)
  !
  !Config Key   = FRAC_CARB_SP
  !Config Desc  = frac_carb_coefficients from slow pool
  !Config if    = OK_STOMATE 
  !Config Def   = 0.03
  !Config Help  = fraction of the slow pool going to the passive pool
  !Config Units = [-] 
  !CALL getin_p('FRAC_CARB_SP',frac_carb_sp)
  !
  !Config Key   = FRAC_CARB_PA
  !Config Desc  = frac_carb_coefficients from passive pool
  !Config if    = OK_STOMATE 
  !Config Def   = 0.45
  !Config Help  = fraction of the passive pool going to the passive pool
  !Config Units = [-] 
  CALL getin_p('FRAC_CARB_PA',frac_carb_pa)
  !
  !Config Key   = FRAC_CARB_PS
  !Config Desc  = frac_carb_coefficients from passive pool
  !Config if    = OK_STOMATE 
  !Config Def   = 0.0
  !Config Help  = fraction of the passive pool going to the passive pool
  !Config Units = [-]
  !CALL getin_p('FRAC_CARB_PS',frac_carb_ps)
  !
  !Config Key   = ACTIVE_TO_PASS_CLAY_FRAC
  !Config Desc  = 
  !Config if    = OK_STOMATE 
  !Config Def   =  .68  
  !Config Help  =
  !Config Units = [-]
  CALL getin_p('ACTIVE_TO_PASS_CLAY_FRAC',active_to_pass_clay_frac)
!
  !Config Key   = CARBON_TAU_IACTIVE
  !Config Desc  = residence times in carbon pools
  !Config if    = OK_STOMATE 
  !Config Def   = 0.3
  !Config Help  =
  !Config Units =  [days] 
  CALL getin_p('CARBON_TAU_IACTIVE',carbon_tau_iactive)
  !
  !Config Key   = CARBON_TAU_ISLOW
  !Config Desc  = residence times in carbon pools
  !Config if    = OK_STOMATE 
  !Config Def   = 1.12
  !Config Help  =
  !Config Units = [days]
  CALL getin_p('CARBON_TAU_ISLOW',carbon_tau_islow)
  !
  !Config Key   = CARBON_TAU_IPASSIVE
  !Config Desc  = residence times in carbon pools
  !Config if    = OK_STOMATE 
  !Config Def   = 461.98
  !Config Help  = residence time in the passive pool
  !Config Units = [days] 
  CALL getin_p('CARBON_TAU_IPASSIVE',carbon_tau_ipassive)
  !
  !Config Key   = PRIMING_PARAM_IACTIVE
  !Config Desc  = priming parameter in carbon pools
  !Config if    = OK_STOMATE 
  !Config Def   = 493.66
  !Config Help  =
  !Config Units =  [-] 
  CALL getin_p('PRIMING_PARAM_IACTIVE',priming_param_iactive)
  !
  !Config Key   = PRIMING_PARAM_ISLOW
  !Config Desc  = priming parameter in carbon pools
  !Config if    = OK_STOMATE 
  !Config Def   = 194.03
  !Config Help  =
  !Config Units =  [-] 
  CALL getin_p('PRIMING_PARAM_ISLOW',priming_param_islow)
  !
  !Config Key   = PRIMING_PARAM_IPASSIVE
  !Config Desc  = priming parameter in carbon pools
  !Config if    = OK_STOMATE 
  !Config Def   = 136.54
  !Config Help  =
  !Config Units =  [-] 
  CALL getin_p('PRIMING_PARAM_IPASSIVE',priming_param_ipassive)
  !
!  !Config Key   = BULK_DENSITY
!  !Config Desc  = value of the soil bulk density 
!  !Config if    = READ_BULK_DENSITY
!  !Config Def   = 1.65
!  !Config Help  =
!  !Config Units =  [kg m-3] 
!  CALL getin_p('BULK_DENSITY',bulk_dens)
  !
  !Config Key   = FLUX_TOT_COEFF
  !Config Desc  =
  !Config if    = OK_STOMATE 
  !Config Def   = 1.2, 1.4,.75
  !Config Help  =
  !Config Units = [days]
  CALL getin_p('FLUX_TOT_COEFF',flux_tot_coeff)
  !
  !Config Key   = PREF_SOIL_VEG
  !Config Desc  = The soil tile number for each vegetation
  !Config if    = OK_SECHIBA or OK_STOMATE
  !Config Def   = 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3
  !Config Help  = Gives the number of the soil tile on which we will
  !Config         put each vegetation. This allows to divide the
  !hydrological column
  !Config Units = [-]        
  CALL getin_p('PREF_SOIL_VEG',pref_soil_veg)
  !
  !Config Key   = DOC_tau_labile
  !Config Desc  = residence time of labile DOC
  !Config if    = OK_STOMATE 
  !Config Def   = 3.76
  !Config Help  = mean from Kalbitz et al., 2003 Geoderma
  !Config Units =  [days] 
  CALL getin_p('DOC_TAU_LABILE',DOC_tau_labile)
  !
  !Config Key   = DOC_tau_stable
  !Config Desc  = residence time of stable DOC
  !Config if    = OK_STOMATE 
  !Config Def   = 602.41
  !Config Help  = mean from Kalbitz et al., 2003 Geoderma
  !Config Units =  [days] 
  CALL getin_p('DOC_TAU_STABLE',DOC_tau_stable)
  !
  !!Config Key  = D_DOC
  !Config Desc  = diffusion coefficient of DOC
  !Config if    = OK_STOMATE 
  !Config Def   = 0.0000004428
  !Config Help  = from Burdige et al., 1999 in Ota et al., 2013
  !Config Units =  [m^2 hr-1] 
  CALL getin_p('D_DOC',D_DOC)
  !
  !!Config Key   = Dif
  !Config Desc  = diffusion coefficient of POC
  !Config if    = OK_STOMATE 
  !Config Def   = 1E-4
  !Config Help  = from Koven et al., 2013 BG
  !Config Units =  [m^2 day-1] 
  CALL getin_p('Dif',Dif)
  !
  !!Config Key   = CUE
  !Config Desc  = Microbial carbon use efficiency
  !Config if    = OK_STOMATE 
  !Config Def   = 0.5
  !Config Help  = Assumed value of CUE=0.5
  !Config Units =  [-] 
  CALL getin_p('CUE',CUE)

  !!-
  !! 2. Computational step
  !! Loop over time - Call of soilcarbon routine at each time step 
  !! Updated soil carbon stocks are stored into carbon variable
  !! We only keep the last value of carbon variable (no time dimension).
  !!-
  iyear=1
  DO i=1,itau_len
     iatt = iatt+1
     IF (iatt > nparan*nbyear) THEN
        IF (debug) WRITE(numout,*) iyear
        iatt = 1
        iyear=iyear+1
     ENDIF

       ! Get diaglev from module vertical for CWRR
       diaglev=znt(1:nbdl)

     CALL soilcarbon &
          &    (kjpindex, dt_forcesoil, clay, &
          &     soilcarbon_input(:,:,:,:,:,iatt),floodcarbon_input(:,:,:,:,iatt), control_temp_soil(:,:,:,iatt), control_moist_soil(:,:,:,iatt), &
          &     carbon, resp_hetero_soil, resp_flood_soil,  &
          &     litter_above(:,:,:,:,iatt),litter_below(:,:,:,:,:,iatt),&
          &     soilhum, DOC, moist_soil(:,:,iatt), DOC_EXP,&
          &     lignin_struc_above(:,:,iatt), lignin_struc_below(:,:,:,iatt),&
          &     floodout(:,iatt), runoff_per_soil(:,:,iatt), drainage_per_soil(:,:,iatt), wat_flux0(:,:,iatt), &
          &     wat_flux(:,:,:,iatt),bulk_dens, soil_ph, poor_soils, veget_max, soil_mc(:,:,:,iatt), soiltile, &
          &     DOC_to_topsoil(:,:,iatt), DOC_to_subsoil(:,:,iatt), flood_frac(:,iatt), &
          &     precip2ground(:,:,iatt), precip2canopy(:,:,iatt), canopy2ground(:,:,iatt), &
          &     dry_dep_canopy(:,:,:,iatt), DOC_precip2ground(:,:,:,iatt), &
          &     DOC_precip2canopy(:,:,:,iatt), DOC_canopy2ground(:,:,:,iatt), &
          &     DOC_infil(:,:,:,iatt), DOC_noinfil(:,:,:,iatt), interception_storage, biomass(:,:,:,:,iatt), runoff_per_soil(:,2,iatt))
  ENDDO
  WRITE(numout,*) "End of soilcarbon LOOP."

  !
  !! Gathering of variables towards main processor in parallel mode
  !
          DO m =1,nvm
             DO l=1,nbdl
  CALL Gather(carbon(:,:,m,l),carbon_g(:,:,m,l))
             ENDDO
          ENDDO
          DO m =1,nbdl
             DO l=1,ndoc
                DO i= 1,npool
  CALL Gather(DOC(:,:,m,l,i,icarbon),DOC_g(:,:,m,l,i,icarbon))
                ENDDO
             ENDDO
          ENDDO
  ! TF-DOC
  CALL Gather(interception_storage,interception_storage_g)
  !!-
  !! 3. write new carbon stocks into the ouput restart file
  !!-
  IF (is_root_prc) THEN
    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z01_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,1), 'scatter', nbp_glo, indices_g)
    ENDDO
    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z02_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,2), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z03_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,3), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z04_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,4), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z05_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,5), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z06_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,6), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z07_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,7), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z07_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,7), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z08_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,8), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z09_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,9), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z10_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,10), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,nvm
         WRITE (part_str, '(I2)') m
         IF (m<10) part_str(1:1)='0'
         var_name = 'carbon_z11_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, ncarb, 1, itau_dep, &
              &                   carbon_g(:,:,m,11), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z1_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,1,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z2_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,2,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z3_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,3,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z4_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,4,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z5_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,5,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z6_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,6,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z7_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,7,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z8_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,8,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z9_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,9,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z10_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,10,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'freedoc_z11_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,11,ifree,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z1_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,1,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z2_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,2,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z3_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,3,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z4_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,4,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z5_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,5,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z6_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,6,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z7_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,7,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z8_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,8,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z9_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,9,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z10_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,10,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO

    DO m=1,npool
         WRITE (part_str, '(I1)') m
         var_name = 'adsdoc_z11_'//part_str(1:LEN_TRIM(part_str))
         CALL restput (rest_id_sto, var_name, nbp_glo, nvm , 1, itau_dep, &
              &                   DOC_g(:,:,11,iadsorbed,m,icarbon), 'scatter', nbp_glo, indices_g)
    ENDDO
    ! TF-DOC
    CALL restput (rest_id_sto, 'interception_storage_DOC', nbp_glo, nvm , 1, itau_dep, &
         &              &                   interception_storage_g(:,:,icarbon), 'scatter', nbp_glo, indices_g)
    !-
     !-
     CALL getin_dump
     CALL restclo
  ENDIF

  CALL MPI_FINALIZE(ier)
  WRITE(numout,*) "End of forcesoil."
  !--------------------
END PROGRAM forcesoil
