"""CarbonTracker Data Assimilation Shell (CTDAS) Copyright (C) 2017 Wouter Peters. 
Users are recommended to contact the developers (wouter.peters@wur.nl) to receive
updates of the code. See also: http://www.carbontracker.eu. 

This program is free software: you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software Foundation, 
version 3. This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS 
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. 

You should have received a copy of the GNU General Public License along with this 
program. If not, see <http://www.gnu.org/licenses/>."""
#!/usr/bin/env python
# optimizer.py

"""
Author : Aki 

Revision History:
File revised on Feb 2017.

"""

import os
import sys
sys.path.append(os.getcwd())
import logging
import numpy as np
from da.baseclasses.optimizer import Optimizer

identifier = 'Ensemble Square Root Filter'
version = '0.0'

################### Begin Class MethaneOptimizer ###################

class MethaneOptimizer(Optimizer):
    """
        This creates an instance of a CarbonTracker optimization object. The base class it derives from is the optimizer object.
        Additionally, this MethaneOptimizer implements a special localization option following the CT2007 method.

        All other methods are inherited from the base class Optimizer.
    """
    #def getid(self):
    #    return identifier

    #def getversion(self):
    #    return version


    def set_localization(self, loctype='None'):
        """ determine which localization to use """

        if loctype == 'CT2007':
            self.localization = True
            self.localizetype = 'CT2007'
            #T-test values for two-tailed student's T-test using 95% confidence interval for some options of nmembers
            if self.nmembers == 50:
                self.tvalue = 2.0086
            elif self.nmembers == 12:
                self.tvalue = 2.1789 
            elif self.nmembers == 20:
                self.tvalue = 2.0860 
            elif self.nmembers == 60:
                self.tvalue = 2.0003 
            elif self.nmembers == 70:
                self.tvalue = 1.9944 
            elif self.nmembers == 100:
                self.tvalue = 1.9840
            elif self.nmembers == 120:
                self.tvalue = 1.97993 
            elif self.nmembers == 150:
                self.tvalue = 1.97591
            elif self.nmembers == 180:
                self.tvalue = 1.973231
            elif self.nmembers == 200:
                self.tvalue = 1.9719    
            elif self.nmembers == 240:
                self.tvalue = 1.969898
            elif self.nmembers == 480:
                self.tvalue = 1.964918
            elif self.nmembers == 500:
                self.tvalue = 1.96472
            elif self.nmembers == 1000:
                self.tvalue = 1.962339
            else: self.tvalue =  0.0   
        else:
            self.localization = False
            self.localizetype = 'None'
    
        logging.info("Current localization option is set to %s" % self.localizetype)
        if self.localization == True:
            if self.tvalue == 0:
                logging.error("Critical tvalue for localization not set for %i ensemble members"%(self.nmembers))
                sys.exit(2)
            else: logging.info("Used critical tvalue %0.05f is based on 95%% probability and %i ensemble members in a two-tailed student's T-test"%(self.tvalue,self.nmembers))

    def localize(self, n):
        """ localize the Kalman Gain matrix """
        import numpy as np

        if not self.localization: 
            logging.debug('Not localized observation %i' % self.obs_ids[n])
            return 
        if self.localizetype == 'CT2007':
            count_localized = 0
            for r in range(self.nlag * self.nparams):
                corr = np.corrcoef(self.HX_prime[n, :], self.X_prime[r, :].squeeze())[0, 1]
                prob = corr / np.sqrt((1.000000001 - corr ** 2) / (self.nmembers - 2))
                if abs(prob) < self.tvalue:
                    self.KG[r] = 0.0
                    count_localized = count_localized + 1
            logging.debug('Localized observation %i, %i%% of values set to 0' % (self.obs_ids[n],count_localized*100/(self.nlag * self.nparams)))

    def set_algorithm(self, algorithm='Serial'):
        """ determine which minimum least squares algorithm to use """

        if algorithm == 'Serial':
            self.algorithm = 'Serial'
        else:
            self.algorithm = 'Bulk'
    
        logging.info("Current minimum least squares algorithm is set to %s" % self.algorithm)



################### End Class MethaneOptimizer ###################

if __name__ == "__main__":
    pass
